/*
   Copyright (C) 2000-2013  Ulric Eriksson <ulric@siag.nu>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.
*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>

static unsigned char mask4[33][4];
static unsigned char mask6[129][16];

static void init_masks(void)
{
	unsigned char m4[4];
	unsigned char m6[16];
	int i, j;

	memset(m4, 0, sizeof m4);
	memset(m6, 0, sizeof m6);
	for (i = 0; i < 33; i++) {
		for (j = 3; j >= 0; j--) {
			mask4[i][j] = m4[j];
			m4[j] >>= 1;
			if (j > 0) {
				m4[j] |= (m4[j-1] << 7);
			} else {
				m4[j] |= (1 << 7);
			}
		}
		printf("%d: %d %d %d %d\n", i, mask4[i][0], mask4[i][1], mask4[i][2], mask4[i][3]);
	}
	for (i = 0; i < 129; i++) {
		for (j = 15; j >= 0; j--) {
			mask6[i][j] = m6[j];
			m6[j] >>= 1;
			if (j > 0) {
				m6[j] |= (m6[j-1] << 7);
			} else {
				m6[j] |= (1 << 7);
			}
		}
	}
}

//#include "pen-net.h"

static void pen_error(char *fmt, ...)
{
        char b[4096];
        va_list ap;
        va_start(ap, fmt);
        vsnprintf(b, sizeof b, fmt, ap);
        fprintf(stderr, "%s\n", b);
        va_end(ap);
        exit(EXIT_FAILURE);
}

static void test_unlookup(struct addrinfo *ai)
{
	char host[1000];
	char serv[1000];
	int n;

	n = getnameinfo(ai->ai_addr, ai->ai_addrlen, host, 1000,
		serv, 1000, NI_NUMERICHOST|NI_NUMERICSERV);
	if (n != 0) {
		printf("getnameinfo: %s\n", gai_strerror(n));
		return;
	}
	printf("addr = '%s', port = '%s'\n", host, serv);
}

static void dump_address(struct addrinfo *ai)
{
	struct sockaddr_storage *ss = ai->ai_addr;
	struct sockaddr_in *sa4;
	struct sockaddr_in6 *sa6;
	unsigned char *b;
	int i;

	printf("ss_family = %d\n", (int)ss->ss_family);
	switch (ss->ss_family) {
	case AF_INET:
		sa4 = ss;
		printf("IPv4\n");
		printf("sin_family = %d\n", sa4->sin_family);
		printf("sin_port = %d\n", ntohs(sa4->sin_port));
		b = &(sa4->sin_addr);
		for (i = 0; i < 4; i++) {
			printf("%d ", (int)b[i]);
		}
		printf("\n");
		break;
	case AF_INET6:
		sa6 = ss;
		printf("IPv6\n");
		printf("sin6_family = %d\n", sa6->sin6_family);
		printf("sin6_port = %d\n", ntohs(sa6->sin6_port));
		b = &(sa6->sin6_addr);
		for (i = 0; i < 16; i++) {
			printf("%d ", (int)b[i]);
		}
		printf("\n");
		break;
	default:
		printf("Unknown family\n");
		break;
	}
}

static void test_lookup(char *arg)
{
	struct addrinfo *ai;
	struct addrinfo hints;
	int n;
	char addr[1000];
	char port[1000];

	if (arg[0] == '[') {
		n = sscanf(arg, "[%[^]]]:%s", addr, port);
	} else {
		n = sscanf(arg, "%[^:]:%s", addr, port);
	}
	if (n != 2) {
		printf("Bogus argument '%s'\n", arg);
		return;
	}

	printf("addr = '%s', port = '%s'\n", addr, port);

	memset(&hints, 0, sizeof(hints));
	hints.ai_family = AF_UNSPEC;
	hints.ai_flags = AI_ADDRCONFIG;
	hints.ai_socktype = SOCK_STREAM;
	n = getaddrinfo(addr, port, &hints, &ai);
	if (n != 0) {
		pen_error("getaddrinfo: %s", gai_strerror(n));
		return;
	}
	printf("family = %d\n", ai->ai_family);
	printf("socktype = %d\n", ai->ai_socktype);
	printf("protocol = %d\n", ai->ai_protocol);
	printf("addrlen = %d\n", (int)ai->ai_addrlen);
	printf("sockaddr = %p\n", ai->ai_addr);
	printf("canonname = %s\n", ai->ai_canonname);
	test_unlookup(ai);
	dump_address(ai);
	freeaddrinfo(ai);
	printf("\n");
}

int main(int argc, char **argv)
{
	int i;

	init_masks();
	for (i = 1; i < argc; i++) {
		test_lookup(argv[i]);
	}
	return 0;
}
