/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%                  L       OOO    CCCC   AAA   L      EEEEE                   %
%                  L      O   O  C      A   A  L      E                       %
%                  L      O   O  C      AAAAA  L      EEE                     %
%                  L      O   O  C      A   A  L      E                       %
%                  LLLLL   OOO    CCCC  A   A  LLLLL  EEEEE                   %
%                                                                             %
%                                                                             %
%                    ImageMagick Locale Message Methods                       %
%                                                                             %
%                                                                             %
%                              Software Design                                %
%                                John Cristy                                  %
%                                Kyle Shorter                                 %
%                               September 2002                                %
%                                                                             %
%                                                                             %
%  Copyright (C) 2003 ImageMagick Studio, a non-profit organization dedicated %
%  to making software imaging solutions freely available.                     %
%                                                                             %
%  Permission is hereby granted, free of charge, to any person obtaining a    %
%  copy of this software and associated documentation files ("ImageMagick"),  %
%  to deal in ImageMagick without restriction, including without limitation   %
%  the rights to use, copy, modify, merge, publish, distribute, sublicense,   %
%  and/or sell copies of ImageMagick, and to permit persons to whom the       %
%  ImageMagick is furnished to do so, subject to the following conditions:    %
%                                                                             %
%  The above copyright notice and this permission notice shall be included in %
%  all copies or substantial portions of ImageMagick.                         %
%                                                                             %
%  The software is provided "as is", without warranty of any kind, express or %
%  implied, including but not limited to the warranties of merchantability,   %
%  fitness for a particular purpose and noninfringement.  In no event shall   %
%  ImageMagick Studio be liable for any claim, damages or other liability,    %
%  whether in an action of contract, tort or otherwise, arising from, out of  %
%  or in connection with ImageMagick or the use or other dealings in          %
%  ImageMagick.
%                                                                             %
%  Except as contained in this notice, the name of the ImageMagick Studio     %
%  shall not be used in advertising or otherwise to promote the sale, use or  %
%  other dealings in ImageMagick without prior written authorization from the %
%  ImageMagick Studio.                                                        %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%
%
*/

/*
  Include declarations.
*/
#include "magick/studio.h"
#include "magick/blob.h"
#include "magick/error.h"
#include "magick/hashmap.h"
#include "magick/utility.h"
/*
  Define declarations.
*/
#define LocaleFilename  "locale.mgk"

/*
  Declare type map.
*/
static char
  *LocaleMap = (char *)
    "<?xml version=\"1.0\"?>"
    "<localemap>"
    "  <locale name=\"english\">"
    "  </locale>"
    "</localemap>";

/*
  Global declarations.
*/
static HashMapInfo
  *locale_map = (HashMapInfo *) NULL;

/*
  Forward declarations.
*/
static unsigned int
  ReadConfigureFile(const char *,const char *,const unsigned long,
    ExceptionInfo *);

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   D e s t r o y L o c a l e I n f o                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  DestroyLocaleInfo() deallocates memory associated the locale messaging
%  system.
%
%  The format of the DestroyLocaleInfo method is:
%
%       void DestroyLocaleInfo(void)
%
%
*/
MagickExport void DestroyLocaleInfo(void)
{
  if (locale_map != (HashMapInfo *) NULL)
    DestroyHashMap(locale_map);
  locale_map=(HashMapInfo *) NULL;
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t L o c a l e M e s s a g e                                           %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetLocaleMessage() returns a message in the current locale that matches the
%  supplied tag.
%
%  The format of the GetLocaleMessage method is:
%
%      const char *GetLocaleMessage(const char *tag)
%
%  A description of each parameter follows:
%
%    o tag: Return a message that matches this tag in the current locale.
%
%
*/
MagickExport const char *GetLocaleMessage(const char *tag)
{
  char
    key[MaxTextExtent];

  const char
    *message;

  static char
    *locale;

  if ((tag == (const char *) NULL) || (*tag == '\0'))
    return(tag);
  if (locale_map == (HashMapInfo *) NULL)
    {
      ExceptionInfo
        exception;

      locale=setlocale(LC_CTYPE,0);
      if ((locale == (char *) NULL) || (*locale == '\0'))
        locale=getenv("LC_ALL");
      if ((locale == (char *) NULL) || (*locale == '\0'))
        locale=getenv("LC_MESSAGES");
      if ((locale == (char *) NULL) || (*locale == '\0'))
        locale=getenv("LC_CTYPE");
      if ((locale == (char *) NULL) || (*locale == '\0'))
        locale=getenv("LANG");
      if ((locale == (char *) NULL) || (*locale == '\0'))
        locale="english";
      GetExceptionInfo(&exception);
      (void) ReadConfigureFile(LocaleFilename,locale,0,&exception);
      if (GetNumberOfEntriesInHashMap(locale_map) == 0)
        (void) ReadConfigureFile("english.mgk",locale,0,&exception);
      DestroyExceptionInfo(&exception);
    }
  FormatString(key,"%.1024s/",tag);
  message=(const char *) GetEntryFromHashMap(locale_map,key);
  if (message != (const char *) NULL)
    return(message);
  return(tag);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%  L i s t L o c a l e I n f o                                                %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  ListLocaleInfo() lists the locale info to a file.
%
%  The format of the ListLocaleInfo method is:
%
%      unsigned int ListLocaleInfo(FILE *file,ExceptionInfo *exception)
%
%  A description of each parameter follows.
%
%    o file:  An pointer to a FILE.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/

#if defined(__cplusplus) || defined(c_plusplus)
extern "C" {
#endif

static int Compare(const void *x,const void *y)
{
  register char
    *p,
    *q;

  p=(char *) x;
  q=(char *) y;
  return(LocaleCompare(p,q));
}

#if defined(__cplusplus) || defined(c_plusplus)
}
#endif

MagickExport unsigned int ListLocaleInfo(FILE *file,ExceptionInfo *exception)
{
  char
    locale[MaxTextExtent];

  const char
    *tag,
    *message;

  LinkedListInfo
    *list_info;

  register char
    *p;

  unsigned int
    status;

  unsigned long
    width;

  if (file == (const FILE *) NULL)
    file=stdout;
  (void) GetLocaleMessage("*");
  list_info=AcquireLinkedList(GetNumberOfEntriesInHashMap(locale_map));
  ResetHashMapIterator(locale_map);
  tag=(const char *) GetNextEntryInHashMap(locale_map);
  while (tag != (const char *) NULL)
  {
    message=GetEntryFromHashMap(locale_map,tag);
    (void) strncpy(locale,tag,MaxTextExtent-1);
    (void) strncat(locale,message,MaxTextExtent-strlen(locale)-1);
    FormatString(locale,"%.1024s%.1024s",tag,message);
    status=InsertElementInSortedLinkedList(list_info,Compare,(void **) NULL,
      (void *) AcquireString(locale));
    tag=(const char *) GetNextEntryInHashMap(locale_map);
  }
  width=80;
  p=(char *) getenv("COLUMNS");
  if (tag != (char *) NULL)
    width=atol(p);
  (void) fprintf(file,"Tag/Message\n");
  (void) fprintf(file,"-------------------------------------------------"
    "------------------------------\n");
  ResetLinkedListIterator(list_info);
  message=(const char *) GetNextElementInLinkedList(list_info);
  while (message != (const char *) NULL)
  {
    if (strlen(message) < (width-1))
      (void) fprintf(file,"%.1024s\n",message);
    else
      {
        (void) strncpy(locale,message,MaxTextExtent-1);
        p=locale+strlen(locale)-1;
        while ((p > locale) && (*p != '/'))
          p--;
        *p++='\0';
        (void) fprintf(file,"%.1024s/\n",locale);
        (void) fprintf(file,"  %.1024s\n",p);
      }
    message=(const char *) GetNextElementInLinkedList(list_info);
  }
  DestroyLinkedList(list_info,LiberateMemory);
  (void) fflush(file);
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+   R e a d C o n f i g u r e F i l e                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  ReadConfigureFile() reads the color configuration file which maps color
%  strings with a particular image format.
%
%  The format of the ReadConfigureFile method is:
%
%      unsigned int ReadConfigureFile(const char *basename,
%        const char *locale,const unsigned long depth,ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o basename:  The color configuration filename.
%
%    o locale: The locale.
%
%    o depth: depth of <include /> statements.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/

static void ChopLocaleComponents(char *path,const unsigned long components)
{
  long
    count;

  register char
    *p;

  if (*path == '\0')
    return;
  p=path+strlen(path)-1;
  if (*p == '/')
    *p='\0';
  for (count=0; (count < (long) components) && (p > path); p--)
    if (*p == '/')
      {
        *p='\0';
        count++;
      }
}

static unsigned int ReadConfigureFile(const char *basename,const char *locale,
  const unsigned long depth,ExceptionInfo *exception)
{
  char
    keyword[MaxTextExtent],
    message[MaxTextExtent],
    path[MaxTextExtent],
    *q,
    tag[MaxTextExtent],
    *token,
    *xml;

  register char
    *p;

  size_t
    length;

  unsigned int
    status;

  /*
    Read the locale configure file.
  */
  if (locale_map == (HashMapInfo *) NULL)
    {
      locale_map=AcquireHashMap(MediumHashMapSize,HashStringType,
        LiberateMemory,LiberateMemory);
      if (locale_map == (HashMapInfo *) NULL)
        {
          (void) ThrowException(exception,ResourceLimitError,
            "MemoryAllocationFailed",path);
          return(False);
        }
    }
  (void) strcpy(path,basename);
  if (depth == 0)
    xml=(char *) GetConfigureBlob(basename,path,&length,exception);
  else
    xml=(char *) FileToBlob(basename,&length,exception);
  if (xml == (char *) NULL)
    xml=AllocateString(LocaleMap);
  *tag='\0';
  token=AllocateString(xml);
  for (q=xml; *q != '\0'; )
  {
    /*
      Interpret XML.
    */
    GetToken(q,&q,token);
    if (*token == '\0')
      break;
    (void) strncpy(keyword,token,MaxTextExtent-1);
    if (LocaleNCompare(keyword,"<!--",4) == 0)
      {
        /*
          Comment element.
        */
        while ((LocaleNCompare(q,"->",2) != 0) && (*q != '\0'))
          GetToken(q,&q,token);
        continue;
      }
    if (LocaleCompare(keyword,"<include") == 0)
      {
        /*
          Include element.
        */
        while ((*token != '>') && (*q != '\0'))
        {
          (void) strncpy(keyword,token,MaxTextExtent-1);
          GetToken(q,&q,token);
          if (*token != '=')
            continue;
          GetToken(q,&q,token);
          if (LocaleCompare(keyword,"locale") == 0)
            {
              if (LocaleCompare(locale,token) != 0)
                break;
              continue;
            }
          if (LocaleCompare(keyword,"file") == 0)
            {
              if (depth > 200)
                (void) ThrowException(exception,ConfigureError,
                  "IncludeElementNestedTooDeeply",path);
              else
                {
                  char
                    filename[MaxTextExtent];

                  GetPathComponent(path,HeadPath,filename);
                  if (*filename != '\0')
                    (void) strcat(filename,DirectorySeparator);
                  (void) strncat(filename,token,MaxTextExtent-
                    strlen(filename)-1);
                  (void) ReadConfigureFile(filename,locale,depth+1,exception);
                }
            }
        }
        continue;
      }
    if (LocaleCompare(keyword,"<locale") == 0)
      {
        /*
          Locale element.
        */
        while ((*token != '>') && (*q != '\0'))
        {
          (void) strncpy(keyword,token,MaxTextExtent-1);
          GetToken(q,&q,token);
          if (*token != '=')
            continue;
          GetToken(q,&q,token);
        }
        continue;
      }
    if (LocaleCompare(keyword,"</locale>") == 0)
      {
        ChopLocaleComponents(tag,1);
        (void) strcat(tag,"/");
        continue;
      }
    if (LocaleCompare(keyword,"<localemap>") == 0)
      continue;
    if (LocaleCompare(keyword,"</localemap>") == 0)
      continue;
    if (LocaleCompare(keyword,"<message") == 0)
      {
        /*
          Message element.
        */
        while ((*token != '>') && (*q != '\0'))
        {
          (void) strncpy(keyword,token,MaxTextExtent-1);
          GetToken(q,&q,token);
          if (*token != '=')
            continue;
          GetToken(q,&q,token);
          if (LocaleCompare(keyword,"name") == 0)
            {
              (void) strncat(tag,token,MaxTextExtent-strlen(tag)-2);
              (void) strcat(tag,"/");
            }
        }
        for (p=q; (*q != '<') && (*q != '\0'); q++);
        while (isspace((int) (*p)))
          p++;
        q--;
        while (isspace((int) (*q)) && (q > p))
          q--;
        (void) strncpy(message,p,q-p+1);
        message[q-p+1]='\0';
        status=PutEntryInHashMap(locale_map,(void *) AcquireString(tag),
          (void *) AcquireString(message));
        if (status == False)
          (void) ThrowException(exception,ConfigureError,"UnableToPutInHashMap",
            tag);
        (void) strncat(tag,message,MaxTextExtent-strlen(tag)-2);
        (void) strcat(tag,"\n");
        continue;
      }
    if (LocaleCompare(keyword,"</message>") == 0)
      {
        ChopLocaleComponents(tag,2);
        (void) strcat(tag,"/");
        continue;
      }
    if (*keyword == '<')
      {
        /*
          Subpath element.
        */
        if (*(keyword+1) == '?')
          continue;
        if (*(keyword+1) == '/')
          {
            ChopLocaleComponents(tag,1);
            (void) strcat(tag,"/");
            continue;
          }
        token[strlen(token)-1]='\0';
        (void) strcpy(token,token+1);
        (void) strncat(tag,token,MaxTextExtent-strlen(message)-2);
        (void) strcat(tag,"/");
        continue;
      }
    GetToken(q,(char **) NULL,token);
    if (*token != '=')
      continue;
  }
  LiberateMemory((void **) &token);
  LiberateMemory((void **) &xml);
  return(True);
}
