/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                       CCCC   OOO   L       OOO   RRRR                       %
%                      C      O   O  L      O   O  R   R                      %
%                      C      O   O  L      O   O  RRRR                       %
%                      C      O   O  L      O   O  R R                        %
%                       CCCC   OOO   LLLLL   OOO   R  R                       %
%                                                                             %
%                                                                             %
%                  Methods to Count the Colors in an Image                    %
%                                                                             %
%                                                                             %
%                              Software Design                                %
%                                John Cristy                                  %
%                                 July 1992                                   %
%                                                                             %
%                                                                             %
%  Copyright (C) 2003 ImageMagick Studio, a non-profit organization dedicated %
%  to making software imaging solutions freely available.                     %
%                                                                             %
%  Permission is hereby granted, free of charge, to any person obtaining a    %
%  copy of this software and associated documentation files ("ImageMagick"),  %
%  to deal in ImageMagick without restriction, including without limitation   %
%  the rights to use, copy, modify, merge, publish, distribute, sublicense,   %
%  and/or sell copies of ImageMagick, and to permit persons to whom the       %
%  ImageMagick is furnished to do so, subject to the following conditions:    %
%                                                                             %
%  The above copyright notice and this permission notice shall be included in %
%  all copies or substantial portions of ImageMagick.                         %
%                                                                             %
%  The software is provided "as is", without warranty of any kind, express or %
%  implied, including but not limited to the warranties of merchantability,   %
%  fitness for a particular purpose and noninfringement.  In no event shall   %
%  ImageMagick Studio be liable for any claim, damages or other liability,    %
%  whether in an action of contract, tort or otherwise, arising from, out of  %
%  or in connection with ImageMagick or the use or other dealings in          %
%  ImageMagick.                                                               %
%                                                                             %
%  Except as contained in this notice, the name of the ImageMagick Studio     %
%  shall not be used in advertising or otherwise to promote the sale, use or  %
%  other dealings in ImageMagick without prior written authorization from the %
%  ImageMagick Studio.                                                        %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%
*/

/*
  Include declarations.
*/
#include "magick/studio.h"
#include "magick/blob.h"
#include "magick/error.h"
#include "magick/quantize.h"
#include "magick/monitor.h"
#include "magick/semaphore.h"
#include "magick/utility.h"

/*
  Define declarations.
*/
#define ColorFilename  "colors.mgk"

/*
  Declare color map.
*/
static char
  *ColorMap = (char *)
    "  <color name=\"AliceBlue\" red=\"240\" green=\"248\" blue=\"255\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"AntiqueWhite\" red=\"250\" green=\"235\" blue=\"215\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"aqua\" red=\"0\" green=\"255\" blue=\"255\" compliance=\"SVG\" />"
    "  <color name=\"aquamarine\" red=\"127\" green=\"255\" blue=\"212\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"azure\" red=\"240\" green=\"255\" blue=\"255\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"beige\" red=\"245\" green=\"245\" blue=\"220\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"bisque\" red=\"255\" green=\"228\" blue=\"196\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"black\" red=\"0\" green=\"0\" blue=\"0\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"BlanchedAlmond\" red=\"255\" green=\"235\" blue=\"205\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"blue\" red=\"0\" green=\"0\" blue=\"255\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"BlueViolet\" red=\"138\" green=\"43\" blue=\"226\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"brown\" red=\"165\" green=\"42\" blue=\"42\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"burlywood\" red=\"222\" green=\"184\" blue=\"135\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"CadetBlue\" red=\"95\" green=\"158\" blue=\"160\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"chartreuse\" red=\"127\" green=\"255\" blue=\"0\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"chocolate\" red=\"210\" green=\"105\" blue=\"30\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"coral\" red=\"255\" green=\"127\" blue=\"80\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"CornflowerBlue\" red=\"100\" green=\"149\" blue=\"237\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"cornsilk\" red=\"255\" green=\"248\" blue=\"220\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"crimson\" red=\"220\" green=\"20\" blue=\"60\" compliance=\"SVG\" />"
    "  <color name=\"cyan\" red=\"0\" green=\"255\" blue=\"255\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"DarkBlue\" red=\"0\" green=\"0\" blue=\"139\" compliance=\"SVG, X11\" />"
    "  <color name=\"DarkCyan\" red=\"0\" green=\"139\" blue=\"139\" compliance=\"SVG, X11\" />"
    "  <color name=\"DarkGoldenrod\" red=\"184\" green=\"134\" blue=\"11\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"DarkGray\" red=\"169\" green=\"169\" blue=\"169\" compliance=\"SVG, X11\" />"
    "  <color name=\"DarkGreen\" red=\"0\" green=\"100\" blue=\"0\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"DarkGrey\" red=\"169\" green=\"169\" blue=\"169\" compliance=\"SVG, X11\" />"
    "  <color name=\"DarkKhaki\" red=\"189\" green=\"183\" blue=\"107\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"DarkMagenta\" red=\"139\" green=\"0\" blue=\"139\" compliance=\"SVG, X11\" />"
    "  <color name=\"DarkOliveGreen\" red=\"85\" green=\"107\" blue=\"47\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"DarkOrange\" red=\"255\" green=\"140\" blue=\"0\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"DarkOrchid\" red=\"153\" green=\"50\" blue=\"204\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"DarkRed\" red=\"139\" green=\"0\" blue=\"0\" compliance=\"SVG, X11\" />"
    "  <color name=\"DarkSalmon\" red=\"233\" green=\"150\" blue=\"122\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"DarkSeaGreen\" red=\"143\" green=\"188\" blue=\"143\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"DarkSlateBlue\" red=\"72\" green=\"61\" blue=\"139\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"DarkSlateGray\" red=\"47\" green=\"79\" blue=\"79\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"DarkSlateGrey\" red=\"47\" green=\"79\" blue=\"79\" compliance=\"SVG, X11\" />"
    "  <color name=\"DarkTurquoise\" red=\"0\" green=\"206\" blue=\"209\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"DarkViolet\" red=\"148\" green=\"0\" blue=\"211\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"DeepPink\" red=\"255\" green=\"20\" blue=\"147\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"DeepSkyBlue\" red=\"0\" green=\"191\" blue=\"255\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"DimGray\" red=\"105\" green=\"105\" blue=\"105\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"DimGrey\" red=\"105\" green=\"105\" blue=\"105\" compliance=\"SVG, X11\" />"
    "  <color name=\"DodgerBlue\" red=\"30\" green=\"144\" blue=\"255\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"firebrick\" red=\"178\" green=\"34\" blue=\"34\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"FloralWhite\" red=\"255\" green=\"250\" blue=\"240\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"ForestGreen\" red=\"34\" green=\"139\" blue=\"34\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"fractal\" red=\"128\" green=\"128\" blue=\"128\" compliance=\"SVG\" />"
    "  <color name=\"fuchsia\" red=\"255\" green=\"0\" blue=\"255\" compliance=\"SVG\" />"
    "  <color name=\"gainsboro\" red=\"220\" green=\"220\" blue=\"220\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"GhostWhite\" red=\"248\" green=\"248\" blue=\"255\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"goldenrod\" red=\"218\" green=\"165\" blue=\"32\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"gray\" red=\"126\" green=\"126\" blue=\"126\" compliance=\"SVG\" />"
    "  <color name=\"green\" red=\"0\" green=\"128\" blue=\"0\" compliance=\"SVG\" />"
    "  <color name=\"grey\" red=\"190\" green=\"190\" blue=\"190\" compliance=\"SVG, X11\" />"
    "  <color name=\"grey0\" red=\"0\" green=\"0\" blue=\"0\" compliance=\"SVG, X11\" />"
    "  <color name=\"grey1\" red=\"3\" green=\"3\" blue=\"3\" compliance=\"SVG, X11\" />"
    "  <color name=\"grey10\" red=\"26\" green=\"26\" blue=\"26 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey100\" red=\"255\" green=\"255\" blue=\"255 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey11\" red=\"28\" green=\"28\" blue=\"28 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey12\" red=\"31\" green=\"31\" blue=\"31 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey13\" red=\"33\" green=\"33\" blue=\"33 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey14\" red=\"36\" green=\"36\" blue=\"36 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey15\" red=\"38\" green=\"38\" blue=\"38 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey16\" red=\"41\" green=\"41\" blue=\"41 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey17\" red=\"43\" green=\"43\" blue=\"43 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey18\" red=\"46\" green=\"46\" blue=\"46 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey19\" red=\"48\" green=\"48\" blue=\"48 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey2\" red=\"5\" green=\"5\" blue=\"5\" compliance=\"SVG, X11\" />"
    "  <color name=\"grey20\" red=\"51\" green=\"51\" blue=\"51 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey21\" red=\"54\" green=\"54\" blue=\"54 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey22\" red=\"56\" green=\"56\" blue=\"56 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey23\" red=\"59\" green=\"59\" blue=\"59 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey24\" red=\"61\" green=\"61\" blue=\"61 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey25\" red=\"64\" green=\"64\" blue=\"64 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey26\" red=\"66\" green=\"66\" blue=\"66 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey27\" red=\"69\" green=\"69\" blue=\"69 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey28\" red=\"71\" green=\"71\" blue=\"71 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey29\" red=\"74\" green=\"74\" blue=\"74 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey3\" red=\"8\" green=\"8\" blue=\"8\" compliance=\"SVG, X11\" />"
    "  <color name=\"grey30\" red=\"77\" green=\"77\" blue=\"77 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey31\" red=\"79\" green=\"79\" blue=\"79 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey32\" red=\"82\" green=\"82\" blue=\"82 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey33\" red=\"84\" green=\"84\" blue=\"84 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey34\" red=\"87\" green=\"87\" blue=\"87 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey35\" red=\"89\" green=\"89\" blue=\"89 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey36\" red=\"92\" green=\"92\" blue=\"92 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey37\" red=\"94\" green=\"94\" blue=\"94 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey38\" red=\"97\" green=\"97\" blue=\"97 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey39\" red=\"99\" green=\"99\" blue=\"99 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey4\" red=\"10\" green=\"10\" blue=\"10 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey40\" red=\"102\" green=\"102\" blue=\"102 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey41\" red=\"105\" green=\"105\" blue=\"105 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey42\" red=\"107\" green=\"107\" blue=\"107 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey43\" red=\"110\" green=\"110\" blue=\"110 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey44\" red=\"112\" green=\"112\" blue=\"112 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey45\" red=\"115\" green=\"115\" blue=\"115 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey46\" red=\"117\" green=\"117\" blue=\"117 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey47\" red=\"120\" green=\"120\" blue=\"120 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey48\" red=\"122\" green=\"122\" blue=\"122 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey49\" red=\"125\" green=\"125\" blue=\"125 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey5\" red=\"13\" green=\"13\" blue=\"13 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey50\" red=\"127\" green=\"127\" blue=\"127 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey51\" red=\"130\" green=\"130\" blue=\"130 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey52\" red=\"133\" green=\"133\" blue=\"133 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey53\" red=\"135\" green=\"135\" blue=\"135 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey54\" red=\"138\" green=\"138\" blue=\"138 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey55\" red=\"140\" green=\"140\" blue=\"140 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey56\" red=\"143\" green=\"143\" blue=\"143 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey57\" red=\"145\" green=\"145\" blue=\"145 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey58\" red=\"148\" green=\"148\" blue=\"148 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey59\" red=\"150\" green=\"150\" blue=\"150 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey6\" red=\"15\" green=\"15\" blue=\"15 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey60\" red=\"153\" green=\"153\" blue=\"153 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey61\" red=\"156\" green=\"156\" blue=\"156 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey62\" red=\"158\" green=\"158\" blue=\"158 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey63\" red=\"161\" green=\"161\" blue=\"161 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey64\" red=\"163\" green=\"163\" blue=\"163 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey65\" red=\"166\" green=\"166\" blue=\"166 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey66\" red=\"168\" green=\"168\" blue=\"168 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey67\" red=\"171\" green=\"171\" blue=\"171 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey68\" red=\"173\" green=\"173\" blue=\"173 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey69\" red=\"176\" green=\"176\" blue=\"176 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey7\" red=\"18\" green=\"18\" blue=\"18 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey70\" red=\"179\" green=\"179\" blue=\"179 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey71\" red=\"181\" green=\"181\" blue=\"181 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey72\" red=\"184\" green=\"184\" blue=\"184 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey73\" red=\"186\" green=\"186\" blue=\"186 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey74\" red=\"189\" green=\"189\" blue=\"189 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey75\" red=\"191\" green=\"191\" blue=\"191 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey76\" red=\"194\" green=\"194\" blue=\"194 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey77\" red=\"196\" green=\"196\" blue=\"196 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey78\" red=\"199\" green=\"199\" blue=\"199 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey79\" red=\"201\" green=\"201\" blue=\"201 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey8\" red=\"20\" green=\"20\" blue=\"20 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey80\" red=\"204\" green=\"204\" blue=\"204 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey81\" red=\"207\" green=\"207\" blue=\"207 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey82\" red=\"209\" green=\"209\" blue=\"209 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey83\" red=\"212\" green=\"212\" blue=\"212 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey84\" red=\"214\" green=\"214\" blue=\"214 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey85\" red=\"217\" green=\"217\" blue=\"217 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey86\" red=\"219\" green=\"219\" blue=\"219 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey87\" red=\"222\" green=\"222\" blue=\"222 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey88\" red=\"224\" green=\"224\" blue=\"224 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey89\" red=\"227\" green=\"227\" blue=\"227 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey9\" red=\"23\" green=\"23\" blue=\"23 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey90\" red=\"229\" green=\"229\" blue=\"229 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey91\" red=\"232\" green=\"232\" blue=\"232 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey92\" red=\"235\" green=\"235\" blue=\"235 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey93\" red=\"237\" green=\"237\" blue=\"237 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey94\" red=\"240\" green=\"240\" blue=\"240 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey95\" red=\"242\" green=\"242\" blue=\"242 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey96\" red=\"245\" green=\"245\" blue=\"245 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey97\" red=\"247\" green=\"247\" blue=\"247 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey98\" red=\"250\" green=\"250\" blue=\"250 \" compliance=\"SVG, X11\" />"
    "  <color name=\"grey99\" red=\"252\" green=\"252\" blue=\"252 \" compliance=\"SVG, X11\" />"
    "  <color name=\"honeydew\" red=\"240\" green=\"255\" blue=\"240\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"HotPink\" red=\"255\" green=\"105\" blue=\"180\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"IndianRed\" red=\"205\" green=\"92\" blue=\"92\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"indigo\" red=\"75\" green=\"0\" blue=\"130\" compliance=\"SVG\" />"
    "  <color name=\"ivory\" red=\"255\" green=\"255\" blue=\"240\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"khaki\" red=\"240\" green=\"230\" blue=\"140\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"lavender\" red=\"230\" green=\"230\" blue=\"250\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"LavenderBlush\" red=\"255\" green=\"240\" blue=\"245\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"LawnGreen\" red=\"124\" green=\"252\" blue=\"0\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"LemonChiffon\" red=\"255\" green=\"250\" blue=\"205\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"LightBlue\" red=\"173\" green=\"216\" blue=\"230\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"LightCoral\" red=\"240\" green=\"128\" blue=\"128\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"LightCyan\" red=\"224\" green=\"255\" blue=\"255\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"LightGoldenrodYellow\" red=\"250\" green=\"250\" blue=\"210\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"LightGray\" red=\"211\" green=\"211\" blue=\"211\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"LightGreen\" red=\"144\" green=\"238\" blue=\"144\" compliance=\"SVG, X11\" />"
    "  <color name=\"LightGrey\" red=\"211\" green=\"211\" blue=\"211\" compliance=\"SVG, X11\" />"
    "  <color name=\"LightPink\" red=\"255\" green=\"182\" blue=\"193\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"LightSalmon\" red=\"255\" green=\"160\" blue=\"122\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"LightSeaGreen\" red=\"32\" green=\"178\" blue=\"170\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"LightSkyBlue\" red=\"135\" green=\"206\" blue=\"250\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"LightSlateGray\" red=\"119\" green=\"136\" blue=\"153\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"LightSlateGrey\" red=\"119\" green=\"136\" blue=\"153\" compliance=\"SVG, X11\" />"
    "  <color name=\"LightSteelBlue\" red=\"176\" green=\"196\" blue=\"222\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"LightYellow\" red=\"255\" green=\"255\" blue=\"224\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"lime\" red=\"0\" green=\"255\" blue=\"0\" compliance=\"SVG\" />"
    "  <color name=\"LimeGreen\" red=\"50\" green=\"205\" blue=\"50\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"linen\" red=\"250\" green=\"240\" blue=\"230\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"magenta\" red=\"255\" green=\"0\" blue=\"255\" compliance=\"SVG\" />"
    "  <color name=\"maroon\" red=\"128\" green=\"0\" blue=\"0\" compliance=\"SVG\" />"
    "  <color name=\"MediumAquamarine\" red=\"102\" green=\"205\" blue=\"170\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"MediumBlue\" red=\"0\" green=\"0\" blue=\"205\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"MediumOrchid\" red=\"186\" green=\"85\" blue=\"211\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"MediumPurple\" red=\"147\" green=\"112\" blue=\"219\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"MediumSeaGreen\" red=\"60\" green=\"179\" blue=\"113\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"MediumSlateBlue\" red=\"123\" green=\"104\" blue=\"238\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"MediumSpringGreen\" red=\"0\" green=\"250\" blue=\"154\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"MediumTurquoise\" red=\"72\" green=\"209\" blue=\"204\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"MediumVioletRed\" red=\"199\" green=\"21\" blue=\"133\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"MidnightBlue\" red=\"25\" green=\"25\" blue=\"112\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"MintCream\" red=\"245\" green=\"255\" blue=\"250\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"MistyRose\" red=\"255\" green=\"228\" blue=\"225\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"moccasin\" red=\"255\" green=\"228\" blue=\"181\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"NavajoWhite\" red=\"255\" green=\"222\" blue=\"173\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"navy\" red=\"0\" green=\"0\" blue=\"128\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"none\" red=\"0\" green=\"0\" blue=\"0\" opacity=\"255\" compliance=\"SVG\" />"
    "  <color name=\"OldLace\" red=\"253\" green=\"245\" blue=\"230\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"olive\" red=\"128\" green=\"128\" blue=\"0\" compliance=\"SVG\" />"
    "  <color name=\"OliveDrab\" red=\"107\" green=\"142\" blue=\"35\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"opaque\" opacity=\"0\" compliance=\"SVG\" />"
    "  <color name=\"orange\" red=\"255\" green=\"165\" blue=\"0\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"OrangeRed\" red=\"255\" green=\"69\" blue=\"0\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"orchid\" red=\"218\" green=\"112\" blue=\"214\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"PaleGoldenrod\" red=\"238\" green=\"232\" blue=\"170\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"PaleGreen\" red=\"152\" green=\"251\" blue=\"152\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"PaleTurquoise\" red=\"175\" green=\"238\" blue=\"238\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"PaleVioletRed\" red=\"219\" green=\"112\" blue=\"147\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"PapayaWhip\" red=\"255\" green=\"239\" blue=\"213\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"PeachPuff\" red=\"255\" green=\"218\" blue=\"185\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"peru\" red=\"205\" green=\"133\" blue=\"63\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"pink\" red=\"255\" green=\"192\" blue=\"203\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"plum\" red=\"221\" green=\"160\" blue=\"221\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"PowderBlue\" red=\"176\" green=\"224\" blue=\"230\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"purple\" red=\"128\" green=\"0\" blue=\"128\" compliance=\"SVG\" />"
    "  <color name=\"red\" red=\"255\" green=\"0\" blue=\"0\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"RosyBrown\" red=\"188\" green=\"143\" blue=\"143\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"RoyalBlue\" red=\"65\" green=\"105\" blue=\"225\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"SaddleBrown\" red=\"139\" green=\"69\" blue=\"19\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"salmon\" red=\"250\" green=\"128\" blue=\"114\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"SandyBrown\" red=\"244\" green=\"164\" blue=\"96\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"SeaGreen\" red=\"46\" green=\"139\" blue=\"87\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"seashell\" red=\"255\" green=\"245\" blue=\"238\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"sienna\" red=\"160\" green=\"82\" blue=\"45\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"silver\" red=\"192\" green=\"192\" blue=\"192\" compliance=\"SVG\" />"
    "  <color name=\"SkyBlue\" red=\"135\" green=\"206\" blue=\"235\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"SlateBlue\" red=\"106\" green=\"90\" blue=\"205\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"SlateGray\" red=\"112\" green=\"128\" blue=\"144\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"SlateGrey\" red=\"112\" green=\"128\" blue=\"144\" compliance=\"SVG, X11\" />"
    "  <color name=\"snow\" red=\"255\" green=\"250\" blue=\"250\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"SpringGreen\" red=\"0\" green=\"255\" blue=\"127\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"SteelBlue\" red=\"70\" green=\"130\" blue=\"180\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"tan\" red=\"210\" green=\"180\" blue=\"140\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"teal\" red=\"0\" green=\"128\" blue=\"128\" compliance=\"SVG\" />"
    "  <color name=\"thistle\" red=\"216\" green=\"191\" blue=\"216\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"tomato\" red=\"255\" green=\"99\" blue=\"71\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"transparent\" opacity=\"255\" compliance=\"SVG\" />"
    "  <color name=\"turquoise\" red=\"64\" green=\"224\" blue=\"208\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"violet\" red=\"238\" green=\"130\" blue=\"238\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"wheat\" red=\"245\" green=\"222\" blue=\"179\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"white\" red=\"255\" green=\"255\" blue=\"255\" compliance=\"SVG, X11\" />"
    "  <color name=\"WhiteSmoke\" red=\"245\" green=\"245\" blue=\"245\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"yellow\" red=\"255\" green=\"255\" blue=\"0\" compliance=\"SVG, X11, XPM\" />"
    "  <color name=\"YellowGreen\" red=\"154\" green=\"205\" blue=\"50\" compliance=\"SVG, X11, XPM\" />"
    "</colormap>";

/*
  Structures.
*/
typedef struct _ColorPacket
{
  PixelPacket
    pixel;

  unsigned short
    index;

  unsigned long
    count;
} ColorPacket;

typedef struct _NodeInfo
{
  struct _NodeInfo
    *child[8];

  ColorPacket
    *list;

  unsigned long
    number_unique;

  unsigned char
    level;
} NodeInfo;

typedef struct _Nodes
{
  NodeInfo
    nodes[NodesInAList];

  struct _Nodes
    *next;
} Nodes;

typedef struct _CubeInfo
{
  NodeInfo
    *root;

  unsigned long
    progress,
    colors,
    free_nodes;

  NodeInfo
    *node_info;

  Nodes
    *node_queue;
} CubeInfo;

/*
  Static declarations.
*/
static SemaphoreInfo
  *color_semaphore = (SemaphoreInfo *) NULL;

static ColorInfo
  *color_list = (ColorInfo *) NULL;

/*
  Forward declarations.
*/
static NodeInfo
  *GetNodeInfo(CubeInfo *,const unsigned int);

static unsigned int
  ReadConfigureFile(const char *,const unsigned long,ExceptionInfo *);

static void
  DestroyColorList(const NodeInfo *),
  Histogram(const Image *,CubeInfo *,const NodeInfo *,FILE *,ExceptionInfo *);

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+   D e s t r o y C o l o r I n f o                                           %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Method DestroyColorInfo deallocates memory associated with the color list.
%
%  The format of the DestroyColorInfo method is:
%
%      DestroyColorInfo(void)
%
%
*/
MagickExport void DestroyColorInfo(void)
{
  ColorInfo
    *color_info;

  register ColorInfo
    *p;

  AcquireSemaphoreInfo(&color_semaphore);
  for (p=color_list; p != (const ColorInfo *) NULL; )
  {
    color_info=p;
    p=p->next;
    if (color_info->path != (char *) NULL)
      LiberateMemory((void **) &color_info->path);
    if (color_info->name != (char *) NULL)
      LiberateMemory((void **) &color_info->name);
    LiberateMemory((void **) &color_info);
  }
  color_list=(ColorInfo *) NULL;
  DestroySemaphoreInfo(&color_semaphore);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+   D e s t r o y C u b e I n f o                                             %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  DestroyCubeInfo() deallocates memory associated with a CubeInfo structure.
%
%  The format of the DestroyCubeInfo method is:
%
%      DestroyCubeInfo(CubeInfo *cube_info)
%
%  A description of each parameter follows:
%
%    o cube_info: The address of a structure of type CubeInfo.
%
%
*/
static void DestroyCubeInfo(CubeInfo *cube_info)
{
  register Nodes
    *nodes;

  /*
    Release color cube tree storage.
  */
  DestroyColorList(cube_info->root);
  do
  {
    nodes=cube_info->node_queue->next;
    LiberateMemory((void **) &cube_info->node_queue);
    cube_info->node_queue=nodes;
  } while (cube_info->node_queue != (Nodes *) NULL);
  LiberateMemory((void **) &cube_info);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+  D e s t r o y C o l o r L i s t                                            %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Method DestroyColorList traverses the color cube tree and frees the list of
%  unique colors.
%
%  The format of the DestroyColorList method is:
%
%      void DestroyColorList(const NodeInfo *node_info)
%
%  A description of each parameter follows.
%
%    o node_info: The address of a structure of type NodeInfo which points to a
%      node in the color cube tree that is to be pruned.
%
%
*/
static void DestroyColorList(const NodeInfo *node_info)
{
  register unsigned int
    id;

  /*
    Traverse any children.
  */
  for (id=0; id < 8; id++)
    if (node_info->child[id] != (NodeInfo *) NULL)
      DestroyColorList(node_info->child[id]);
  if (node_info->list != (ColorPacket *) NULL)
    LiberateMemory((void **) &node_info->list);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+   F u z z y C o l o r C o m p a r e                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  FuzzyColorCompare() returns true if two pixels are identical in color.
%
%  The format of the ColorMatch method is:
%
%      void FuzzyColorCompare(const Image *image,const PixelPacket *p,
%        const PixelPacket *q)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o p: Pixel p.
%
%    o q: Pixel q.
%
%
*/
MagickExport unsigned int FuzzyColorCompare(const Image *image,
  const PixelPacket *p,const PixelPacket *q)
{
  double
    fuzz;

  DoublePixelPacket
    pixel;

  register double
    distance;

  if (image->fuzz == 0.0)
    {
      if (image->colorspace == CMYKColorspace)
        {
          if ((p->red == q->red) && (p->green == q->green) &&
              (p->blue == q->blue) && (p->opacity == q->opacity))
            return(True);
        }
      else
        if ((p->red == q->red) && (p->green == q->green) &&
            (p->blue == q->blue))
          return(True);
    }
  fuzz=image->fuzz*image->fuzz;
  if (image->colorspace != CMYKColorspace)
    pixel.red=p->red-(double) q->red;
  else
    pixel.red=(((double) (MaxRGB-p->red)*(MaxRGB-p->opacity))/MaxRGB)-
      (((double) (MaxRGB-q->red)*(MaxRGB-q->opacity))/MaxRGB);
  distance=pixel.red*pixel.red;
  if (distance > fuzz)
    return(False);
  if (image->colorspace != CMYKColorspace)
    pixel.green=p->green-(double) q->green;
  else
    pixel.green=(((double) (MaxRGB-p->green)*(MaxRGB-p->opacity))/MaxRGB)-
      (((double) (MaxRGB-q->green)*(MaxRGB-q->opacity))/MaxRGB);
  distance+=pixel.green*pixel.green;
  if (distance > fuzz)
    return(False);
  if (image->colorspace != CMYKColorspace)
    pixel.blue=p->blue-(double) q->blue;
  else
    pixel.blue=(((double) (MaxRGB-p->blue)*(MaxRGB-p->opacity))/MaxRGB)-
      (((double) (MaxRGB-q->blue)*(MaxRGB-q->opacity))/MaxRGB);
  distance+=pixel.blue*pixel.blue;
  if (distance > fuzz)
    return(False);
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+   G e t C o l o r I n f o                                                   %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Method GetColorInfo searches the color list for the specified name and if
%  found returns attributes for that color.
%
%  The format of the GetColorInfo method is:
%
%      const PixelPacket *GetColorInfo(const char *name,
%        ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o color_info: Method GetColorInfo searches the color list for the
%      specified name and if found returns attributes for that color.
%
%    o name: The color name.
%
%    o compliance: Define the required color standard.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
MagickExport const ColorInfo *GetColorInfo(const char *name,
  ExceptionInfo *exception)
{
  char
    colorname[MaxTextExtent];

  register char
    *q;

  register ColorInfo
    *p;

  if (color_list == (ColorInfo *) NULL)
    {
      AcquireSemaphoreInfo(&color_semaphore);
      if (color_list == (ColorInfo *) NULL)
        (void) ReadConfigureFile(ColorFilename,0,exception);
      LiberateSemaphoreInfo(&color_semaphore);
    }
  if ((name == (const char *) NULL) || (LocaleCompare(name,"*") == 0))
    return((const ColorInfo *) color_list);
  /*
    Search for named color.
  */
  (void) strncpy(colorname,name,MaxTextExtent-1);
  for (q=colorname; *q != '\0'; q++)
  {
    if (*q != ' ')
      continue;
    (void) strcpy(q,q+1);
    q--;
  }
  AcquireSemaphoreInfo(&color_semaphore);
  for (p=color_list; p != (ColorInfo *) NULL; p=p->next)
    if (LocaleCompare(colorname,p->name) == 0)
      break;
  if (p == (ColorInfo *) NULL)
    (void) ThrowException(exception,OptionWarning,"UnrecognizedColor",name);
  else
    if (p != color_list)
      {
        /*
          Self-adjusting list.
        */
        if (p->previous != (ColorInfo *) NULL)
          p->previous->next=p->next;
        if (p->next != (ColorInfo *) NULL)
          p->next->previous=p->previous;
        p->previous=(ColorInfo *) NULL;
        p->next=color_list;
        color_list->previous=p;
        color_list=p;
      }
  LiberateSemaphoreInfo(&color_semaphore);
  return((const ColorInfo *) p);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t C o l o r l i s t                                                   %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Method GetColorList returns any colors that match the specified pattern
%  and color standard.
%
%  The format of the GetColorList function is:
%
%      filelist=GetColorList(const char *pattern,unsigned long *number_colors)
%
%  A description of each parameter follows:
%
%    o filelist: Method GetColorList returns a list of colors that match the
%      specified pattern and color standard.
%
%    o pattern: Specifies a pointer to a text string containing a pattern.
%
%    o number_colors:  This integer returns the number of colors in the list.
%
%
*/
MagickExport char **GetColorList(const char *pattern,
  unsigned long *number_colors)
{
  char
    **colorlist;

  ExceptionInfo
    exception;

  register long
    i;

  register const ColorInfo
    *p;

  /*
    Allocate color list.
  */
  assert(pattern != (char *) NULL);
  assert(number_colors != (unsigned long *) NULL);
  *number_colors=0;
  GetExceptionInfo(&exception);
  p=GetColorInfo("*",&exception);
  DestroyExceptionInfo(&exception);
  if (p == (const ColorInfo *) NULL)
    return((char **) NULL);
  i=0;
  for (p=color_list; p != (const ColorInfo *) NULL; p=p->next)
    i++;
  colorlist=(char **) AcquireMemory(i*sizeof(char *));
  if (colorlist == (char **) NULL)
    return((char **) NULL);
  i=0;
  for (p=color_list; p != (const ColorInfo *) NULL; p=p->next)
  {
    if (p->stealth)
      continue;
    if (GlobExpression(p->name,pattern))
      colorlist[i++]=AllocateString(p->name);
  }
  *number_colors=i;
  return(colorlist);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+   G e t C o l o r T u p l e                                                 %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetColorTuple() returns a color as a color tuple string.
%
%  The format of the GetColorTuple method is:
%
%      GetColorTuple(const PixelPacket *color,const unsigned int depth,
%        const unsigned int matte,const unsigned int hex,char *tuple)
%
%  A description of each parameter follows.
%
%    o color: The color.
%
%    o depth: The color depth.
%
%    o matte: A value other than zero returns the opacity in the tuple.
%
%    o hex: A value other than zero returns the tuple in a hexidecimal format.
%
%    o tuple: Return the color tuple as this string.
%
%
*/
MagickExport void GetColorTuple(const PixelPacket *color,
  const unsigned int depth,const unsigned int matte,const unsigned int hex,
  char *tuple)
{
  assert(color != (const PixelPacket *) NULL);
  assert(tuple != (char *) NULL);
  if (matte)
    {
      if (depth <= 8)
        {
          FormatString(tuple,hex ? "#%02X%02X%02X%02X" : "(%3u,%3u,%3u,%3u)",
            ScaleQuantumToChar(color->red),ScaleQuantumToChar(color->green),
            ScaleQuantumToChar(color->blue),ScaleQuantumToChar(color->opacity));
          return;
        }
      if (depth <= 16)
        {
          FormatString(tuple,hex ? "#%04X%04X%04X%04X" : "(%5u,%5u,%5u,%5u)",
            ScaleQuantumToShort(color->red),ScaleQuantumToShort(color->green),
            ScaleQuantumToShort(color->blue),
            ScaleQuantumToShort(color->opacity));
          return;
        }
      FormatString(tuple,
        hex ? "#%08lX%08lX%08lX%08lX" : "(%10lu,%10lu,%10lu,%10lu)",
        ScaleQuantumToLong(color->red),ScaleQuantumToLong(color->green),
        ScaleQuantumToLong(color->blue),ScaleQuantumToLong(color->opacity));
      return;
    }
  if (depth <= 8)
    {
      FormatString(tuple,hex ? "#%02X%02X%02X" : "(%3u,%3u,%3u)",
        ScaleQuantumToChar(color->red),ScaleQuantumToChar(color->green),
        ScaleQuantumToChar(color->blue));
      return;
    }
  if (depth <= 16)
    {
      FormatString(tuple,hex ? "#%04X%04X%04X" : "(%5u,%5u,%5u)",
        ScaleQuantumToShort(color->red),ScaleQuantumToShort(color->green),
        ScaleQuantumToShort(color->blue));
      return;
    }
  FormatString(tuple,hex ? "#%08lX%08lX%08lX" : "(%10lu,%10lu,%10lu)",
    ScaleQuantumToLong(color->red),ScaleQuantumToLong(color->green),
    ScaleQuantumToLong(color->blue));
  return;
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+   G e t C u b e I n f o                                                     %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Method GetCubeInfo initialize the CubeInfo data structure.
%
%  The format of the GetCubeInfo method is:
%
%      cube_info=GetCubeInfo()
%
%  A description of each parameter follows.
%
%    o cube_info: A pointer to the Cube structure.
%
%
*/
static CubeInfo *GetCubeInfo(void)
{
  CubeInfo
    *cube_info;

  /*
    Initialize tree to describe color cube.
  */
  cube_info=(CubeInfo *) AcquireMemory(sizeof(CubeInfo));
  if (cube_info == (CubeInfo *) NULL)
    return((CubeInfo *) NULL);
  (void) memset(cube_info,0,sizeof(CubeInfo));
  /*
    Initialize root node.
  */
  cube_info->root=GetNodeInfo(cube_info,0);
  if (cube_info->root == (NodeInfo *) NULL)
    return((CubeInfo *) NULL);
  return(cube_info);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+  G e t N o d e I n f o                                                      %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Method GetNodeInfo allocates memory for a new node in the color cube tree
%  and presets all fields to zero.
%
%  The format of the GetNodeInfo method is:
%
%      node_info=GetNodeInfo(cube_info,level)
%
%  A description of each parameter follows.
%
%    o cube_info: A pointer to the CubeInfo structure.
%
%    o level: Specifies the level in the storage_class the node resides.
%
%
*/
static NodeInfo *GetNodeInfo(CubeInfo *cube_info,const unsigned int level)
{
  NodeInfo
    *node_info;

  if (cube_info->free_nodes == 0)
    {
      Nodes
        *nodes;

      /*
        Allocate a new nodes of nodes.
      */
      nodes=(Nodes *) AcquireMemory(sizeof(Nodes));
      if (nodes == (Nodes *) NULL)
        return((NodeInfo *) NULL);
      nodes->next=cube_info->node_queue;
      cube_info->node_queue=nodes;
      cube_info->node_info=nodes->nodes;
      cube_info->free_nodes=NodesInAList;
    }
  cube_info->free_nodes--;
  node_info=cube_info->node_info++;
  (void) memset(node_info,0,sizeof(NodeInfo));
  node_info->level=level;
  return(node_info);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%  G e t N u m b e r C o l o r s                                              %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Method GetNumberColors returns the number of unique colors in an image.
%
%  The format of the GetNumberColors method is:
%
%      unsigned long GetNumberColors(const Image *image,FILE *file,
%        ExceptionInfo *exception)
%
%  A description of each parameter follows.
%
%    o image: The image.
%
%    o file:  Write a histogram of the color distribution to this file handle.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
MagickExport unsigned long GetNumberColors(const Image *image,FILE *file,
  ExceptionInfo *exception)
{
#define ComputeImageColorsText  "  Compute image colors...  "

  CubeInfo
    *cube_info;

  long
    y;

  NodeInfo
    *node_info;

  register const PixelPacket
    *p;

  register long
    i,
    x;

  register unsigned int
    id,
    index,
    level;

  unsigned long
    number_colors;

  /*
    Initialize color description tree.
  */
  assert(image != (const Image *) NULL);
  assert(image->signature == MagickSignature);
  cube_info=GetCubeInfo();
  if (cube_info == (CubeInfo *) NULL)
    {
      (void) ThrowException(exception,ResourceLimitError,"MemoryAllocationFailed",
        "UnableToDetermineTheNumberOfImageColors");
      return(0);
    }
  for (y=0; y < (long) image->rows; y++)
  {
    p=AcquireImagePixels(image,0,y,image->columns,1,exception);
    if (p == (const PixelPacket *) NULL)
      return(False);
    for (x=0; x < (long) image->columns; x++)
    {
      /*
        Start at the root and proceed level by level.
      */
      node_info=cube_info->root;
      index=MaxTreeDepth-1;
      for (level=1; level <= MaxTreeDepth; level++)
      {
        id=(unsigned int)
          (((ScaleQuantumToChar(p->red) >> index) & 0x01) << 2 |
           ((ScaleQuantumToChar(p->green) >> index) & 0x01) << 1 |
           ((ScaleQuantumToChar(p->blue) >> index) & 0x01));
        if (node_info->child[id] == (NodeInfo *) NULL)
          {
            node_info->child[id]=GetNodeInfo(cube_info,level);
            if (node_info->child[id] == (NodeInfo *) NULL)
              {
                (void) ThrowException(exception,ResourceLimitError,
                  "MemoryAllocationFailed",
                  "UnableToDetermineTheNumberOfImageColors");
                return(0);
              }
          }
        node_info=node_info->child[id];
        index--;
        if (level != MaxTreeDepth)
          continue;
        for (i=0; i < (long) node_info->number_unique; i++)
          if (ColorMatch(p,&node_info->list[i].pixel))
            break;
        if (i < (long) node_info->number_unique)
          {
            node_info->list[i].count++;
            continue;
          }
        if (node_info->number_unique == 0)
          node_info->list=(ColorPacket *) AcquireMemory(sizeof(ColorPacket));
        else
          ReacquireMemory((void **) &node_info->list,
            (i+1)*sizeof(ColorPacket));
        if (node_info->list == (ColorPacket *) NULL)
          {
            (void) ThrowException(exception,ResourceLimitError,
              "MemoryAllocationFailed",
              "UnableToDetermineTheNumberOfImageColors");
            return(0);
          }
        node_info->list[i].pixel=(*p);
        node_info->list[i].count=1;
        node_info->number_unique++;
        cube_info->colors++;
      }
      p++;
    }
    if (QuantumTick(y,image->rows))
      if (!MagickMonitor(ComputeImageColorsText,y,image->rows,exception))
        break;
  }
  if (file != (FILE *) NULL)
    {
      (void) fprintf(file,"\n");
      Histogram(image,cube_info,cube_info->root,file,exception);
      (void) fflush(file);
    }
  number_colors=cube_info->colors;
  DestroyCubeInfo(cube_info);
  return(number_colors);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+  H i s t o g r a m                                                          %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Method Histogram traverses the color cube tree and produces a list of
%  unique pixel field values and the number of times each occurs in the image.
%
%  The format of the Histogram method is:
%
%      void Histogram(const Image *image,CubeInfo *cube_info,
%        const NodeInfo *node_info,FILE *file,ExceptionInfo *exception
%
%  A description of each parameter follows.
%
%    o cube_info: A pointer to the CubeInfo structure.
%
%    o node_info: The address of a structure of type NodeInfo which points to a
%      node in the color cube tree that is to be pruned.
%
%
*/
static void Histogram(const Image *image,CubeInfo *cube_info,
  const NodeInfo *node_info,FILE *file,ExceptionInfo *exception)
{
#define HistogramImageTag  "HistogramImageTag"

  register unsigned int
    id;

  /*
    Traverse any children.
  */
  for (id=0; id < 8; id++)
    if (node_info->child[id] != (NodeInfo *) NULL)
      Histogram(image,cube_info,node_info->child[id],file,exception);
  if (node_info->level == MaxTreeDepth)
    {
      char
        name[MaxTextExtent],
        tuple[MaxTextExtent];

      register ColorPacket
        *p;

      register long
        i;

      p=node_info->list;
      for (i=0; i < (long) node_info->number_unique; i++)
      {
        GetColorTuple(&p->pixel,image->depth,image->matte,False,tuple);
        (void) fprintf(file,"%10lu: %.1024s  ",p->count,tuple);
        (void) fprintf(file,"  ");
        (void) QueryColorname(image,&p->pixel,SVGCompliance,name,exception);
        (void) fprintf(file,"%.1024s",name);
        (void) fprintf(file,"\n");
        p++;
      }
      if (QuantumTick(cube_info->progress,cube_info->colors))
        (void) MagickMonitor(HistogramImageTag,cube_info->progress,
          cube_info->colors,exception);
      cube_info->progress++;
    }
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%     I s G r a y I m a g e                                                   %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  IsGrayImage() returns True if all the pixels in the image have the same
%  red, green, and blue intensities.
%
%  The format of the IsGrayImage method is:
%
%      unsigned int IsGrayImage(const Image *image,ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o status: Method IsGrayImage returns True if the image is grayscale
%      otherwise False is returned.
%
%    o image: The image.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
MagickExport unsigned int IsGrayImage(const Image *image,
  ExceptionInfo *exception)
{
  long
    y;

  register const PixelPacket
    *p;

  register long
    i,
    x;

  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  switch (image->storage_class)
  {
    case DirectClass:
    case UndefinedClass:
    {
      for (y=0; y < (long) image->rows; y++)
      {
        p=AcquireImagePixels(image,0,y,image->columns,1,exception);
        if (p == (const PixelPacket *) NULL)
          return(False);
        for (x=(long) image->columns; x > 0; x--)
        {
          if (!IsGray(*p))
            return(False);
          p++;
        }
      }
      break;
    }
    case PseudoClass:
    {
      for (i=0; i < (long) image->colors; i++)
        if (!IsGray(image->colormap[i]))
          return(False);
      break;
    }
  }
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%   I s M o n o c h r o m e I m a g e                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  IsMonochromeImage() returns True if all the pixels in the image have
%  the same red, green, and blue intensities and the intensity is either
%  0 or MaxRGB.
%
%  The format of the IsMonochromeImage method is:
%
%      unsigned int IsMonochromeImage(const Image *image,
%        ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
MagickExport unsigned int IsMonochromeImage(const Image *image,
  ExceptionInfo *exception)
{
  long
    y;

  register const PixelPacket
    *p;

  register long
    i,
    x;

  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  switch (image->storage_class)
  {
    case DirectClass:
    case UndefinedClass:
    {
      for (y=0; y < (long) image->rows; y++)
      {
        p=AcquireImagePixels(image,0,y,image->columns,1,exception);
        if (p == (const PixelPacket *) NULL)
          return(False);
        for (x=(long) image->columns; x > 0; x--)
        {
          if (!IsGray(*p))
            return(False);
          if ((p->red != 0) && (p->red != MaxRGB))
            return(False);
          p++;
        }
      }
      break;
    }
    case PseudoClass:
    {
      for (i=0; i < (long) image->colors; i++)
      {
        if ((image->colormap[i].red != 0) && (image->colormap[i].red != MaxRGB))
          return(False);
        if (!IsGray(image->colormap[i]))
          return(False);
      }
      break;
    }
  }
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%     I s O p a q u e I m a g e                                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  IsOpaqueImage() returns True if none of the pixels in the image have an
%  opacity value other than opaque (0).
%
%  The format of the IsOpaqueImage method is:
%
%      unsigned int IsOpaqueImage(const Image *image,ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o status: Method IsOpaqueImage returns False if the image has one or more
%      pixels that are transparent otherwise True is returned.
%
%    o image: The image.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
MagickExport unsigned int IsOpaqueImage(const Image *image,
  ExceptionInfo *exception)
{
  long
    y;

  register const PixelPacket
    *p;

  register long
    x;

  /*
    Determine if image is grayscale.
  */
  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (!image->matte)
    return(True);
  for (y=0; y < (long) image->rows; y++)
  {
    p=AcquireImagePixels(image,0,y,image->columns,1,exception);
    if (p == (const PixelPacket *) NULL)
      return(False);
    for (x=0; x < (long) image->columns; x++)
    {
      if (p->opacity != OpaqueOpacity)
        return(False);
      p++;
    }
  }
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%  I s P a l e t t e I m a g e                                                %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Method IsPaletteImage returns True if the image is PseudoClass and has 256
%  unique colors or less.
%
%  The format of the IsPaletteImage method is:
%
%      unsigned int IsPaletteImage(const Image *image,ExceptionInfo *exception)
%
%  A description of each parameter follows.
%
%    o status:  Method IsPaletteImage returns True is the image is
%      PseudoClass or has 256 color or less.
%
%    o image: The image.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
MagickExport unsigned int IsPaletteImage(const Image *image,
  ExceptionInfo *exception)
{
  CubeInfo
    *cube_info;

  long
    y;

  register const PixelPacket
    *p;

  register long
    x;

  register NodeInfo
    *node_info;

  register long
    i;

  unsigned long
    id,
    index,
    level;

  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if ((image->storage_class == PseudoClass) && (image->colors <= 256))
    return(True);
  if (image->storage_class == PseudoClass)
    return(False);
  /*
    Initialize color description tree.
  */
  cube_info=GetCubeInfo();
  if (cube_info == (CubeInfo *) NULL)
    {
      (void) ThrowException(exception,ResourceLimitError,"MemoryAllocationFailed",
        "UnableToDetermineImageClass");
      return(False);
    }
  for (y=0; y < (long) image->rows; y++)
  {
    p=AcquireImagePixels(image,0,y,image->columns,1,exception);
    if (p == (const PixelPacket *) NULL)
      return(False);
    for (x=0; x < (long) image->columns; x++)
    {
      /*
        Start at the root and proceed level by level.
      */
      node_info=cube_info->root;
      index=MaxTreeDepth-1;
      for (level=1; level < MaxTreeDepth; level++)
      {
        id=(unsigned int)
          (((ScaleQuantumToChar(p->red) >> index) & 0x01) << 2 |
           ((ScaleQuantumToChar(p->green) >> index) & 0x01) << 1 |
           ((ScaleQuantumToChar(p->blue) >> index) & 0x01));
        if (node_info->child[id] == (NodeInfo *) NULL)
          {
            node_info->child[id]=GetNodeInfo(cube_info,level);
            if (node_info->child[id] == (NodeInfo *) NULL)
              {
                (void) ThrowException(exception,ResourceLimitError,
                  "MemoryAllocationFailed","UnableToDetermineImageClass");
                return(False);
              }
          }
        node_info=node_info->child[id];
        index--;
      }
      for (i=0; i < (long) node_info->number_unique; i++)
        if (ColorMatch(p,&node_info->list[i].pixel))
          break;
      if (i == (long) node_info->number_unique)
        {
          /*
            Add this unique color to the color list.
          */
          if (node_info->number_unique == 0)
            node_info->list=(ColorPacket *) AcquireMemory(sizeof(ColorPacket));
          else
            ReacquireMemory((void **) &node_info->list,
              (i+1)*sizeof(ColorPacket));
          if (node_info->list == (ColorPacket *) NULL)
            {
              (void) ThrowException(exception,ResourceLimitError,
                "MemoryAllocationFailed","UnableToDetermineImageClass");
              return(False);
            }
          node_info->list[i].pixel=(*p);
          node_info->list[i].index=(unsigned short) cube_info->colors++;
          node_info->number_unique++;
          if (cube_info->colors > 256)
            {
              DestroyCubeInfo(cube_info);
              return(False);
            }
        }
      p++;
    }
  }
  DestroyCubeInfo(cube_info);
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%  L i s t C o l o r I n f o                                                  %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  ListColorInfo() lists color names to the specified file.  Color names
%  are a convenience.  Rather than defining a color by its red, green, and
%  blue intensities just use a color name such as white, blue, or yellow.
%
%  The format of the ListColorInfo method is:
%
%      unsigned int ListColorInfo(FILE *file,ExceptionInfo *exception)
%
%  A description of each parameter follows.
%
%    o file:  List color names to this file handle.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
MagickExport unsigned int ListColorInfo(FILE *file,ExceptionInfo *exception)
{
  register long
    i;

  register const ColorInfo
    *p;

  if (file == (const FILE *) NULL)
    file=stdout;
  (void) GetColorInfo("*",exception);
  AcquireSemaphoreInfo(&color_semaphore);
  for (p=color_list; p != (const ColorInfo *) NULL; p=p->next)
  {
    if ((p->previous == (ColorInfo *) NULL) ||
        (LocaleCompare(p->path,p->previous->path) != 0))
      {
        if (p->previous != (ColorInfo *) NULL)
          (void) fprintf(file,"\n");
        if (p->path != (char *) NULL)
          (void) fprintf(file,"Path: %.1024s\n\n",p->path);
        (void) fprintf(file,
          "Name                   Color                          Compliance\n");
        (void) fprintf(file,"-------------------------------------------------"
          "------------------------------\n");
      }
    if (p->stealth)
      continue;
    (void) fprintf(file,"%.1024s",p->name);
    for (i=(long) strlen(p->name); i <= 22; i++)
      (void) fprintf(file," ");
    if (p->color.opacity == OpaqueOpacity)
      (void) fprintf(file,"rgb(%5d,%5d,%5d)          ",p->color.red,
        p->color.green,p->color.blue);
    else
      (void) fprintf(file,"rgba(%5d,%5d,%5d,%5d)  ",p->color.red,p->color.green,
        p->color.blue,p->color.opacity);
    if (p->compliance & SVGCompliance)
      (void) fprintf(file,"SVG ");
    if (p->compliance & X11Compliance)
      (void) fprintf(file,"X11 ");
    if (p->compliance & XPMCompliance)
      (void) fprintf(file,"XPM ");
    (void) fprintf(file,"\n");
  }
  (void) fflush(file);
  LiberateSemaphoreInfo(&color_semaphore);
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   Q u e r y C o l o r D a t a b a s e                                       %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  QueryColorDatabase() returns the red, green, blue, and opacity intensities
%  for a given color name.
%
%  The format of the QueryColorDatabase method is:
%
%      unsigned int QueryColorDatabase(const char *name,PixelPacket *color,
%        ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o name: The color name (e.g. white, blue, yellow).
%
%    o color: The red, green, blue, and opacity intensities values of the
%      named color in this structure.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
MagickExport unsigned int QueryColorDatabase(const char *name,
  PixelPacket *color,ExceptionInfo *exception)
{
  double
    scale;

  DoublePixelPacket
    pixel;

  GeometryInfo
    geometry_info;

  register const ColorInfo
    *p;

  register long
    i;

  unsigned int
    flags;

  /*
    Initialize color return value.
  */
  assert(color != (PixelPacket *) NULL);
  (void) memset(color,0,sizeof(PixelPacket));
  color->opacity=TransparentOpacity;
  if ((name == (char *) NULL) || (*name == '\0'))
    name=BackgroundColor;
  while (isspace((int) (*name)))
    name++;
  if (*name == '#')
    {
      char
        c;

      long
        n;

      LongPixelPacket
        pixel;

      unsigned int
        range;

      memset(&pixel,0,sizeof(LongPixelPacket));
      name++;
      for (n=0; isxdigit((int) name[n]); n++);
      if ((n == 3) || (n == 6) || (n == 9) || (n == 12))
        {
          /*
            Parse RGB specification.
          */
          n/=3;
          do
          {
            pixel.red=pixel.green;
            pixel.green=pixel.blue;
            pixel.blue=0;
            for (i=n-1; i >= 0; i--)
            {
              c=(*name++);
              pixel.blue<<=4;
              if ((c >= '0') && (c <= '9'))
                pixel.blue|=c-'0';
              else
                if ((c >= 'A') && (c <= 'F'))
                  pixel.blue|=c-('A'-10);
                else
                  if ((c >= 'a') && (c <= 'f'))
                    pixel.blue|=c-('a'-10);
                  else
                    return(False);
            }
          } while (isxdigit((int) *name));
        }
      else
        if ((n != 4) && (n != 8) && (n != 16) && (n != 32))
          {
            ThrowException(exception,OptionWarning,"UnrecognizedColor",name);
            return(False);
          }
        else
          {
            /*
              Parse RGBA specification.
            */
            n/=4;
            do
            {
              pixel.red=pixel.green;
              pixel.green=pixel.blue;
              pixel.blue=pixel.opacity;
              pixel.opacity=0;
              for (i=n-1; i >= 0; i--)
              {
                c=(*name++);
                pixel.opacity<<=4;
                if ((c >= '0') && (c <= '9'))
                  pixel.opacity|=c-'0';
                else
                  if ((c >= 'A') && (c <= 'F'))
                    pixel.opacity|=c-('A'-10);
                  else
                    if ((c >= 'a') && (c <= 'f'))
                      pixel.opacity|=c-('a'-10);
                    else
                      return(False);
              }
            } while (isxdigit((int) *name));
          }
      n<<=2;
      range=(1UL << n)-1;
      if (n == 32)
        range=4294967295UL;
      color->red=(Quantum) (((double) MaxRGB*pixel.red)/range+0.5);
      color->green=(Quantum) (((double) MaxRGB*pixel.green)/range+0.5);
      color->blue=(Quantum) (((double) MaxRGB*pixel.blue)/range+0.5);
      color->opacity=OpaqueOpacity;
      if ((n != 3) && (n != 6) && (n != 9) && (n != 12) && (n != 24))
        color->opacity=(Quantum) (((double) MaxRGB*pixel.opacity)/range+0.5);
      return(True);
    }
  if (LocaleNCompare(name,"rgb(",4) == 0)
    {
      flags=ParseGeometry(name+4,&geometry_info);
      pixel.red=geometry_info.rho;
      pixel.green=geometry_info.sigma;
      if (!(flags & SigmaValue))
        pixel.green=pixel.red;
      pixel.blue=geometry_info.xi;
      if (!(flags & XiValue))
        pixel.blue=pixel.red;
      scale=flags & PercentValue ? ScaleQuantumToChar(MaxRGB)/100.0 : 1.0;
      color->red=ScaleCharToQuantum(scale*pixel.red);
      color->green=ScaleCharToQuantum(scale*pixel.green);
      color->blue=ScaleCharToQuantum(scale*pixel.blue);
      color->opacity=OpaqueOpacity;
      return(True);
    }
  if (LocaleNCompare(name,"rgba(",5) == 0)
    {
      flags=ParseGeometry(name+5,&geometry_info);
      pixel.red=geometry_info.rho;
      pixel.green=geometry_info.sigma;
      if (!(flags & SigmaValue))
        pixel.green=pixel.red;
      pixel.blue=geometry_info.xi;
      if (!(flags & XiValue))
        pixel.blue=pixel.red;
      pixel.opacity=geometry_info.psi;
      if (!(flags & XiValue))
        pixel.opacity=OpaqueOpacity;
      scale=flags & PercentValue ? ScaleQuantumToChar(MaxRGB)/100.0 : 1.0;
      color->red=ScaleCharToQuantum(scale*pixel.red);
      color->green=ScaleCharToQuantum(scale*pixel.green);
      color->blue=ScaleCharToQuantum(scale*pixel.blue);
      color->opacity=ScaleCharToQuantum(scale*pixel.opacity);
      return(True);
    }
  p=GetColorInfo(name,exception);
  if (p == (const ColorInfo *) NULL)
    return(False);
  if ((LocaleCompare(p->name,"opaque") == 0) ||
      (LocaleCompare(p->name,"transparent") == 0))
    {
      color->opacity=p->color.opacity;
      return(True);
    }
  *color=p->color;
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%  Q u e r y C o l o r n a m e                                                %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  QueryColorname() returns a named color for the given color intensity.  If
%  an exact match is not found, a hex value is return instead.  For example
%  an intensity of rgb:(0,0,0) returns black whereas rgb:(223,223,223)
%  returns #dfdfdf.
%
%  The format of the QueryColorname method is:
%
%      unsigned int QueryColorname(const Image *image,const PixelPacket *color,
%        const ComplianceType compliance,char *name,ExceptionInfo *exception)
%
%  A description of each parameter follows.
%
%    o image: The image.
%
%    o color: The color intensities.
%
%    o Compliance: Adhere to this color standard: SVG, X11, or XPM.
%
%    o name: Return the color name or hex value.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
MagickExport unsigned int QueryColorname(const Image *image,
  const PixelPacket *color,const ComplianceType compliance,char *name,
  ExceptionInfo *exception)
{
  register const ColorInfo
    *p;

  unsigned int
    depth,
    matte;

  *name='\0';
  p=GetColorInfo("*",exception);
  if (p != (const ColorInfo *) NULL)
    {
      for (p=color_list; p != (const ColorInfo *) NULL; p=p->next)
      {
        if (!(p->compliance & compliance))
          continue;
        if ((p->color.red != color->red) || (p->color.green != color->green) ||
            (p->color.blue != color->blue) ||
            (p->color.opacity != color->opacity))
          continue;
        (void) strncpy(name,p->name,MaxTextExtent-1);
        return(True);
      }
    }
  matte=compliance != XPMCompliance ? image->matte : False;
  depth=compliance != XPMCompliance ? image->depth : Min(image->depth,16);
  GetColorTuple(color,depth,matte,True,name);
  return(False);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+   R e a d C o n f i g u r e F i l e                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Method ReadConfigureFile reads the color configuration file which maps
%  color strings with a particular image format.
%
%  The format of the ReadConfigureFile method is:
%
%      unsigned int ReadConfigureFile(const char *basename,
%        const unsigned long depth,ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o status: Method ReadConfigureFile returns True if at least one color
%      is defined otherwise False.
%
%    o basename:  The color configuration filename.
%
%    o depth: depth of <include /> statements.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
static unsigned int ReadConfigureFile(const char *basename,
  const unsigned long depth,ExceptionInfo *exception)
{
  char
    keyword[MaxTextExtent],
    path[MaxTextExtent],
    *q,
    *token,
    *xml;

  size_t
    length;

  /*
    Read the color configure file.
  */
  (void) strcpy(path,basename);
  if (depth == 0)
    xml=(char *) GetConfigureBlob(basename,path,&length,exception);
  else
    xml=(char *) FileToBlob(basename,&length,exception);
  if (xml == (char *) NULL)
    xml=AllocateString(ColorMap);
  token=AllocateString(xml);
  for (q=xml; *q != '\0'; )
  {
    /*
      Interpret XML.
    */
    GetToken(q,&q,token);
    if (*token == '\0')
      break;
    (void) strncpy(keyword,token,MaxTextExtent-1);
    if (LocaleNCompare(keyword,"<!--",4) == 0)
      {
        /*
          Comment element.
        */
        while ((LocaleNCompare(q,"->",2) != 0) && (*q != '\0'))
          GetToken(q,&q,token);
        continue;
      }
    if (LocaleCompare(keyword,"<include") == 0)
      {
        /*
          Include element.
        */
        while ((*token != '>') && (*q != '\0'))
        {
          (void) strncpy(keyword,token,MaxTextExtent-1);
          GetToken(q,&q,token);
          if (*token != '=')
            continue;
          GetToken(q,&q,token);
          if (LocaleCompare(keyword,"file") == 0)
            {
              if (depth > 200)
                (void) ThrowException(exception,ConfigureError,
                  "IncludeElementNestedTooDeeply",path);
              else
                {
                  char
                    filename[MaxTextExtent];

                  GetPathComponent(path,HeadPath,filename);
                  if (*filename != '\0')
                    (void) strcat(filename,DirectorySeparator);
                  (void) strncat(filename,token,MaxTextExtent-
                    strlen(filename)-1);
                  (void) ReadConfigureFile(filename,depth+1,exception);
                }
              if (color_list != (ColorInfo *) NULL)
                while (color_list->next != (ColorInfo *) NULL)
                  color_list=color_list->next;
            }
        }
        continue;
      }
    if (LocaleCompare(keyword,"<color") == 0)
      {
        ColorInfo
          *color_info;

        /*
          Allocate memory for the color list.
        */
        color_info=(ColorInfo *) AcquireMemory(sizeof(ColorInfo));
        if (color_info == (ColorInfo *) NULL)
          MagickFatalError(ResourceLimitFatalError,"MemoryAllocationFailed",
            "UnableToAllocateColorInfo");
        (void) memset(color_info,0,sizeof(ColorInfo));
        color_info->path=AcquireString(path);
        color_info->signature=MagickSignature;
        if (color_list == (ColorInfo *) NULL)
          {
            color_list=color_info;
            continue;
          }
        color_list->next=color_info;
        color_info->previous=color_list;
        color_list=color_list->next;
        continue;
      }
    if (color_list == (ColorInfo *) NULL)
      continue;
    GetToken(q,(char **) NULL,token);
    if (*token != '=')
      continue;
    GetToken(q,&q,token);
    GetToken(q,&q,token);
    switch (*keyword)
    {
      case 'B':
      case 'b':
      {
        if (LocaleCompare((char *) keyword,"blue") == 0)
          {
            color_list->color.blue=ScaleCharToQuantum(atol(token));
            break;
          }
        break;
      }
      case 'C':
      case 'c':
      {
        if (LocaleCompare((char *) keyword,"compliance") == 0)
          {
            long
              compliance;

            compliance=color_list->compliance;
            if (GlobExpression(token,"*SVG*"))
              compliance|=SVGCompliance;
            if (GlobExpression(token,"*X11*"))
              compliance|=X11Compliance;
            if (GlobExpression(token,"*XPM*"))
              compliance|=XPMCompliance;
            color_list->compliance=(ComplianceType) compliance;
            break;
          }
        break;
      }
      case 'G':
      case 'g':
      {
        if (LocaleCompare((char *) keyword,"green") == 0)
          {
            color_list->color.green=ScaleCharToQuantum(atol(token));
            break;
          }
        break;
      }
      case 'N':
      case 'n':
      {
        if (LocaleCompare((char *) keyword,"name") == 0)
          {
            color_list->name=AcquireString(token);
            break;
          }
        break;
      }
      case 'O':
      case 'o':
      {
        if (LocaleCompare((char *) keyword,"opacity") == 0)
          {
            color_list->color.opacity=ScaleCharToQuantum(atol(token));
            break;
          }
        break;
      }
      case 'R':
      case 'r':
      {
        if (LocaleCompare((char *) keyword,"red") == 0)
          {
            color_list->color.red=ScaleCharToQuantum(atol(token));
            break;
          }
        break;
      }
      case 'S':
      case 's':
      {
        if (LocaleCompare((char *) keyword,"stealth") == 0)
          {
            color_list->stealth=LocaleCompare(token,"True") == 0;
            break;
          }
        break;
      }
      default:
        break;
    }
  }
  LiberateMemory((void **) &token);
  LiberateMemory((void **) &xml);
  if (color_list == (ColorInfo *) NULL)
    return(False);
  while (color_list->previous != (ColorInfo *) NULL)
    color_list=color_list->previous;
  return(True);
}
