/*
 * warning.cpp
 *
 * Copyright (c) 1999 Paul Campbell <paul@taniwha.com>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "warning.h"
#include "portable.h"
#include "version.h"

#include <klocale.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <kconfig.h>
#include <kurlrequester.h>
#include <kapplication.h>

#include <qlayout.h>
#include <qcheckbox.h>
#include <qspinbox.h>


WarningConfig::WarningConfig (int t, QWidget * parent, const char *name)
  : KCModule(parent, name),
    checkSuspend(0),
    checkStandby(0),
    checkHibernate(0)
{
  type = t;
  apm = laptop_portable::has_power_management();
  config =  new KConfig("kcmlaptoprc");

  my_load(0);

  if (!apm) {
    QVBoxLayout *top_layout = new QVBoxLayout( this, KDialog::marginHint(),
					       KDialog::spacingHint() );
	
    KActiveLabel* explain = laptop_portable::no_power_management_explanation(this);
    top_layout->addWidget(explain, 0);

    top_layout->addStretch(1);
  } else {
    QGridLayout *grid = new QGridLayout( this, 10, 2, /* rows x cols */
					 KDialog::marginHint(),
					 KDialog::spacingHint() );
    grid->setColStretch( 1, 1 );
    grid->setRowStretch( 8, 1 );

    // setup the trigger stuff:
    QLabel* low_label;
    if (type)
      low_label = new QLabel( i18n("Critical &trigger:"), this );
    else
      low_label = new QLabel( i18n("Low &trigger:"), this );
    grid->addWidget(low_label, 0, 0);

    if ( have_time ) {
      editLow = new QSpinBox( 1, 60*24 /*1 day*/, 1, this ); // min,max,step
      editLow->setSuffix( i18n("keep short, unit in spinbox", "min") );
    } else {
      editLow = new QSpinBox( 1, 100, 1, this ); // min,max.step
      editLow->setSuffix( i18n("keep short, unit in spinbox", "%") );
    }
    low_label->setBuddy( editLow );
    grid->addWidget( editLow, 0, 1, Qt::AlignLeft );
    connect(editLow, SIGNAL(valueChanged(int)), this, SLOT(configChanged()));

    // setup the Run Command stuff
    checkRunCommand = new QCheckBox(i18n("Run &command:"), this);
    grid->addWidget(checkRunCommand, 1, 0);

    editRunCommand = new KURLRequester( this );
    editRunCommand->setEnabled(false);
    connect( checkRunCommand, SIGNAL(toggled(bool)),
	     editRunCommand,  SLOT(setEnabled(bool)) );
    connect( checkRunCommand, SIGNAL(clicked()),
	     this, SLOT(configChanged()) );
    connect( editRunCommand, SIGNAL(textChanged(const QString&)),
	     this, SLOT(configChanged()) );
    grid->addWidget(editRunCommand, 1, 1);

    // setup the Play Sound stuff
    checkPlaySound = new QCheckBox(i18n("&Play sound:"), this);
    grid->addWidget(checkPlaySound, 2, 0);

    editPlaySound = new KURLRequester( this );
    editPlaySound->setEnabled(false);
    connect( checkPlaySound, SIGNAL(toggled(bool)),
	     editPlaySound,  SLOT(setEnabled(bool)) );
    connect( checkPlaySound, SIGNAL(clicked()),
	     this, SLOT(configChanged()) );
    connect( editPlaySound, SIGNAL(textChanged(const QString&)),
	     this, SLOT(configChanged()) );
    grid->addWidget(editPlaySound, 2, 1);

    // setup the System Sound stuff
    checkBeep = new QCheckBox(i18n("System &beep"), this);
    grid->addWidget(checkBeep, 3, 0);
    connect(checkBeep, SIGNAL(clicked()), this, SLOT(configChanged()));

    // setup the System Sound stuff
    checkNotify = new QCheckBox(i18n("&Notify"), this);
    grid->addWidget(checkNotify, 4, 0);
    connect(checkNotify, SIGNAL(clicked()), this, SLOT(configChanged()));

    int can_suspend = laptop_portable::has_suspend();
    int can_standby = laptop_portable::has_standby();
    int can_hibernate = laptop_portable::has_hibernation();

    int curRow = 5;
    if (can_suspend) {
      checkSuspend = new QCheckBox(i18n("&Suspend"), this);
      checkSuspend->setMinimumSize(checkSuspend->sizeHint());
      grid->addWidget(checkSuspend, curRow, 0);
      connect(checkSuspend, SIGNAL(clicked()), this, SLOT(configChanged()));
      ++curRow;
    }
    if (can_standby) {
      checkStandby = new QCheckBox(i18n("Standb&y"), this);
      checkStandby->setMinimumSize(checkStandby->sizeHint());
      grid->addWidget(checkStandby, curRow, 0);
      connect(checkStandby, SIGNAL(clicked()), this, SLOT(configChanged()));
      ++curRow;
    }
    if (can_hibernate) {
      checkHibernate = new QCheckBox(i18n("H&ibernate"), this);
      checkHibernate->setMinimumSize(checkHibernate->sizeHint());
      grid->addWidget(checkHibernate, curRow, 0);
      connect(checkHibernate, SIGNAL(clicked()), this, SLOT(configChanged()));
      ++curRow;
    }

    QLabel* explain;
    if (type) {
      explain = new QLabel(i18n("This panel controls how and when you receive warnings\nthat your battery power is going to run out VERY VERY soon."), this);
    } else {
      explain = new QLabel(i18n("This panel controls how and when you receive warnings\nthat your battery power is about to run out"), this);
    }
    grid->addMultiCellWidget(explain, curRow, curRow, 0, 1,
			     Qt::AlignLeft|Qt::AlignTop);
    ++curRow;

    if (!can_suspend || !can_standby || !can_hibernate) {
      // display help text:
      QLabel* note = laptop_portable::how_to_do_suspend_resume(this);
      grid->addMultiCellWidget(note, curRow, 8, 0, 1,
			       Qt::AlignLeft|Qt::AlignTop);
    }

    grid->addWidget(new QLabel( i18n("Version: %1").arg(LAPTOP_VERSION), this),
		    9, 1, Qt::AlignRight);

  }
  my_load(1);
}

WarningConfig::~WarningConfig()
{
  delete config;
}

void WarningConfig::save()
{
    if (apm) {
    	runcommand = checkRunCommand->isChecked();
    	playsound = checkPlaySound->isChecked();
    	beep = checkBeep->isChecked();
    	notify = checkNotify->isChecked();
    	do_suspend = (checkSuspend? checkSuspend->isChecked() : false);
    	do_standby = (checkStandby? checkStandby->isChecked() : false);
    	do_hibernate = (checkHibernate? checkHibernate->isChecked() : false);
    	runcommand_val = editRunCommand->url();
    	low_val = editLow->value();
    	sound_val = editPlaySound->url();
    }
    config->setGroup((type?"BatteryCritical":"BatteryLow"));
       
    config->writeEntry("LowVal", low_val);
    config->writeEntry("RunCommand", runcommand);
    config->writeEntry("PlaySound", playsound);
    config->writeEntry("SystemBeep", beep);
    config->writeEntry("Notify", notify);
    config->writeEntry("Suspend", do_suspend);
    config->writeEntry("Standby", do_standby);
    config->writeEntry("Hibernate", do_hibernate);
    config->writeEntry("RunCommandPath", runcommand_val);
    config->writeEntry("PlaySoundPath", sound_val);
    config->sync();
    kapp->startServiceByDesktopName("klaptopdaemon");
}

void WarningConfig::load()
{
	if (config == NULL)
		return;
	my_load(0);
	my_load(1);
}

void WarningConfig::my_load(int x)
{

	// open the config file
	if (!x) {
		config->setGroup((type?"BatteryCritical":"BatteryLow"));

		low_val = config->readNumEntry("LowVal", (type? 5:15));
		runcommand = config->readBoolEntry("RunCommand", false);
		playsound = config->readBoolEntry("PlaySound", false);
		beep = config->readBoolEntry("SystemBeep", true);
		notify = config->readBoolEntry("Notify", (type && checkSuspend ? false : true));
		do_suspend = config->readBoolEntry("Suspend", (type && checkSuspend ? true :false));
		do_standby = config->readBoolEntry("Standby", false);
		do_hibernate = config->readBoolEntry("Hibernate", false);
		runcommand_val = config->readEntry("RunCommandPath");
		sound_val = config->readEntry("PlaySoundPath");
		have_time = config->readNumEntry("HaveTime", 2);
		if (laptop_portable::has_power_management())
			have_time = laptop_portable::has_battery_time();
	
	} else
	if (apm) {
		checkRunCommand->setChecked(runcommand);
		checkPlaySound->setChecked(playsound);
		checkBeep->setChecked(beep);
		checkNotify->setChecked(notify);
		if (checkHibernate)
			checkHibernate->setChecked(do_hibernate);
		if (checkStandby)
			checkStandby->setChecked(do_standby);
		if (checkSuspend)
			checkSuspend->setChecked(do_suspend);
		editRunCommand->setURL(runcommand_val);
		editLow->setValue(low_val);
		editPlaySound->setURL(sound_val);
	}
	changed(false);

}

void WarningConfig::defaults()
{
	// open the config file
	runcommand = false;
	playsound = false;
	beep = true;
	notify = !(type && checkSuspend);
	do_standby = false;
	do_hibernate = false;
	do_suspend = (type && checkSuspend);
	runcommand_val = "";
	low_val = (type?5:15);
	sound_val = "";
	

	if (apm) {
		checkRunCommand->setChecked(runcommand);
		checkPlaySound->setChecked(playsound);
		checkBeep->setChecked(beep);
		checkNotify->setChecked(notify);
		if (checkHibernate)
			checkHibernate->setChecked(do_hibernate);
		if (checkStandby)
			checkStandby->setChecked(do_standby);
		if (checkSuspend)
			checkSuspend->setChecked(do_suspend);
		editRunCommand->setURL(runcommand_val);
		editLow->setValue(low_val);
		editPlaySound->setURL(sound_val);
	}
	changed(false);
}


void WarningConfig::configChanged()
{
  emit changed(true);
}


#if 0
void WarningConfig::enableRunCommand(bool enable)
{
	editRunCommand->setEnabled(enable);
	buttonBrowseRunCommand->setEnabled(enable);
	configChanged();
}

void WarningConfig::enablePlaySound(bool enable)
{
	editPlaySound->setEnabled(enable);
	buttonBrowsePlaySound->setEnabled(enable);
	configChanged();
}

void WarningConfig::browseRunCommand()
{
  KURL url = KFileDialog::getOpenURL(QString::null, QString::null, this  );
  
  if( url.isEmpty() )
    return;
  
  if( !url.isLocalFile() )
  {
    KMessageBox::sorry( 0L, i18n( "Only local files are currently supported." ) );
    return;
  }
  
  editRunCommand->setText( url.path() );
  configChanged();
}

void WarningConfig::browsePlaySound()
{
  KURL url = KFileDialog::getOpenURL(QString::null, QString::null, this  );
  
  if( url.isEmpty() )
    return;
  
  if( !url.isLocalFile() )
  {
    KMessageBox::sorry( 0L, i18n( "Only local files are currently supported." ) );
    return;
  }
  
  editPlaySound->setText( url.path() );
  configChanged();
}
#endif

QString WarningConfig::quickHelp() const
{
  return i18n("<h1>Low battery Warning</h1>This module allows you to "
	"set an alarm in case your battery's capacity is exceeded.");
}

#include "warning.moc"
