/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 2002 by Stanislav Visnovsky
  
  based on code of Matthias Kiefer <matthias.kiefer@gmx.de> and 
   Andrea Rizzi <rizzi@kde.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */
#ifndef _ARG_EXTRACTOR_H_
#define _ARG_EXTRACTOR_H_

#include <qptrlist.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qvaluelist.h>
#include <qregexp.h>

/**
 * holds a description of what an argument is 
 */
class ArgListEntry
{
public:
  ArgListEntry();
  ArgListEntry(QString t,bool isReg);
  
  bool operator==(const ArgListEntry& t2);
  QString asString() const { return arg; }
  QRegExp asRegExp() const { return reg; }
  bool isRegExp() const { return isReg; }

  /**
   * @return the first match of this argument in st,
   * QString::null, if it does not match
   */
  QString firstMatch(const QString st) const;

private:
  QString arg;
  QRegExp reg;
  bool isReg;
};
 

struct ArgInfo{
    uint index;
    QString arg;
};


/**
 * class to extract arguments from a string
 * @author Stanislav Visnovsky <visnovsky@kde.org>
 * @author Andrea Rizzi <rizzi@kde.org>
 */
class ArgExtractor 
{

public:
  /**
   * Create an argument extractor for "string"
   */ 
  ArgExtractor(QString string=QString::null);

  /**
   * Set the string of this argument extractor
   */
  void setString(QString string);

  /**
   * @return the number of arguments found
   */
  uint countArgs();

  /**
   * @return the first argument, and set the internal cursor to
   * the beginning
   */
  QString firstArg();

  /**
   * @return the next argument and move cursor forward
   */
  QString nextArg();

  /**
   * @return the n-th argument. It also moves the cursor.
   */
  QString arg(uint argnumber);

  /**
   * @return the previous argument and move cursor forward
   */
  QString prevArg();

  /**
   * @return the last argument and move the cursor to the end
   */
  QString lastArg();
  
  /**
   * @return a list of all arguments
   */
  QStringList args();

  /**
   * @return the string without arguments
   *
   * @param keepPos if false, the arguments are just removed and so the
   * position of the other words in the string will change. If true,
   * the arguments are replaced with ' ' and therefore the position of the
   * words will not change
   */
  QString plainString(bool keepPos=false);

  /**
   * @return the string, where arguments are replaced with the given string
   */
  QString argsReplaced(const QString& replace);


  //Functions that allow user to define his own tags.
  
  /**
   * Add a tag to the list of tag identifier.
   * It may be a tag (e.g. <HTML> or <BODY>)
   * or a regexp (if the isRegExp flag is true)
   * for arguments. 
   */
   void addArgIdentifier(QString arg,bool isRegExp);
   

  /**
   * Delete from the arg list the arg identified by
   * arg and isRegExp specified.
   */
  void deleteArgIdentifier(QString arg,bool isRegExp);

  /**
   * Return a list of actual arg regular expressions.
   */
  static QStringList *argExpressions();

  /**
   * Set a new list of tag regular expressions. It also
   * deletes the old tags.
   * @param list	a list of regular expressions
   */
  static void setArgExpressions(QStringList* list);
  
protected:
  /**
   * @return the static list of args
   */
  static QValueList<ArgListEntry> *argList();
  void processString();
  QPtrList<ArgInfo> _args;
  QString _string;
  
private: 
  static QValueList<ArgListEntry> *_argList; 
};




#endif
