/*
**************************************************************************
                                 description
                             --------------------
    copyright            : (C) 2002 by Andreas Zehender
    email                : zehender@kde.org
**************************************************************************

**************************************************************************
*                                                                        *
*  This program is free software; you can redistribute it and/or modify  *
*  it under the terms of the GNU General Public License as published by  *
*  the Free Software Foundation; either version 2 of the License, or     *
*  (at your option) any later version.                                   *
*                                                                        *
**************************************************************************/


#include "pmblob.h"

#include "pmoutputdevice.h"
#include "pmxmlhelper.h"
#include "pmblobedit.h"
#include "pmmemento.h"

#include <kdebug.h>
#include "pmglobals.h"

#include <klocale.h>

const double c_defaultThreshold = 0.5;
bool c_defaultSturm = false;
bool c_defaultHierarchy = false;

PMBlob::PMBlob( )
      : Base( )
{
   m_threshold = c_defaultThreshold;
   m_sturm = c_defaultSturm;
   m_hierarchy = c_defaultHierarchy;
}

PMBlob::~PMBlob( )
{
}

QString PMBlob::description( ) const
{
   return i18n( "blob" );
}

void PMBlob::serialize( PMOutputDevice& dev ) const
{
   dev.objectBegin( "blob" );

   serializeName( dev );
   dev.writeLine( QString( "threshold %1" ).arg( m_threshold ) );
   
   Base::serialize( dev );

   if( m_sturm )
      dev.writeLine( "sturm" );
   if( m_hierarchy )
      dev.writeLine( "hierarchy" );
   
   dev.objectEnd( );
}

void PMBlob::serialize( QDomElement& e, QDomDocument& doc ) const
{
   e.setAttribute( "threshold", m_threshold );
   e.setAttribute( "sturm", m_sturm );
   e.setAttribute( "hierarchy", m_hierarchy );
   Base::serialize( e, doc );
}

void PMBlob::readAttributes( const PMXMLHelper& h )
{
   m_threshold = h.doubleAttribute( "threshold", c_defaultThreshold );
   m_sturm = h.boolAttribute( "sturm", c_defaultSturm );
   m_hierarchy = h.boolAttribute( "hierarchy", c_defaultHierarchy );
   Base::readAttributes( h );
}

bool PMBlob::isA( PMObjectType t ) const
{
   if( t == PMTBlob )
      return true;
   return Base::isA( t );
}

bool PMBlob::canInsert( PMObjectType t, const PMObject* after,
                        const PMObjectList* objectsBetween ) const
{
   bool afterInsertPoint = false;
   bool componentsBehind = false, omBefore = false;
   PMObject* o;
   
   if( !after )
      afterInsertPoint = true;
   for( o = firstChild( ); o; o = o->nextSibling( ) )
   {
      countChild( o->type( ), omBefore, componentsBehind,
                  afterInsertPoint );
      if( o == after )
         afterInsertPoint = true;
   }

   if( objectsBetween )
   {
      PMObjectListIterator it( *objectsBetween );
      for( ; it.current( ); ++it )
         countChild( it.current( )->type( ), omBefore, componentsBehind,
                     false );
   }

   return canInsert( t, omBefore, componentsBehind );
}

int PMBlob::canInsert( const QValueList<PMObjectType>& list,
                       const PMObject* after ) const
{
   bool afterInsertPoint = false;
   bool componentsBehind = false, omBefore = false;
   QValueList<PMObjectType>::ConstIterator it;
   PMObject* o;
   PMObjectType t;
   int numInserts = 0;

   if( !after )
      afterInsertPoint = true;
   for( o = firstChild( ); o; o = o->nextSibling( ) )
   {
      countChild( o->type( ), omBefore, componentsBehind,
                  afterInsertPoint );
      if( o == after )
         afterInsertPoint = true;
   }
   for( it = list.begin( ); it != list.end( ); ++it )
   {
      t = *it;
      if( canInsert( t, omBefore, componentsBehind ) )
         numInserts++;
      countChild( t, omBefore, componentsBehind, false );
   }
   return numInserts;
}

int PMBlob::canInsert( const PMObjectList& list,
                      const PMObject* after ) const
{
   bool afterInsertPoint = false;
   bool componentsBehind = false, omBefore = false;
   PMObjectListIterator it( list );
   PMObject* o;
   PMObjectType t;
   int numInserts = 0;

   if( !after )
      afterInsertPoint = true;
   for( o = firstChild( ); o; o = o->nextSibling( ) )
   {
      countChild( o->type( ), omBefore, componentsBehind,
                  afterInsertPoint );
      if( o == after )
         afterInsertPoint = true;
   }
   for( ; it.current( ); ++it )
   {
      t = it.current( )->type( );
      if( canInsert( t, omBefore, componentsBehind ) )
         numInserts++;
      countChild( t, omBefore, componentsBehind, false );
   }
   return numInserts;
}

void PMBlob::countChild( PMObjectType t, bool& omBefore,
                         bool& componentsBehind, bool afterInsertPoint ) const
{
   switch( t )
   {
      case PMTComment:
      case PMTRaw:
         break;
      case PMTTranslate:
      case PMTRotate:
      case PMTScale:
      case PMTMatrix:
      case PMTTexture:
      case PMTPigment:
      case PMTNormal:
      case PMTFinish:
      case PMTInterior:
      case PMTMaterial:
         if( !afterInsertPoint )
            omBefore = true;
         break;
      case PMTBlobCylinder:
      case PMTBlobSphere:
         if( afterInsertPoint )
            componentsBehind = true;
         break;
      default:
         break;
   }
}

bool PMBlob::canInsert( PMObjectType t, bool omBefore,
                        bool componentsBehind ) const
{
   switch( t )
   {
      case PMTComment:
      case PMTRaw:
         return true;
      case PMTTranslate:
      case PMTRotate:
      case PMTScale:
      case PMTMatrix:
      case PMTTexture:
      case PMTPigment:
      case PMTNormal:
      case PMTFinish:
      case PMTInterior:
      case PMTMaterial:
         return !componentsBehind;
      case PMTBlobCylinder:
      case PMTBlobSphere:
         return !omBefore;
      default:
         break;
   }
   return false;
}

void PMBlob::setThreshold( double t )
{
   if( t <= 0.0 )
   {
      kdError( PMArea ) << "Threshold is not positive in PMBlob::setThreshold\n";
      t = 1.0;
   }
   
   if( t != m_threshold )
   {
      if( m_pMemento )
         m_pMemento->addData( PMTBlob, PMThresholdID, m_threshold );
      m_threshold = t;
   }
}

void PMBlob::setSturm( bool s )
{
   if( s != m_sturm )
   {
      if( m_pMemento )
         m_pMemento->addData( PMTBlob, PMSturmID, m_sturm );
      m_sturm = s;
   }
}

void PMBlob::setHierarchy( bool h )
{
   if( h != m_hierarchy )
   {
      if( m_pMemento )
         m_pMemento->addData( PMTBlob, PMHierarchyID, m_hierarchy );
      m_hierarchy = h;
   }
}

PMDialogEditBase* PMBlob::editWidget( QWidget* parent ) const
{
   return new PMBlobEdit( parent );
}

void PMBlob::restoreMemento( PMMemento* s )
{
   PMMementoDataIterator it( s );
   PMMementoData* data;

   for( ; it.current( ); ++it )
   {
      data = it.current( );
      if( data->objectType( ) == PMTBlob )
      {
         switch( data->valueID( ) )
         {
            case PMThresholdID:
               setThreshold( data->doubleData( ) );
               break;
            case PMSturmID:
               setSturm( data->boolData( ) );
               break;
            case PMHierarchyID:
               setHierarchy( data->boolData( ) );
               break;
            default:
               kdError( PMArea ) << "Wrong ID in PMBlob::restoreMemento\n";
               break;
         }
      }
   }
   Base::restoreMemento( s );
}
