/*
 * METALBASE 5.1
 *
 * Released January 1st, 1993 by Huan-Ti [ t-richj@microsoft.com ]
 *
 */

#include <mbase.h>
#include <parse.h>


/*
 * MACROS ---------------------------------------------------------------------
 *
 */

#define isEOL(x)   (x=='\n' || x=='\r')
#define isQuote(x) (x=='\"')
#define isWhite(x) (x==' ' || x=='\t')

#define isChar(x)  (isalnum(x) || (x=='.') || (x=='_'))

#define isToken(x) ( x=='(' || x==')' || x==';' || x=='#'  || \
                     x=='[' || x==']' || x==':' || x=='\'' || \
                     x=='{' || x=='}' || x==',')

#define isDupEOL(p) ( (*p == '\n' && *(p+1) == '\r') || \
                      (*p == '\r' && *(p+1) == '\n') )


/*
 * PROTOTYPES -----------------------------------------------------------------
 *
 */

   static bool    fpFillBuffer  XARGS( (FParse *) );
   static charptr fpCleanup     XARGS( (FParse *, charptr, charptr) );


/*
 * FILE PARSING ROUTINES ------------------------------------------------------
 *
 */

void
fpClose (fp)
FParse  *fp;
{
   if (fp != NULL)
      {
      if (fp->fh > 0)
         {
         close (fp->fh);
         }
      free (fp);
      }
}

FParse *
fpInitStr (str, fLower)
charptr    str;
bool            fLower;
{
   FParse *fp;

   if ((fp = New (FParse)) == NULL)
      return NULL;

   fp->fh       =  0;
   fp->str      =  str;
   fp->fLower   = fLower;
   fp->lastpos  = -1L;
   fp->curpos   =  0L;
   fp->fEOF     =  FALSE;
   fp->fQuoted  =  FALSE;
   fp->buf[0]   =  0;
   fp->nBuf     =  0;
   fp->nLines   =  1;

   return fp;
}

FParse *
fpInit (name, fLower)
charptr name;
bool          fLower;
{
   FParse *fp;

   if ((fp = New (FParse)) == NULL)
      return NULL;

   if ((fp->fh = openx (name, READMODE)) <= 0)
      {
      free (fp);
      return NULL;
      }

   fp->str      =  NULL;
   fp->fLower   = fLower;
   fp->lastpos  = -1L;
   fp->curpos   =  0L;
   fp->fEOF     =  FALSE;
   fp->fQuoted  =  FALSE;
   fp->buf[0]   =  0;
   fp->nBuf     =  0;
   fp->nLines   =  1;

   return fp;
}

void
fpSetPos (fp, pos, lineno)
FParse   *fp;
long          pos;
int                lineno;
{
   fp->lastpos  = -1L;
   fp->curpos   =  pos;
   fp->fEOF     =  FALSE;
   fp->fQuoted  =  FALSE;
   fp->buf[0]   =  0;
   fp->nBuf     =  0;
   fp->nLines   = max (lineno, 1);

   if (fp->fh > 0)  lseek (fp->fh, pos, 0);
}

bool
fpBackup (fp)
FParse   *fp;
{
   if (! fp || (fp->lastpos == -1L))
      return FALSE;

   fp->curpos   =  fp->lastpos;
   fp->lastpos  = -1L;
   fp->buf[0]   =  0;
   fp->nBuf     =  0;
   fp->fEOF     =  FALSE;
   fp->nLines   =  fp->nLast;

   if (fp->fh > 0)  lseek (fp->fh, fp->curpos, 0);

   return TRUE;
}

charptr
fpWord (fp)
FParse *fp;
{
   charptr  pLine, pWord;

   if (! fpFillBuffer (fp))
      return NULL;

   *( pWord = fp->word ) = 0;
   for (pLine = fp->buf; *pLine; pLine++)
      {
      *pWord = 0;

      if (! fp->fQuoted)
         {
         if (! strncmp (fp->word, "\/\*", 2))  break;
         if (! strncmp (fp->word, "\*\/", 2))  break;
         if (! strncmp (fp->word, "\/\/", 2))  break;

         if ( (pWord - fp->word) >= 2 )
            {
            if ( (isChar(*(pWord-2)) && !isChar(*(pWord-1))) ||
                 (isChar(*(pWord-1)) && !isChar(*(pWord-2))) )
               {
               pLine--;
               pWord--;
               break;
               }
            }
         }

      if (isQuote (*pLine))
         {
         if (fp->fQuoted)
            {
            pLine++;
            break;
            }
         if (pWord != fp->word)
            break;
         fp->fQuoted = TRUE;
         continue;
         }

      if (isEOL (*pLine))
         {
         if (pWord != fp->word)  /* A newline terminates a quoted string; */
            break;               /* there is no error condition for this. */
         fp->nLines++;
         if (isDupEOL (pLine))
            pLine++;
         continue;
         }

      if (! fp->fQuoted)
         {
         if (isWhite (*pLine))
            {
            if (pWord != fp->word)
               break;
            continue;
            }

         if (isToken (*pLine))
            {
            if (pWord != fp->word)
               break;
            *pWord = tolower (*pLine);
            pWord++;
            pLine++;
            break;
            }
 
         if (fp->fLower)
            {
            *pWord = tolower (*pLine);
            pWord++;
            continue;
            }
         }

      *pWord = *pLine;
      pWord++;
      }

   return fpCleanup (fp, pLine, pWord);
}

charptr
fpLine (fp)
FParse *fp;
{
   charptr  pLine, pWord;
   bool     fQuoted = FALSE;

   if (! fpFillBuffer (fp))
      return NULL;

   pWord = fp->word;
   for (pLine = fp->buf; *pLine; pLine++, pWord++)
      {
      if (isEOL (*pLine))
         {
         fp->nLines++;

         if (isDupEOL (pLine))
            pLine++;
         pLine++;

         break;
         }
      if (isQuote (*pLine))
         {
         fQuoted = (fQuoted) ? FALSE : TRUE;
         }

      if (fp->fLower && !fQuoted)
         {
         *pWord = tolower(*pLine);
         }
      else
         {
         *pWord = *pLine;
         }
      }

   return fpCleanup (fp, pLine, pWord);
}


/*
 * SERVICE ROUTINES -----------------------------------------------------------
 *
 */

static charptr
fpCleanup (fp, pLine, pWord)
FParse    *fp;
charptr        pLine, pWord;
{
   fp->curpos += (pLine - fp->buf);
   fp->nBuf   -= (pLine - fp->buf);

   strzcpy (fp->buf, pLine, strlen(pLine));  /* Ensure strcpy() goes -> */

   *pWord = 0;

   return fp->word;
}

static bool
fpFillBuffer (fp)
FParse       *fp;
{
   int  nRead;

   if (! fp)
      return FALSE;

   fp->fLastEOF = FALSE;
   fp->nLast    = fp->nLines;
   fp->lastpos  = fp->curpos;
   fp->fQuoted  = FALSE;

   if (fp->nBuf < MAX_LINE)
      {
      if (fp->fh > 0)
         {
         nRead = readx (fp->fh, &fp->buf [fp->nBuf], MAX_BUF - fp->nBuf);

         fp->nBuf += nRead;
         fp->buf [fp->nBuf] = 0;
         }
      else
         {
         strncpy (fp->buf, &fp->str[fp->curpos], MAX_BUF-fp->nBuf);
         fp->buf[MAX_BUF] = 0;

         fp->nBuf = nRead = strlen (fp->buf);
         }

      if (nRead == 0)
         {
         fp->fEOF = TRUE;
         }
      }

   return TRUE;
}

