------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             S E M _ D I S T                              --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.19 $                              --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  This package contains utility routines used for validation of the
--  use of the categorization pragmas relevant to the distribution annexe.

with Types; use Types;

package Sem_Dist is

   procedure CW_Remote_Extension_Add_Receiver (N : Node_Id);
   --  In case N, the (library) package body node or its corresponding spec
   --  has classwide type remote extensions (check comments in following
   --  function Is_Class_Wide_Type_Remote_Extension for definition) then
   --  generates receiver subprogram for this type (extension). Receiver
   --  is used to handle incoming remote dispatching calls.

   function Enclosing_Lib_Unit_Entity return Entity_Id;
   --  Returns the entity of enclosing N_Compilation_Unit Node which is the
   --  root of the current scope (which must not be Standard_Standard, and
   --  the caller is responsible for ensuring this condition).

   function Enclosing_Lib_Unit_Node (N : Node_Id) return Node_Id;
   --  Returns the enclosing N_Compilation_Unit Node that is the root
   --  of a subtree containing N.

   function Inside_Preelaborated_Unit return Boolean;
   --  Determines if the current scope is within a preelaborated compilation
   --  unit, that is one to which one of the pragmas Preelaborate, Pure,
   --  Shared_Passive, Remote_Types, or inside a unit other than a package
   --  body with pragma Remote_Call_Interface.

   function Inside_Pure_Unit return Boolean;
   --  Determines if the current scope is within pure compilation unit,
   --  that is, one to which the pragmas Pure is applied.

   function Inside_Remote_Call_Interface_Unit return Boolean;
   --  Determines if the current scope is within a Remote Call Interface
   --  compilation unit.

   function Inside_Remote_Types_Unit return Boolean;
   --  Determines if the current scope is within a Remote Types compilation
   --  unit.

   function Inside_Shared_Passive_Unit return Boolean;
   --  Determines if the current scope is within a Shared Passive compilation
   --  unit.

   function Inside_Subprogram_Task_Protected_Unit return Boolean;
   --  Determines if the current scope is within a subprogram, task
   --  or protected unit. Used to validate if the library unit is Pure
   --  (RM 10.2.1(16)).

   function Inside_Subprogram_Unit return Boolean;
   --  Determines if the current scope is within a subprogram compilation
   --  unit (inside a subprogram declaration, subprogram body, or generic
   --  subprogram declaration). The test is for appearing anywhere within
   --  such a construct (that is it does not need to be directly within).

   function Is_Class_Wide_Type_Remote_Extension (N : Node_Id) return Boolean;
   --  Return True if N is an extension of a root abstract-tagged-limited-
   --  private-type and the root abstract-type is the designated type of
   --  an RCI remote access-to-limited-class-wide-type.

   function Is_Remote_Access_To_Class_Wide_Type (E : Entity_Id) return Boolean;
   --  Return True if E is a remote access-to-class-wide-limited_private type

   function Is_Remote_Access_To_Subprogram_Type
     (E    : Entity_Id)
      return Boolean;
   --  Return True if E is a remote access to subprogram type.

   function Should_Declare_Partition_ID (L : List_Id) return Boolean;
   --  Given a non-empty list L of the Visible_Declarations of the package
   --  specification of a libbrary, unit, determines whether the package is
   --  System.RPC or is categorized as pure. This is used to limit the
   --  library package units that declare Predefined_Partition_Id to those
   --  that really need to.

   procedure Set_Categorization_From_Following_Pragmas (N : Node_Id);
   --  Since validation of categorization dependency is done during analyze
   --  so categorization flags from following pragmas should be set before
   --  validation begin. N is the N_Compilation_Unit node.

   procedure Validate_Access_Type_Declaration (T : Entity_Id; N : Node_Id);
   --  Validate all constraints against declaration of access types in
   --  categorized library units. Usually this is a violation in Pure unit,
   --  Shared_Passive unit. N is the declaration node.

   procedure Validate_Categorization_Dependency (N : Node_Id; E : Entity_Id);
   --  There are restrictions on lib unit that semantically depends on other
   --  units (RM E.2(5), 10.2.1(11). This procedure checks the restrictions
   --  on categorizations. N is the current unit node, and E is the current
   --  library unit entity.

   procedure Validate_Object_Declaration
     (N   : Node_Id;
      Id  : Entity_Id;
      E   : Node_Id;
      Odf : Node_Id;
      T   : Entity_Id);
   --  Validate all the constraints in a preelaborable lib unit against
   --  an object declaration, including the creation of task object, etc.

   procedure Validate_RCI_Access_Object_Type_Declaration (T : Entity_Id);
   --  Check validity of declaration if RCI unit. It should not contain
   --  the declaration of an access-to-object type unless it is a
   --  general access type that designates a class-wide limited
   --  private type. There are also constraints about the primitive
   --  subprograms of the class-wide type. RM E.2.3(14).

   procedure Validate_RCI_Limited_Type_Declaration (N : Node_Id);
   --  Make sure that the visible part of an RCI unit does not contain
   --  declaration of limited type (RM E.2.3(10)).

   procedure Validate_RCI_Nested_Generic_Declaration (N : Node_Id);
   --  Make sure that the visible part of an RCI unit does not contain nested
   --  generic declarations (RM E.2.3(11)). N is the compilation unit node.

   procedure Validate_RCI_Subprogram_Declaration (N : Node_Id);
   --  Check for RCI unit subprogram declarations with respect to
   --  in-lined subprogram and subprogram with access parameter or
   --  limited type parameter without Read and Write.

   procedure Validate_Remote_Access_To_Class_Wide_Type (N : Node_Id);
   --  Checks that Storage_Pool and Storage_Size attribute references are
   --  not applied to remote access-to-class-wide types. And the expected
   --  type for an allocator shall not be a remote access-to-class-wide
   --  type. And a remote access-to-class-wide type shall not be an actual
   --  parameter for a generic formal access type. RM E.2.3(22).

   procedure Validate_Remote_Type_Type_Conversion (N : Node_Id);
   --  Check for remote-type type conversion constraints. First, a value of
   --  a remote access-to-subprogram type can be converted only to another
   --  type conformant remote access-to-subprogram type. Secondly, a value
   --  of a remote access-to-class-wide type can be converted only to another
   --  remote access-to-class-wide type (RM E.2.3(17,20)).

   --  Following name is too long ???
   procedure Validate_Shared_Passive_Access_Object_Type_Declaration
     (T : Entity_Id);
   --  Check validity of declaration if shared passive unit. It should not
   --  contain the declaration of an access-to-object type whose designated
   --  type is a class-wide type ,task type or protected type. E.2.1(7).
   --  What is T ???

end Sem_Dist;
