------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             G N A T 1 D R V                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.39 $                             --
--                                                                          --
--        Copyright (c) 1992,1993,1994,1995 NYU, All Rights Reserved        --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Comperr;
with Csets;    use Csets;
with Back_End;
with Errout;   use Errout;
with Features;
with Frontend;
with Gnatvsn;  use Gnatvsn;
with Lib;      use Lib;
with Lib.Writ; use Lib.Writ;
with Namet;    use Namet;
with Opt;      use Opt;
with Osint;    use Osint;
with Output;   use Output;
with Par;
with Sinfo;    use Sinfo;
with Snames;
with Sprint;   use Sprint;
with Stringt;
with System.Assertions;
with Tree_Gen;
with Treepr;   use Treepr;
with Types;    use Types;
with Uintp;
with Uname;    use Uname;
with Urealp;
with Usage;

procedure Gnat1drv is
   Main_Unit_Node : Node_Id;
   --  Compilation unit node for main unit

   Main_Kind : Node_Kind;
   --  Kind of main compilation unit node.

begin
   --  This inner block is set up to catch assertion errors and constraint
   --  errors. Since the code for handling these errors can cause another
   --  exception to be raised (namely Unrecoverable_Error), we need two
   --  nested blocks, so that the outer one handles unrecoverable error.

   begin
      Osint.Initialize (Compiler);
      Csets.Initialize;
      Uintp.Initialize;
      Urealp.Initialize;
      Errout.Initialize;
      Namet.Initialize;
      Snames.Initialize;
      Stringt.Initialize;
      Features.Initialize;

      if Verbose_Mode or Full_List then
         Write_Eol;
         Write_Str ("NYU GNAT Compiler Version ");
         Write_Str (Gnat_Version_String);
         Write_Str (" (C) Copyright NYU, 1992,1993,1994,1995");
         Write_Eol;
      end if;

      Frontend;

      if Errors_Detected /= 0 then
         Errout.Finalize;
         Exit_Program (E_Errors);
      end if;

      if Operating_Mode /= Generate_Code then
         Errout.Finalize;
         Tree_Gen;
         Namet.Finalize;
         Features.Finalize;
         return;
      end if;

      --  Check for unit that generates no code, and if so, generate
      --  warning message and suppress expander and code generation.

      Main_Unit_Node := Cunit (Main_Unit);
      Main_Kind := Nkind (Unit (Main_Unit_Node));

      --  Generate code for subprogram bodies only if they have
      --  a corresponding non-generic subprogram declaration. Note
      --  that the check for No (Library_Unit) here is a defensive
      --  check that should not be necessary, since the Library_Unit
      --  field should always be set properly.

      if Main_Kind = N_Subprogram_Body
        and then (No (Library_Unit (Main_Unit_Node))
                   or else Nkind (Unit (Library_Unit (Main_Unit_Node))) /=
                                          N_Generic_Subprogram_Declaration)
      then
         null;

      --  Generate code for package bodies only if they have
      --  a corresponding non-generic package declaration

      elsif Main_Kind = N_Package_Body
        and then (No (Library_Unit (Main_Unit_Node))
           or else Nkind (Unit (Library_Unit (Main_Unit_Node))) /=
                      N_Generic_Package_Declaration)
      then
         null;

      --  Generate code for package declarations that do not
      --  require a corresponding body

      elsif Main_Kind = N_Package_Declaration
        and then not Body_Required (Main_Unit_Node)
      then
         null;

      --  Compilation units that are renamings do not require
      --  bodies either.

      elsif Main_Kind = N_Package_Renaming_Declaration
        or else Main_Kind = N_Subprogram_Renaming_Declaration
      then
         null;

      --  In all other cases (specs which have bodies, and generics)
      --  we cannot generate code and we generate a warning message.
      --  Note that generic instantiations are gone at this stage
      --  since they have been replaced by their instances.

      --  Also note that we exit with an error, to prevent the backend
      --  from generating an object module, which is wrong, and more
      --  significantly, might cause a legitimate object module for the
      --  corresponding body to be clobbered.

      else
         Write_Str ("No code generated for ");
         Write_Unit_Name (Unit_Name (Main_Unit));
         Write_Str (" in file ");
         Write_Name (Unit_File_Name (Main_Unit));
         Write_Eol;
         Errout.Finalize;
         Tree_Gen;
         Namet.Finalize;

         --  In case a generic unit is being compiled exit with a Success exit
         --  code in preparation of compiling generic units. This is code
         --  which will disappear when we *do* compile generic units. ???

         if Main_Kind = N_Subprogram_Body
           and then Present (Library_Unit (Main_Unit_Node))
           and then Nkind (Unit (Library_Unit (Main_Unit_Node))) =
                                           N_Generic_Subprogram_Declaration
         then
            Exit_Program (E_Success);

         elsif Main_Kind = N_Package_Body
           and then Present (Library_Unit (Main_Unit_Node))
           and then Nkind (Unit (Library_Unit (Main_Unit_Node))) =
                                           N_Generic_Package_Declaration
         then
            Exit_Program (E_Success);

         else
            Exit_Program (E_Errors);
         end if;
      end if;

      Set_Generate_Code (Main_Unit);

      --  If we have a corresponding spec, then we need object
      --  code for the spec unit as well

      if Nkind (Unit (Main_Unit_Node)) in N_Unit_Body
        and then not Acts_As_Spec (Main_Unit_Node)
      then
         Set_Generate_Code
           (Get_Cunit_Unit_Number (Library_Unit (Main_Unit_Node)));
      end if;

      --  Generate back end tables and library information

      Back_End;
      Errout.Finalize;
      Tree_Gen;
      Features.Finalize;

      --  Only write the library if the backend did not generate any error
      --  messages. Otherwise signal errors to the driver program so that
      --  there will be no attempt to generate an object file.

      if Errors_Detected /= 0 then
         Exit_Program (E_Errors);
      end if;

      Lib.Writ.Write_Library_Info;
      Namet.Finalize;

   exception
      --  Handle fatal internal compiler errors

      when System.Assertions.Assert_Failure =>
         Comperr.Compiler_Abort ("Assert_Failure");

      when Constraint_Error =>
         Comperr.Compiler_Abort ("Constraint_Error");

      when Program_Error =>
         Comperr.Compiler_Abort ("Program_Error");

      when Storage_Error =>
         Comperr.Compiler_Abort ("Storage_Error");
   end;

--  The outer exception handles an unrecoverable error

exception
   when Unrecoverable_Error =>
      Errout.Finalize;
      Set_Standard_Error;
      Write_Str ("compilation abandoned");
      Write_Eol;
      Set_Standard_Output;

      Tree_Dump;
      Source_Dump;
      Exit_Program (E_Errors);

end Gnat1drv;
