------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             E X P _ D I S P                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.16 $                              --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Einfo;    use Einfo;
with Elists;   use Elists;
with Errout;   use Errout;
with Exp_TSS;  use Exp_TSS;
with Exp_Util; use Exp_Util;
with Expander; use Expander;
with Itypes;   use Itypes;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Rtsfind;  use Rtsfind;
with Sem;      use Sem;
with Sem_Disp; use Sem_Disp;
with Sem_Res;  use Sem_Res;
with Sem_Util; use Sem_Util;
with Sinfo;    use Sinfo;
with Snames;   use Snames;
with Tbuild;   use Tbuild;
with Uintp;    use Uintp;

package body Exp_Disp is

   Ada_Actions : constant array (DT_Access_Action) of RE_Id :=
      (Set_Prim_Op_Address     => RE_Set_Prim_Op_Address,
       Get_Prim_Op_Address     => RE_Get_Prim_Op_Address,
       Set_Inheritance_Depth   => RE_Set_Inheritance_Depth,
       Get_Inheritance_Depth   => RE_Get_Inheritance_Depth,
       Set_Ancestor_Tags       => RE_Set_Ancestor_Tags,
       Get_Ancestor_Tags       => RE_Get_Ancestor_Tags,
       DT_Size                 => RE_DT_Size,
       Inherit_DT              => RE_Inherit_DT,
       CW_Membership           => RE_CW_Membership);

   CPP_Actions : constant array (DT_Access_Action) of RE_Id :=
      (Set_Prim_Op_Address     => RE_Set_Vfunction_Address,
       Get_Prim_Op_Address     => RE_Get_Vfunction_Address,
       Set_Inheritance_Depth   => RE_Set_Idepth,
       Get_Inheritance_Depth   => RE_Get_Idepth,
       Set_Ancestor_Tags       => RE_Set_Ancestor_Vptrs,
       Get_Ancestor_Tags       => RE_Get_Ancestor_Vptrs,
       DT_Size                 => RE_Vtable_Size,
       Inherit_DT              => RE_Inherit_Vtable,
       CW_Membership           => RE_CPP_Membership);

   Action_Is_Proc : constant array (DT_Access_Action) of Boolean :=
      (Set_Prim_Op_Address     => True,
       Get_Prim_Op_Address     => False,
       Set_Inheritance_Depth   => True,
       Get_Inheritance_Depth   => False,
       Set_Ancestor_Tags       => True,
       Get_Ancestor_Tags       => False,
       DT_Size                 => False,
       Inherit_DT              => True,
       CW_Membership           => False);

   Action_Nb_Arg : constant array (DT_Access_Action) of Int :=
     (Set_Prim_Op_Address     => 3,
      Get_Prim_Op_Address     => 2,
      Set_Inheritance_Depth   => 2,
      Get_Inheritance_Depth   => 1,
      Set_Ancestor_Tags       => 2,
      Get_Ancestor_Tags       => 1,
      DT_Size                 => 1,
      Inherit_DT              => 3,
      CW_Membership           => 2);


   ---------------------------
   -- Make_DT_Access_Action --
   ---------------------------

   function Make_DT_Access_Action
     (Typ    : Entity_Id;
      Action : DT_Access_Action;
      Args   : List_Id)
      return Node_Id
   is
      Loc         : constant Source_Ptr := Sloc (First (Args));
      Action_Name : Entity_Id;
   begin
      pragma Assert (List_Length (Args) = Action_Nb_Arg (Action));

      if Is_CPP_Class (Root_Type (Typ)) then
         Action_Name := RTE (CPP_Actions (Action));
      else
         Action_Name := RTE (Ada_Actions (Action));
      end if;

      if Action_Is_Proc (Action) then
         return
           Make_Procedure_Call_Statement (Loc,
             Name => New_Reference_To (Action_Name, Loc),
             Parameter_Associations => Args);
      else
         return
           Make_Function_Call (Loc,
             Name => New_Reference_To (Action_Name, Loc),
             Parameter_Associations => Args);
      end if;
   end Make_DT_Access_Action;

   -------------------------
   -- Set_All_DT_Position --
   -------------------------

   procedure Set_All_DT_Position (Typ : Entity_Id) is
      First_Prim : constant Elmt_Id := First_Elmt (Primitive_Operations (Typ));
      Nb_Prim    : Int;
      Prim       : Entity_Id;
      The_Tag    : constant Entity_Id := Tag_Component (Typ);
      Prim_Elmt  : Elmt_Id;

   begin

      --  C++ Case, check that pragma CPP_Class, CPP_Virtual and CPP_Vtable
      --  give a coherent set of information

      if Is_CPP_Class (Typ) then

         --  Compute the number of primitive operations in the main Vtable

         Prim_Elmt := First_Prim;
         Nb_Prim := 0;
         while Present (Prim_Elmt) loop
            Prim := Node (Prim_Elmt);

            if Present (Alias (Prim)) then
               Set_DTC_Entity (Prim, DTC_Entity (Alias (Prim)));
            end if;

            if No (DTC_Entity (Prim)) then
               Error_Msg_NE
                 ("is a primitive operation of&, pragma CPP_Virtual required",
                  Prim, Typ);

            elsif DTC_Entity (Prim) = The_Tag then
               Nb_Prim := Nb_Prim + 1;
               if DT_Position (Prim) = No_Uint then
                  Set_DT_Position (Prim, UI_From_Int (Nb_Prim));
               end if;
            end if;
            Prim_Elmt := Next_Elmt (Prim_Elmt);
         end loop;

         --  Check that the declared size of the Vtable is bigger or equal
         --  than the number of primitive operations (if bigger it means that
         --  some of the c++ virtual functions were not imported, that is
         --  allowed)

         if DT_Entry_Count (The_Tag) = No_Uint then
            Set_DT_Entry_Count (The_Tag, UI_From_Int (Nb_Prim));

         elsif UI_To_Int (DT_Entry_Count (The_Tag)) < Nb_Prim then
            Error_Msg_N ("not enough room in the Vtable for all virtual"
              & " functions", The_Tag);
         end if;

         --  Check that Positions are not duplicate nor outside the range of
         --  the Vtable

         declare
            Size : constant Int := UI_To_Int (DT_Entry_Count (The_Tag));
            Pos  : Int;
            Prim_Pos_Table : array (1 .. Size) of Entity_Id
              := (others => Empty);

         begin

            Prim_Elmt := First_Prim;
            while Present (Prim_Elmt) loop
               Prim := Node (Prim_Elmt);

               if DTC_Entity (Prim) = The_Tag then
                  Pos := UI_To_Int (DT_Position (Prim));

                  if Pos not in Prim_Pos_Table'Range then
                     Error_Msg_N
                       ("position not in range of virtual table", Prim);

                  elsif Present (Prim_Pos_Table (Pos)) then
                     Error_Msg_NE ("cannot be at the same position in the"
                       & " vtable than&", Prim, Prim_Pos_Table (Pos));

                  else
                     Prim_Pos_Table (Pos) := Prim;
                  end if;
               end if;

               Prim_Elmt := Next_Elmt (Prim_Elmt);
            end loop;
         end;

      --  For regular Ada tagged types, just set the DT_Position for each
      --  primitive operation.

      else
         Nb_Prim := 0;
         Prim_Elmt := First_Prim;
         while Present (Prim_Elmt) loop
            Nb_Prim := Nb_Prim + 1;
            Prim := Node (Prim_Elmt);
            Set_DTC_Entity (Prim, The_Tag);
            Set_DT_Position (Prim, UI_From_Int (Nb_Prim));
            Prim_Elmt := Next_Elmt (Prim_Elmt);
         end loop;

         Set_DT_Entry_Count (The_Tag, UI_From_Int (Nb_Prim));
      end if;
   end Set_All_DT_Position;


   -------------
   -- Make_DT --
   -------------

   function Make_DT (Typ : Entity_Id) return List_Id is
      Result  : constant List_Id := New_List;
      Loc     : constant Source_Ptr := Sloc (Typ);

      Tname       : constant Name_Id := Chars (Typ);
      Name_DT     : constant Name_Id := New_External_Name (Tname, 'T');
      Name_DT_Ptr : constant Name_Id := New_External_Name (Tname, 'P');
      Name_ATT    : constant Name_Id := New_External_Name (Tname, 'B');

      DT     : constant Node_Id := Make_Defining_Identifier (Loc, Name_DT);
      DT_Ptr : constant Node_Id := Make_Defining_Identifier (Loc, Name_DT_Ptr);
      ATT    : constant Node_Id := Make_Defining_Identifier (Loc, Name_ATT);

      I_Depth         : Int;
      Generalized_Tag : Entity_Id;

   begin
      if Is_CPP_Class (Root_Type (Typ)) then
         Generalized_Tag := RTE (RE_Vtable_Ptr);
      else
         Generalized_Tag := RTE (RE_Tag);
      end if;

      --  Create the Dispatch_Table object as an array of storage element
      --   DT : Storage_Array (1 .. DT_Size (nb_prim));

      Append_To (Result,
        Make_Object_Declaration (Loc,
          Defining_Identifier => DT,
          Aliased_Present     => True,
          Object_Definition   =>
            Make_Subtype_Indication (Loc,
              Subtype_Mark => New_Reference_To (RTE (RE_Storage_Array), Loc),
              Constraint   => Make_Index_Or_Discriminant_Constraint (Loc,
                Constraints => New_List (
                  Make_Range (Loc,
                    Low_Bound  => Make_Integer_Literal (Loc, Uint_1),
                    High_Bound =>
                      Make_DT_Access_Action (Typ,
                        Action => DT_Size,
                        Args   => New_List (
                          Make_Integer_Literal (Loc,
                            DT_Entry_Count (Tag_Component (Typ)))))))))));

      --  Create the pointer to the dispatch table
      --    DT_Ptr : Tag := Tag!(DT'Address);                 Ada case
      --  or
      --    DT_Ptr : Vtable_Ptr := Vtable_Ptr!(DT'Address);   CPP case


      Append_To (Result,
        Make_Object_Declaration (Loc,
          Defining_Identifier => DT_Ptr,
          Constant_Present    => True,
          Object_Definition   => New_Reference_To (Generalized_Tag, Loc),
          Expression          =>
            Make_Unchecked_Type_Conversion (Loc,
              Subtype_Mark => New_Reference_To (Generalized_Tag, Loc),
              Expression   =>
                Make_Attribute_Reference (Loc,
                  Prefix         => New_Reference_To (DT, Loc),
                  Attribute_Name => Name_Address))));

      --  Set Access_Disp_Table field to be the dispatch table pointer

      Set_Access_Disp_Table (Typ, DT_Ptr);

      --  Count ancestors to compute the inheritance depth. For private
      --  extensions, always go to the full view in order to compute the real
      --  inheritance depth.

      declare
         Parent_Type : Entity_Id := Typ;
         P           : Entity_Id;
      begin
         I_Depth := 0;

         loop
            P := Etype (Parent_Type);

            if Is_Private_Type (P) then
               P := Full_View (P);
            end if;

            exit when P = Parent_Type;

            I_Depth := I_Depth + 1;
            Parent_Type := P;
         end loop;
      end;

      --  Generate Ancestor tags Table:
      --   ATT : aliased Address_Array (0 .. I_Depth);

      Append_To (Result,
        Make_Object_Declaration (Loc,
          Defining_Identifier => ATT,
          Aliased_Present     => True,
          Object_Definition   =>
            Make_Subtype_Indication (Loc,
              Subtype_Mark => New_Reference_To (RTE (RE_Address_Array), Loc),
              Constraint   => Make_Index_Or_Discriminant_Constraint (Loc,
                Constraints => New_List (
                  Make_Range (Loc,
                    Low_Bound  => Make_Integer_Literal (Loc, Uint_0),
                    High_Bound =>
                      Make_Integer_Literal (Loc,
                        UI_From_Int (Int (I_Depth)))))))));

      --  Put the Address of the Ancestor Table in the Dispatch Table

      Append_To (Result,
        Make_DT_Access_Action (Typ,
          Action => Set_Ancestor_Tags,
          Args   => New_List (
            New_Reference_To (DT_Ptr, Loc),                  -- DTptr
              Make_Attribute_Reference (Loc,                 -- Value
              Prefix          => New_Reference_To (ATT, Loc),
              Attribute_Name  => Name_Address))));


      --  For a root type set the Inheritance_Depth and fill the Ancestor Table
      --  Direct Ada descendant of a CPP_Class are considered

      if Typ = Etype (Typ)
        or else (Is_CPP_Class (Etype (Typ)) and then not Is_CPP_Class (Typ))
      then

         --  Set_Inheritance_Depth (DT_ptr, Idepth)

         Append_To (Result,
           Make_DT_Access_Action (Typ,
           Action => Set_Inheritance_Depth,
           Args   => New_List (
             New_Reference_To (DT_Ptr, Loc),                       -- DTptr
             Make_Integer_Literal (Loc, UI_From_Int (I_Depth))))); -- Value


         --  ATT (0) := Address!(DT_Ptr);

         Append_To (Result,
           Make_Assignment_Statement (Loc,
             Name =>
               Make_Indexed_Component (Loc,
                 Prefix      => New_Reference_To (ATT, Loc),
                 Expressions => New_List (Make_Integer_Literal (Loc, Uint_0))),
              Expression =>
                Make_Unchecked_Type_Conversion (Loc,
                  Subtype_Mark => New_Reference_To (RTE (RE_Address), Loc),
                  Expression   => New_Reference_To (DT_Ptr, Loc))));

      --  For a derived type, that is not a direct CPP_Class, call Inherit_DT:
      --     Inherit_DT (Parent_Typ'Tag, DT_Ptr, Parent_Typ'DT_Entry_Count);

      elsif not Is_CPP_Class (Typ) then

         Append_To (Result,
           Make_DT_Access_Action (Typ,
             Action => Inherit_DT,
             Args   => New_List (
               New_Reference_To                           -- Old_DTptr
                 (Access_Disp_Table (Etype (Typ)), Loc),

               New_Reference_To (DT_Ptr, Loc),            -- New_DTptr

               Make_Integer_Literal (Loc,                 -- Entry_Count
                 DT_Entry_Count (Tag_Component (Etype (Typ)))))));
      end if;

      return Result;
   end Make_DT;

   -------------
   -- Fill_DT --
   -------------

   function Fill_DT_Entry
     (Loc  : Source_Ptr;
      Prim : Entity_Id)
      return Node_Id
   is
      Typ    : constant Entity_Id := Scope (DTC_Entity (Prim));
      DT_Ptr : constant Entity_Id := Access_Disp_Table (Typ);

   begin
      return
        Make_DT_Access_Action (Typ,
          Action => Set_Prim_Op_Address,
          Args   => New_List (
            New_Reference_To (DT_Ptr, Loc),                     -- DTptr

            Make_Integer_Literal (Loc, DT_Position (Prim)),     -- Position

            Make_Attribute_Reference (Loc,                      -- Value
              Prefix          => New_Reference_To (Prim, Loc),
              Attribute_Name  => Name_Address)));
   end Fill_DT_Entry;

   --------------------------
   -- Expand_Dispatch_Call --
   --------------------------

   procedure Expand_Dispatch_Call (Call_Node : Node_Id) is
      Call_Typ       : constant Entity_Id := Etype (Call_Node);
      Ctrl_Arg       : constant Node_Id := Controlling_Argument (Call_Node);
      Loc            : constant Source_Ptr := Sloc (Call_Node);
      New_Actions    : constant List_Id := New_List;
      Param_List     : constant List_Id := Parameter_Associations (Call_Node);
      Subp           : constant Entity_Id  := Entity (Name (Call_Node));

      E_Ctrl         : Multi_Use.Exp_Id;
      CW_Typ         : Entity_Id;
      Itype_Node     : Node_Id;
      New_Call       : Node_Id;
      New_Call_Name  : Node_Id;
      New_Params     : List_Id := No_List;
      Param          : Node_Id;
      E_Param        : Multi_Use.Exp_Id;
      Res_Typ        : Entity_Id;
      Subp_Ptr_Typ   : Entity_Id;
      Subp_Typ       : Entity_Id;
      Typ            : Entity_Id;
      use Multi_Use;

      function New_Value (E : Exp_Id; From : Node_Id) return Node_Id;
      --  From is the original Expression, E is the Exp_Id generated by
      --  Multi_Use. New_Value is equivalent to Multi_Use.New_Ref but with
      --  an explicit dereference when From is an access parameter

      function New_Value (E : Exp_Id; From : Node_Id) return Node_Id is
         Loc : constant Source_Ptr := Sloc (From);

      begin
         if Is_Access_Type (Etype (From)) then
            return Make_Explicit_Dereference (Loc, New_Ref (E, Loc));
         else
            return New_Ref (E, Loc);
         end if;
      end New_Value;

   begin
      --  Expand_Dispatch is called directly from the semantics, so we need
      --  a check to see whether expansion is active before proceeding

      if not Expander_Active then
         return;
      end if;

      E_Ctrl := Multi_Use.New_Exp_Id (Ctrl_Arg, New_Actions);

      --  For function calls we attach the Implicit types to an expression
      --  action node, for a procedure call we use an explicit implicit
      --  type node

      if Nkind (Call_Node) = N_Function_Call then
         Itype_Node :=
           Make_Expression_Actions (Loc,
             Actions    => No_List,
             Expression => Empty);
      else
         Itype_Node := Make_Implicit_Types (Loc);
         Append_To (New_Actions, Itype_Node);
      end if;

      --  Definition of the ClassWide Type and the Tagged type

      if Is_Access_Type (Etype (Ctrl_Arg)) then
         CW_Typ := Designated_Type (Etype (Ctrl_Arg));
      else
         CW_Typ := Etype (Ctrl_Arg);
      end if;

      Typ := Root_Type (CW_Typ);

      if Is_CPP_Class (Root_Type (Typ)) then

         --  Create a new parameter list with the displaced 'this'

         New_Params := New_List;
         Param := First_Actual (Call_Node);
         while Present (Param) loop

            --  We assume that dispatching through the main dispatch table
            --  (referenced by Tag_Component) doesn't require a displacement
            --  so the expansion below is only done when dispatching on
            --  another vtable pointer, in which case the first argument
            --  is expanded into :

            --     typ!(Displaced_This (Address!(Param)))

            if Param = Ctrl_Arg
              and then DTC_Entity (Subp) /= Tag_Component (Typ)
            then
               Append_To (New_Params,

                 Make_Unchecked_Type_Conversion (Loc,
                   Subtype_Mark => New_Reference_To (Etype (Param), Loc),
                   Expression   =>
                     Make_Function_Call (Loc,
                       Name => New_Reference_To (RTE (RE_Displaced_This), Loc),
                       Parameter_Associations => New_List (

                        --  Current_This

                         Make_Unchecked_Type_Conversion (Loc,
                           Subtype_Mark =>
                             New_Reference_To (RTE (RE_Address), Loc),
                           Expression   => Relocate_Node (Param)),

                        --  Vptr

                         Make_Selected_Component (Loc,
                            Prefix => Multi_Use.New_Ref (E_Ctrl, Loc),
                            Selector_Name =>
                              New_Reference_To (DTC_Entity (Subp), Loc)),

                        --  Position

                         Make_Integer_Literal (Loc, DT_Position (Subp))))));

            else
               Append_To (New_Params, Relocate_Node (Param));
            end if;

            Param := Next_Actual (Param);
         end loop;

      elsif Present (Param_List) then

         --  Generate the Tag checks when appropriate

         New_Params := New_List;

         Param := First_Actual (Call_Node);
         while Present (Param) loop

            --  No tag check with itself

            if Param = Ctrl_Arg then
               Append_To (New_Params, Multi_Use.New_Ref (E_Ctrl, Loc));

            --  No tag check for parameter whose type is neither tagged nor
            --  access to tagged (for access parameters)

            elsif No (Find_Controlling_Arg (Param)) then
               Append_To (New_Params, Relocate_Node (Param));

            --  No tag check for function dispatching on result it the
            --  Tag given by the context is this one

            elsif Find_Controlling_Arg (Param) = Ctrl_Arg then
               Append_To (New_Params, Relocate_Node (Param));

            --  "=" is the only dispatching operation allowed to get
            --  operands with incompatible tags (it just returns false)

            elsif Chars (Subp) = Name_Op_Eq then
               Append_To (New_Params, Relocate_Node (Param));

            --  No check in presence of suppress flags

            elsif Tag_Checks_Suppressed (Etype (Param))
              or else (Is_Access_Type (Etype (Param))
                         and then Tag_Checks_Suppressed
                                    (Designated_Type (Etype (Param))))
            then
               Append_To (New_Params, Relocate_Node (Param));

            --  Optimization: no tag checks if the parameters are identicals

            elsif Is_Entity_Name (Param)
              and then Is_Entity_Name (Ctrl_Arg)
              and then Entity (Param) = Entity (Ctrl_Arg)
            then
               Append_To (New_Params, Relocate_Node (Param));

            --  Now we need to generate the Tag check

            else

               E_Param := Multi_Use.New_Exp_Id (Param, New_Actions);

               --  Generate code for tag equality check

               Append_To (New_Actions,
                 Make_If_Statement (Loc,
                   Condition =>
                     Make_Op_Ne (Loc,
                       Left_Opnd =>
                         Make_Selected_Component (Loc,
                           Prefix => New_Value (E_Ctrl, Ctrl_Arg),
                           Selector_Name =>
                             New_Reference_To (Tag_Component (Typ), Loc)),

                       Right_Opnd =>
                         Make_Selected_Component (Loc,
                           Prefix =>
                             Make_Unchecked_Type_Conversion (Loc,
                               Subtype_Mark => New_Occurrence_Of (Typ, Loc),
                               Expression   => New_Value (E_Param, Param)),
                           Selector_Name =>
                             New_Reference_To (Tag_Component (Typ), Loc))),

                   Then_Statements =>
                     New_List (New_Constraint_Error (Loc))));

               Append_To (New_Params, Multi_Use.New_Ref (E_Param, Loc));
            end if;

            Param := Next_Actual (Param);
         end loop;
      end if;

      --  Generate the appropriate Subprogram pointer type

      if  Etype (Subp) = Typ then
         Res_Typ := CW_Typ;
      else
         Res_Typ :=  Etype (Subp);
      end if;

      Subp_Typ  := New_Itype (E_Subprogram_Type, Itype_Node);
      Subp_Ptr_Typ := New_Itype (E_Access_Subprogram_Type, Itype_Node);
      Set_Etype (Subp_Typ, Res_Typ);

      --  Create a new list of parameters which is a copy of the old formal
      --  list including the creation of a new set of matching entities.

      declare
         Old_Formal : Entity_Id := First_Formal (Subp);
         New_Formal : Entity_Id;
      begin
         if Present (Old_Formal) then
            New_Formal := New_Copy (Old_Formal);
            Set_First_Entity (Subp_Typ, New_Formal);
            Param := First_Actual (Call_Node);
            loop

               --  Change all the controlling argument types to be class-wide
               --  to avoid a recursion in dispatching

               if Is_Controlling_Actual (Param) then
                  Set_Etype (New_Formal, Etype (Param));
               end if;

               Old_Formal := Next_Formal (Old_Formal);
               exit when No (Old_Formal);

               Set_Next_Entity (New_Formal, New_Copy (Old_Formal));
               New_Formal := Next_Entity (New_Formal);
               Param := Next_Actual (Param);
            end loop;
         end if;
      end;

      Set_Etype (Subp_Ptr_Typ, Subp_Ptr_Typ);
      Set_Directly_Designated_Type (Subp_Ptr_Typ, Subp_Typ);

      --  Generate:
      --   Subp_Ptr_Typ!(Get_Prim_Op_Address (Ctrl._Tag, pos));

      New_Call_Name :=
        Make_Unchecked_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Subp_Ptr_Typ, Loc),
          Expression   =>
            Make_DT_Access_Action (Typ,
              Action => Get_Prim_Op_Address,
              Args => New_List (

               --  Vptr

                Make_Selected_Component (Loc,
                  Prefix => Multi_Use.New_Ref (E_Ctrl, Loc),
                  Selector_Name => New_Reference_To (DTC_Entity (Subp), Loc)),

               --  Position

                Make_Integer_Literal (Loc, DT_Position (Subp)))));


      if Nkind (Call_Node) = N_Function_Call then
         New_Call := Itype_Node;
         Set_Actions (New_Call, New_Actions);
         Set_Expression (New_Call,
           Make_Function_Call (Loc,
             Name => New_Call_Name,
             Parameter_Associations => New_Params));

      else
         Insert_List_Before_And_Analyze (Call_Node, New_Actions);
         New_Call :=
           Make_Procedure_Call_Statement (Loc,
             Name => New_Call_Name,
             Parameter_Associations => New_Params);
      end if;

      Rewrite_Substitute_Tree (Call_Node, New_Call);
      Analyze (Call_Node);
      Resolve (Call_Node, Call_Typ);
   end Expand_Dispatch_Call;

   -----------------------------
   -- Set_Default_Constructor --
   -----------------------------

   procedure Set_Default_Constructor (Typ : Entity_Id) is
      Loc   : Source_Ptr;
      Init  : Entity_Id;
      Param : Entity_Id;
      Decl  : Node_Id;
      E     : Entity_Id;
   begin

      --  Look for the default constructor entity
      --  For now only the default constructor has the flag Is_Constructor.

      E := Next_Entity (Typ);
      while Present (E)
        and then (Ekind (E) /= E_Function or else not Is_Constructor (E))
      loop
         E := Next_Entity (E);
      end loop;

      --  Create the init procedure

      if Present (E) then
         Loc   := Sloc (E);
         Init  := Make_Defining_Identifier (Loc, Name_uInit_Proc);
         Param := Make_Defining_Identifier (Loc, Name_X);
         Decl  :=
           Make_Subprogram_Declaration (Loc,
             Make_Procedure_Specification (Loc,
               Defining_Unit_Name => Init,
               Parameter_Specifications => New_List (
                 Make_Parameter_Specification (Loc,
                   Defining_Identifier => Param,
                   Parameter_Type      =>
                     Make_Access_Definition (Loc,
                      Subtype_Mark => New_Reference_To (Typ, Loc))))));

         Set_Init_Proc (Typ, Init);
         Set_Is_Imported (Init);
         Set_Interface_Name (Init, Interface_Name (E));
         Set_Convention (Init, Convention_C);
         Set_Is_Public (Init);
         Set_Has_Completion (Init);
      end if;
   end Set_Default_Constructor;

end Exp_Disp;
