------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              E X P _ C H 2                               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.25 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Einfo;    use Einfo;
with Elists;   use Elists;
with Nmake;    use Nmake;
with Sem;      use Sem;
with Sem_Util; use Sem_Util;
with Sinfo;    use Sinfo;
with Tbuild;   use Tbuild;
with Snames;   use Snames;

package body Exp_Ch2 is

   -----------------------
   -- Local Subprograms --
   -----------------------

   procedure Expand_Discriminant (N : Node_Id);
   --  An occurence of a discriminant within a discriminated type is replaced
   --  with the corresponding discriminal, that is to say the formal parameter
   --  of the initialization procedure for the type that is associated with
   --  that particular discriminant. This replacement is not performed for
   --  discriminants of records that appear in constraints of component of the
   --  record, because Gigi uses the discriminant name to retrieve its value.
   --  In the other hand, it has to be performed for default expressions of
   --  components because they are used in the record init procedure.  See
   --  Einfo for more details, and Exp_Ch3, Exp_Ch9 for examples of use.

   procedure Expand_Entry_Index_Parameter (N : Node_Id);
   --  A reference to the identifier in the entry index specification
   --  of a protected entry body is modified to a reference to a constant
   --  definintion equal to the index of the entry family member being
   --  called.  This constant is calculated as part of the elaboration
   --  of the expanded code for the body, and is calculated from the
   --  object-wide entry index returned by Next_Entry_Call.

   procedure Expand_Entry_Parameter (N : Node_Id);
   --  A reference to an entry parameter is modified to be a reference to
   --  the corresponding component of the entry parameter record that is
   --  passed by the runtime to the accept body procedure

   procedure Expand_Formal (N : Node_Id);
   --  A reference to a formal parameter of a protected subprogram is
   --  expanded to the corresponding formal of the unprotected procedure
   --  used to represent the protected subprogram within the protected object.

   procedure Expand_Protected_Private (N : Node_Id);
   --  A reference to a private object of a protected type is expanded
   --  to a component selected from the record used to implement
   --  the protected object.  Such a record is passed to all operations
   --  on a protected object in a parameter named _object.  Such an object
   --  can be a left-hand side only within a protected procedure
   --  (or entry, not yet implemented ???).

   -------------------------
   -- Expand_Discriminant --
   -------------------------

   procedure Expand_Discriminant (N : Node_Id) is
      Scop     : constant Entity_Id := Scope (Entity (N));
      P        : Node_Id := N;
      Parent_P : Node_Id := Parent (P);

   begin
      if Ekind (Scop) = E_Record_Type then

         --  Find the origin by walking up the tree till the component
         --  declaration

         while Present (Parent_P)
           and then Nkind (Parent_P) /= N_Component_Declaration
         loop
            P := Parent_P;
            Parent_P := Parent (P);
         end loop;

         --  If the discriminant reference was part of the default expression
         --  it has to be 'discriminalized'

         if Present (Parent_P) and then P = Expression (Parent_P) then
            Set_Entity (N, Discriminal (Entity (N)));
         end if;

      else
         Set_Entity (N, Discriminal (Entity (N)));
      end if;
   end Expand_Discriminant;

   ----------------------------------
   -- Expand_Entry_Index_Parameter --
   ----------------------------------

   procedure Expand_Entry_Index_Parameter (N : Node_Id) is
   begin
      Set_Entity (N, Entry_Index_Constant (Entity (N)));
   end Expand_Entry_Index_Parameter;

   ----------------------------
   -- Expand_Entry_Parameter --
   ----------------------------

   procedure Expand_Entry_Parameter (N : Node_Id) is
      Loc        : constant Source_Ptr := Sloc (N);
      Ent_Formal : constant Entity_Id  := Entity (N);
      Ent_Spec   : constant Entity_Id  := Scope (Ent_Formal);
      Parm_Type  : constant Entity_Id  := Entry_Parameters_Type (Ent_Spec);
      Acc_Stack  : constant Elist_Id   := Accept_Address (Ent_Spec);
      Addr_Ent   : constant Entity_Id  := Node (Last_Elmt (Acc_Stack));
      P_Comp_Ref : Entity_Id;

   begin
      --  What we need is a reference to the corresponding component of the
      --  parameter record object. The Accept_Address field of the entry
      --  entity references the address variable that contains the address
      --  of the accept parameters record. We first have to do an unchecked
      --  conversion to turn this into a pointer to the parameter record and
      --  then we select the required parameter field.

      P_Comp_Ref :=
        Make_Selected_Component (Loc,
          Prefix =>
            Make_Unchecked_Type_Conversion (Loc,
              Subtype_Mark => New_Reference_To (Parm_Type, Loc),
              Expression   => New_Reference_To (Addr_Ent,  Loc)),
          Selector_Name =>
            New_Reference_To (Entry_Component (Ent_Formal), Loc));

      --  That's exactly what we want for the case of an elementary type,
      --  but for a composite type, we have a pointer, since we must pass
      --  such objects by reference. Thus we must dereference them to
      --  access them (this will often be redundant, since the needed
      --  deference is implicit, but no harm is done by making it explicit).

      if Is_By_Copy_Type (Etype (Ent_Formal)) then
         Rewrite_Substitute_Tree (N, P_Comp_Ref);
      else
         Rewrite_Substitute_Tree (N,
           Make_Explicit_Dereference (Loc, P_Comp_Ref));
      end if;

      Analyze (N);
   end Expand_Entry_Parameter;

   -------------------
   -- Expand_Formal --
   -------------------

   procedure Expand_Formal (N : Node_Id) is
      Loc  : constant Source_Ptr := Sloc (N);
      E    : constant Entity_Id  := Entity (N);
      Subp : constant Entity_Id  := Scope (E);

   begin
      if Is_Protected_Type (Scope (Subp))
        and then Chars (Subp) /= Name_uInit_Proc
      then
         Set_Entity (N, Protected_Formal (Entity (N)));
      end if;
   end Expand_Formal;

   ------------------------------
   -- Expand_Protected_Private --
   ------------------------------

   procedure Expand_Protected_Private (N : Node_Id) is
      Loc : constant Source_Ptr := Sloc (N);
      E   : constant Entity_Id  := Entity (N);
      Op  : constant Node_Id    := Protected_Operation (E);

   begin
      if Nkind (Op) /= N_Subprogram_Body
        or else Ekind (Defining_Unit_Name (Specification (Op))) /= E_Function
      then
         Set_Ekind (Prival (E), E_Variable);
      else
         Set_Ekind (Prival (E), E_Constant);
      end if;

      Set_Entity (N, Prival (E));
   end Expand_Protected_Private;

   ----------------------------
   -- Expand_N_Expanded_Name --
   ----------------------------

   --  Performs discriminal and entry parameter replacement as described above

   procedure Expand_N_Expanded_Name (N : Node_Id) is
      E : constant Entity_Id := Entity (N);

   begin
      if Ekind (E) = E_Discriminant then
         Expand_Discriminant (N);

      elsif Is_Entry_Formal (E) then
         Expand_Entry_Parameter (N);

      elsif Ekind (E) = E_Component
        and then Is_Protected_Private (E)
      then
         Expand_Protected_Private (N);

      elsif Ekind (E) = E_Entry_Index_Parameter then
         Expand_Entry_Index_Parameter (N);

      elsif Ekind (E) in Formal_Kind then
         Expand_Formal (N);
      end if;
   end Expand_N_Expanded_Name;

   -------------------------
   -- Expand_N_Identifier --
   -------------------------

   procedure Expand_N_Identifier (N : Node_Id) is
      E : constant Entity_Id := Entity (N);

   begin
      if Ekind (E) = E_Discriminant then
         Expand_Discriminant (N);

      elsif Is_Entry_Formal (E) then
         Expand_Entry_Parameter (N);

      elsif Ekind (E) = E_Component
        and then Is_Protected_Private (E)
      then
         Expand_Protected_Private (N);

      elsif Ekind (E) = E_Entry_Index_Parameter then
         Expand_Entry_Index_Parameter (N);

      elsif Ekind (E) in Formal_Kind then
         Expand_Formal (N);
      end if;
   end Expand_N_Identifier;

end Exp_Ch2;
