/* Main program for the Midnight Commander
   Copyright (C) 1994, 1995 Miguel de Icaza, Janne Kukonlehto
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
#include <ncurses.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/param.h>

#include <sys/stat.h>
#include <sys/types.h>

#ifdef HAVE_UNISTD_H
    #include <unistd.h>
#endif

/* unistd.h defines _POSIX_VERSION on POSIX.1 systems. */
#if defined(HAVE_DIRENT_H) || defined(_POSIX_VERSION)
    #include <dirent.h>
    #define NLENGTH(dirent) (strlen ((dirent)->d_name))
#else
    #define dirent direct
    #define NLENGTH(dirent) ((dirent)->d_namlen)

    #ifdef HAVE_SYS_NDIR_H
        #include <sys/ndir.h>
    #endif /* HAVE_SYS_NDIR_H */

    #ifdef HAVE_SYS_DIR_H
        #include <sys/dir.h>
    #endif /* HAVE_SYS_DIR_H */

    #ifdef HAVE_NDIR_H
        #include <ndir.h>
    #endif /* HAVE_NDIR_H */
#endif /* not (HAVE_DIRENT_H or _POSIX_VERSION) */

#if HAVE_SYS_WAIT_H
    #include <sys/wait.h>	/* For waitpid() */
#endif

#include <errno.h>
#include <pwd.h>
#include <ctype.h>
#include <fcntl.h>	/* For O_RDWR */
#include <signal.h>

/* Program include files */
#include "mad.h"
#include "dir.h"
#include "color.h"
#include "panel.h"
#include "input.h"
#include "global.h"
#include "dialog.h"
#include "menu.h"
#include "file.h"
#include "main.h"
#include "util.h"
#include "win.h"
#include "user.h"
#include "mem.h"
#include "mouse.h"
#include "option.h"
#include "tree.h"
void set_navig_label (WINDOW *fkeys);
#include "chmod.h"
#include "chown.h"
#include "cons.saver.h"
#include "subshell.h"
#include "help.h"
#include "key.h"	/* For init_key() and mi_getch() */
#include "auto.h"	/* try_auto_mount() */
#include "ext.h"	/* regex_command() */
#include "view.h"	/* view() */
#include "find.h"	/* do_find() */
#include "setup.h"	/* save_setup() */
#include "profile.h"	/* free_profiles() */
#include "boxes.h"
#include "hotlist.h"
#include "layout.h"

/* Listbox for the command history feature */
#include "dlg.h"
#include "widget.h"
#include "wtools.h"

static char rcsid [] = "$Id: main.c,v 1.22 1995/01/27 02:35:34 miguel Exp $";

/* Some forward declarations, to prevent compile warnings */
static inline int ITEMS (Panel *p);

/* The structures for the panels */
Panel left_panel, right_panel;

/* Pointers to the selected and unselected panel */
Panel *current_panel = NULL, *other_panel = NULL;

/* This holds the command line */
Input *cmdline;

/* Set when we want use advanced chmod command instead of chmod and/or chown */
int advanced_chfns = 0;

/* Set when main loop should be terminated */
volatile int quit = 0;

/* If set then dialogs just clean the screen when refreshing, else */
/* they do a complete refresh, refreshing all the parts of the program */
int fast_refresh = 0;

/* If true, marking a files moves the cursor down */
int   mark_moves_down = 1;

/* If true, at startup the user-menu is invoked */
int   auto_menu = 0;

/* If true, after executing a command, wait for a keystroke */
enum { pause_never, pause_on_dumb_terminals, pause_always };

int   pause_after_run = pause_on_dumb_terminals;

/* It true saves the setup when quitting */
int auto_save_setup = 0;

/* If true, on Linux Systems, auto-mount and auto-umount directories */
int auto_mount = 0;

/* If true, be eight bit clean */
int eight_bit_clean = 0;

/* If true, next character is quoted */
int quote = 0;

/* If true use the internal viewer */
int use_internal_view = 1;

/* Have we shown the fast-reload warning in the past? */
int fast_reload_w = 0;

/* Move page/item? When clicking on the top or bottom of a panel */
int mouse_move_pages = 1;

/* If true: l&r arrows are used to chdir if the input line is empty */
int navigate_with_arrows = 0;

/* If true all typed chars go to the incremental search routine */
int searching = 0;
static int was_searching = 0;

char *prompt;

/* For slow terminals */
static int force_slow = 0;

/* use mouse? */
int use_mouse_p = GPM_MOUSE;

/* If true, assume we are running on an xterm terminal */
static int force_xterm = 0;

/* Controls screen clearing before an exec */
int clear_before_exec = 1;

/* Asks for confirmation before deleting a file */
int confirm_delete = 1;

/* Asks for confirmation when using F3 to view a directory and there
   are tagged files */
int confirm_view_dir = 0;

/* This flag is set by xterm detection routine in function main() */
/* It is used by function view_other_cmd() */
int xterm_flag = 0;

/* This flag indicates if the pull down menus by default drop down */
int drop_menus = 0;

/* Subshell: if set, then the prompt was not saved on CONSOLE_SAVE */
/* We need to paint it after CONSOLE_RESTORE, see: load_prompt */
int update_prompt = 0;

/* WINDOW handles for the command line and the function keys */
WINDOW *menubar_win = NULL;
WINDOW *output_win = NULL;
WINDOW *cmdline_win = NULL;
WINDOW *fkeys = NULL;
WINDOW *clean_screen = NULL;

/* The currently selected file */
file_entry *selection;

/* The home directory */
char *home_dir;

/* The value of the other directory, only used when loading the setup */
char *other_dir = 0;

/* If true, then print on stdout the last directory we were at */
static int print_last_wd = 0;

/* Ugly hack in order to distinguish between left and right panel in menubar */
int is_right;
#define MENU_PANEL  (is_right ? &right_panel : &left_panel)
#define OTHER_PANEL (is_right ? &left_panel : &right_panel)

/* Quick search status */
char search_buffer [256] = { 0 };
char cmd_buf [512];

struct my_statfs myfs_stats;

static void save_setup_cmd (void);
static void menu_cmd (void);
static void menu_mouse_cmd (int item);
static const int status_mouse_support = 
#ifdef HAVE_LIBGPM
    1;
#else
    0;
#endif
static const int status_using_ncurses = 
#ifdef OTHER_CURSES
    0;
#else
    1;
#endif
static const int status_using_old_tools =
#ifdef OLD_TOOLS
    1;
#else
    0;
#endif

static inline void unselect_item (Panel *panel)
{
    set_attr (cpanel, 0, selection->f.marked);
    repaint_file (cpanel, cpanel->selected, 1);
    panel_refresh (cpanel);
}

#define SELECT_ITEM(panel) \
    selection = &current_panel->dir.list [current_panel->selected]; \
    set_attr (cpanel, 1, selection->f.marked);

/* This function sets the selection variable and redisplays the data */
void select_item (Panel *panel)
{
    int repaint = 0;
    
    /* Although currently all over the code we set the selection and
       top file to decent values before calling select_item, I could
       forget it someday, so it's better to do the actual fitting here */

    if (panel->top_file < 0){
	repaint = 1;
	panel->top_file = 0;
    }

    if (panel->selected < 0)
	panel->selected = 0;

    if (panel->selected > panel->count-1)
	panel->selected = panel->count - 1;

    if (panel->top_file > panel->count-1){
	repaint = 1;
	panel->top_file = panel->count-1;
    }
    
    if (panel->selected < panel->top_file){
	repaint = 1;
	panel->top_file = panel->selected;
    }

    if ((panel->selected - panel->top_file) >= ITEMS (panel)){
	repaint = 1;
	panel->top_file = panel->selected - ITEMS (panel) + 1;
    }
    
    SELECT_ITEM (panel);
    if (repaint)
	paint_panel (panel);
    else
	repaint_file (cpanel, cpanel->selected, 1);
    panel_refresh (panel);
    display_mini_info (panel);
    if (panel == current_panel &&
	(other_panel->view_type == view_info
	 || other_panel->view_type == view_quick)){
	if (horizontal_split || cpanel->win_file != cpanel->big_frame)
	    paint_panel (other_panel);
    }
}

int panel_event    (Gpm_Event *event, Panel *panel);
int menu_bar_event (Gpm_Event *event, void *);

static void init_panels (void)
{
    do_init_panel (&left_panel);
    left_panel.active = 1;
    left_panel.count  = do_load_dir (&left_panel.dir, left_panel.sort_type,
				     left_panel.reverse, left_panel.filter);
    if (other_dir)
	chdir (other_dir);
    
    do_init_panel (&right_panel);
    right_panel.count = do_load_dir (&right_panel.dir, right_panel.sort_type, 
				     right_panel.reverse, right_panel.filter);

    /* Don't select the information window */
    if (left_panel.view_type == view_info){
	cpanel = &right_panel;
	opanel = &left_panel;
	my_statfs (&myfs_stats, opanel->cwd);
    } else {
	cpanel = &left_panel;
	opanel   = &right_panel;
	if (other_dir)
	    chdir (left_panel.cwd);
    }
    cpanel->active = 1;
    opanel->active = 0;

    select_item (cpanel);	/* Inits selection variable */

    display_mini_info (cpanel); /* It needs selection defined */

    display_mini_info (opanel); /* Same here */

    if (other_dir)
	free (other_dir);

    if (the_info_panel)
	my_statfs (&myfs_stats, opanel->cwd);
    
    /* Now, update the screen */
    paint_panel (opanel);
    paint_panel (cpanel);
}

static void done_panels (void)
{
    clean_dir (&cpanel->dir, current_panel->count);
    clean_dir (&opanel->dir, other_panel->count);

    /* Free used memory */
    free (cpanel->user_format);
    free (opanel->user_format);
    free (cpanel->mini_status_format);
    free (opanel->mini_status_format);
    free (cpanel->format);
    free (opanel->format);
    free (cpanel->dir.list);
    free (opanel->dir.list);
}

void try_to_select (Panel *panel, char *name)
{
    Xtry_to_select (panel, name);
#if 0
    /* Original code, I have the impression that I can't make the
       substitution */
    SELECT_ITEM (panel);
#endif
    select_item (panel);
    display_mini_info (panel);
}

/* If we moved to the parent directory move the selection pointer to
   the old directory name */
void cd_try_to_select (Panel *panel)
{
    if (strlen (panel->lwd) > strlen (panel->cwd)
	&& strncmp (panel->cwd, panel->lwd, strlen (panel->cwd)) == 0
	&& strchr (panel->lwd + strlen (panel->cwd) + 1, '/') == 0)
	try_to_select (panel, panel->lwd);
    else
	try_to_select (panel, NULL);
}

/* This routine reloads the directory in both panels. It tries to
** select current_file in current_panel and other_file in other_panel.
** If current_file == -1 then it automatically sets current_file and
** other_file to the currently selected files in the panels.
**
** if force_update has the UP_ONLY_CURRENT bit toggled on, then it
** will not reload the other panel.
*/
void update_panels (int force_update, char *current_file, char *other_file)
{
    int free_pointers = 0;
    int reload_other = !(force_update & UP_ONLY_CURRENT);

    if (force_update & UP_RELOAD){
	cpanel->dont_reload = 0;
	opanel->dont_reload = 0;
	bzero (&(cpanel->dir_stat), sizeof (cpanel->dir_stat));
	if (reload_other)
	    bzero (&(opanel->dir_stat), sizeof (opanel->dir_stat));
    }
    
    /* If current_file == -1 (an invalid pointer) then preserve selection */
    if (current_file == UP_KEEPSEL){
	free_pointers = 1;
	current_file = strdup (cpanel->dir.list [cpanel->selected].fname);
	other_file   = strdup (opanel->dir.list [opanel->selected].fname);
    }
    if (!cpanel->dont_reload)
	panel_reload (cpanel);
    if (cpanel->view_type != view_tree)
	try_to_select (cpanel, current_file);
    if (reload_other){
	if (!opanel->dont_reload)
	    panel_reload (opanel);
	if (cpanel->view_type != view_tree)
	    try_to_select (opanel, other_file);
	paint_dir (opanel);
    }
    paint_dir (cpanel);
    if (free_pointers){
	free (current_file);
	free (other_file);
    }
    chdir (cpanel->cwd);
}

/* Called by parse_control_file */
static int index_by_name (file_entry *list, int count)
{
    char *name;
    int i;

    name = strtok (NULL, " \t\n");
    if (!name || !*name)
	return -1;
    for (i = 0; i < count; i++){
	if (strcmp (name, list[i].fname) == 0)
	    return i;
    }
    return -1;
}

static void select_by_index (Panel *panel, int i);
static void change_panel (void);

/* Called by my_system
   No error reporting, just exits on the first sign of trouble */
static void parse_control_file (void)
{
    char *data, *current;
    Panel *panel;
    file_entry *list;
    int i;
    FILE *file;
    struct stat s;
    
    if ((file = fopen (control_file, "r")) == NULL){
	return;
    }
    /* Use of fstat prevents race conditions */
    if (fstat (fileno (file), &s) != 0){
	fclose (file);
	return;
    }
    /* Security: Check that the user owns the control file to prevent
       other users from playing tricks on him/her. */
    if (s.st_uid != getuid ()){
	fclose (file);
	return;
    }
    data = (char *) xmalloc (s.st_size+1, "main, parse_control_file");
    if (!data){
	fclose (file);
	return;
    }
    if (s.st_size != fread (data, 1, s.st_size, file)){
	free (data);
	fclose (file);
	return;
    }
    data [s.st_size] = 0;
    fclose (file);
    
    /* The Control file has now been loaded to memory -> start parsing. */
    current = strtok (data, " \t\n");
    while (current && *current){
	if (isupper (*current))
	    panel = other_panel;
	else
	    panel = current_panel;

	if ((panel->view_type == view_info) ||
	    (panel->view_type == view_tree) ||
	    (panel->view_type == view_quick))
	    break;

	list = panel->dir.list;
	*current = tolower (*current);

	if (strcmp (current, "clear_tags") == 0){
	    panel->marked = 0;
	    panel->total = 0;
	    panel->dirs_marked = 0;
	    for (i = 0; i < panel->count; i++){
		list [i].f.marked = 0;
	    }
	} else if (strcmp (current, "tag") == 0){
	    i = index_by_name (list, panel->count);
	    if (i >= 0 && ! list[i].f.marked){
		list[i].f.marked = 1;
		if (S_ISDIR (list[i].buf.st_mode))
		    panel->dirs_marked++;
		panel->marked++;
		panel->total += list[i].buf.st_size;
	    }
	} else if (strcmp (current, "untag") == 0){
	    i = index_by_name (list, panel->count);
	    if (i >= 0 && list[i].f.marked){
		list [i].f.marked = 0;
		if (S_ISDIR (list[i].buf.st_mode))
		    panel->dirs_marked--;
		panel->marked--;
		panel->total -= list[i].buf.st_size;
	    }
	} else if (strcmp (current, "select") == 0){
	    i = index_by_name (list, panel->count);
	    if (i >= 0){
		select_by_index (panel, i);
	    }
	} else if (strcmp (current, "change_panel") == 0){
	    change_panel ();
	} else if (strcmp (current, "cd") == 0){
	    int change = 0;
	    current = strtok (NULL, " \t\n");
	    if (!current) break;
	    if (cpanel != panel){
		change_panel ();
		change = 1;
	    }
	    if (do_cd (current)){
		cd_try_to_select (cpanel);
		paint_panel (cpanel);
		select_item (cpanel);
	    }
	    if (change)
		change_panel ();
	} else {
	    /* Unknown command -> let's give up */
	    break;
	}
	current = strtok (NULL, " \t\n");
    }

    free (data);
    paint_panel (cpanel);
    paint_panel (opanel);
}

/* Sets up the terminal before executing a program */
static void pre_exec (void)
{
    if (clear_before_exec){
	wstandend (clean_screen);
	werase (clean_screen);
	touchwin (clean_screen);
	wrefresh (clean_screen);
    } else {
	printf ("\n\n");
    }
    channels_down ();
    keypad (stdscr, FALSE);
    reset_shell_mode ();
    if (use_mouse_p)
	shut_mouse ();
    endwin ();
    if (!status_using_ncurses)
	do_exit_ca_mode ();  /* Done automatically by ncurses' endwin() */
}

/* Restores the terminal after an exec, see execute for extra post-exec code */
static void post_exec (void)
{
    do_enter_ca_mode ();
    reset_prog_mode ();
    flushinp ();
    raw ();
    keypad (stdscr, TRUE);
    channels_up ();
    if (use_mouse_p)
	init_mouse ();
}

/* Save current stat of directories to avoid reloading the panels */
/* when no modifications have taken place */
static void save_cwds_stat (void)
{
    if (fast_reload){
	stat (cpanel->cwd, &(cpanel->dir_stat));
	stat (opanel->cwd, &(opanel->dir_stat));
    }
}

#ifdef HAVE_SUBSHELL_SUPPORT
static void do_possible_cd (char *new_dir)
{
    if (!do_cd (new_dir))
	message (1, " Warning ",
		 " The Commander can't change to the directory that \n"
		 " the subshell claims you are in.  Perhaps you have \n"
		 " deleted your working directory, or given yourself \n"
		 " extra access permissions with the \"su\" command? ");
    paint_panel (cpanel);
    select_item (cpanel);
}

void inline do_update_prompt ()
{
    if (update_prompt){
	printf ("%s", subshell_prompt);
	fflush (stdout);
	update_prompt = 0;
    }
}
#endif

void restore_console (void)
{
    handle_console (CONSOLE_RESTORE);
#ifdef HAVE_SUBSHELL_SUPPORT
    do_update_prompt ();
#endif
}

static void do_execute (const char *shell, const char *command, int internal_command)
{
    #ifdef HAVE_SUBSHELL_SUPPORT
    char *new_dir = NULL;
    #endif
    
    save_cwds_stat ();
    pre_exec ();
    if (console_flag)
	restore_console ();

    unlink (control_file);
    if (!use_subshell && !internal_command)
	printf ("%s%s\n", prompt, command);


    #ifdef HAVE_SUBSHELL_SUPPORT
    if (use_subshell && !internal_command){
	do_update_prompt ();
	if (invoke_subshell (command, VISIBLY, &new_dir))
	    return;  /* User did `exit' or `logout'; close down MC quietly */
    } else
    #endif
	my_system (!internal_command, shell, command);

    if (!internal_command){
	if (console_flag){
	    if (output_lines && keybar_visible) putchar('\n');
	    handle_console (CONSOLE_SAVE);
	}

	if (pause_after_run == pause_always ||
	    (pause_after_run == pause_on_dumb_terminals &&
	     !xterm_flag && !console_flag)){
	    printf ("Press any key to continue...");
	    fflush (stdout);
	    raw ();
	    xgetch ();
	    /* printf ("\r\n");
	    fflush (stdout); */
	}
    }

    post_exec ();
    if (console_flag && output_lines)
	show_console_contents (output_win, LINES-output_lines-keybar_visible-1,
			       LINES-keybar_visible-1);

    #ifdef HAVE_SUBSHELL_SUPPORT
	if (new_dir)
	    do_possible_cd (new_dir);
    #endif

    update_panels (UP_OPTIMIZE, UP_KEEPSEL, 0);
    refresh_screen (NULL);
    parse_control_file ();
    unlink (control_file);
}

void execute (char *command)
{
    #ifdef HAVE_SUBSHELL_SUPPORT
    if (use_subshell)
	if (subshell_state == INACTIVE)
	    do_execute (shell, command, 0);
	else
	    message (1, " Error ", " The shell is already running a command ");
    else
    #endif

	do_execute (shell, command, 0);
}

static void execute_internal (const char *command, const char *args)
{
    do_execute (command, args, 1);
}

static void change_labels (void)
{
    if (cpanel->view_type == view_quick){
	view_labels (fkeys);
	set_label_text (fkeys, 3, "");
	set_label_text (fkeys, 8, "");
    } else {
	if (cpanel->view_type == view_tree){
	    set_label_text (fkeys, 2, "Rescan");
	    set_label_text (fkeys, 3, "Forget");
	    set_navig_label (fkeys);
	} else {
	    set_label_text (fkeys, 2, "Menu");
	    set_label_text (fkeys, 3, "View");
	    set_label_text (fkeys, 4, "Edit");
	}
	set_label_text (fkeys, 5, "Copy");
	set_label_text (fkeys, 6, "RenMov");
	set_label_text (fkeys, 7, "MkDir");
	set_label_text (fkeys, 8, "Delete");
    }
    if (keybar_visible && fkeys){
	touchwin (fkeys);
	wrefresh (fkeys);
    }
}

/* Changes the currently selected panel to the other panel */
static void change_panel (void)
{
    if (other_panel->view_type == view_info)
	return;
    if (other_panel->view_type == view_quick
	&& S_ISDIR (cpanel->dir.list [cpanel->selected].buf.st_mode))
	return;

    current_panel->active = 0;
    if (current_panel->view_type != view_quick &&
	current_panel->view_type != view_tree){
	show_dir (current_panel);
	unselect_item (current_panel);
    }
    if (current_panel->view_type == view_tree)
	paint_panel (current_panel);
    other_panel = current_panel;
    if (current_panel == &left_panel)
	current_panel = &right_panel;
    else
	current_panel = &left_panel;
    current_panel->active = 1;
    change_labels ();
    if (current_panel->view_type != view_quick &&
	current_panel->view_type != view_tree){
	if (chdir (current_panel->cwd) != 0){
	    message (1, " Error ", " Can't chdir to %s \n %s ",
		     current_panel->cwd, unix_error_string (errno));
	}
	show_dir (current_panel);
	select_item (current_panel);
    }
    if (current_panel->view_type == view_tree
	|| current_panel->view_type == view_quick)
	paint_panel (current_panel);
}

static int quit_cmd (void)
{
    if (query_dialog (" The Midnight Commander ",
		      " Do you really want to quit the Midnight Commander? ",
		      0, 2, " Yes ", " No ") == 0)
    #ifdef HAVE_SUBSHELL_SUPPORT
	if (use_subshell)
	    exit_subshell ();
	else
    #endif
	    quit = 1;

    return quit;
}

/* Returns the number of items in the given panel */
static inline int ITEMS (Panel *p)
{
    if (p->view_type == view_brief
	|| (p->view_type == view_user && p->split))
	return p->lines * 2;
    else
	return p->lines;
}

/* Called by parse_control_file */
static void select_by_index (Panel *panel, int i)
{
    if (i >= panel->count)
	return;
    
    unselect_item (panel);
    panel->selected = i;
    
    while (panel->selected - panel->top_file >= ITEMS (panel)){
	/* Scroll window half screen */
	panel->top_file += ITEMS (panel)/2;
	paint_dir (panel);
	select_item (panel);
    } 
    while (panel->selected < panel->top_file){
	/* Scroll window half screen */
	panel->top_file -= ITEMS (panel)/2;
	if (panel->top_file < 0) panel->top_file = 0;
	paint_dir (panel);
    } 
    select_item (panel);
    panel_refresh (panel);
}

static void move_down (void)
{
    if (cpanel->view_type == view_tree){
	tree_move_forward (1);
	paint_panel (cpanel);
	return;
    }

    if (cpanel->selected+1 == cpanel->count)
	return;
    
    unselect_item (cpanel);
    cpanel->selected++;
    
    if (cpanel->selected - cpanel->top_file == ITEMS (cpanel)){
	/* Scroll window half screen */
	cpanel->top_file += ITEMS (cpanel)/2;
	paint_dir (cpanel);
	select_item (cpanel);
    } 
    select_item (cpanel);
    panel_refresh (cpanel);
}

static void move_up (void)
{
    if (cpanel->view_type == view_tree){
	tree_move_backward (1);
	paint_panel (cpanel);
	return;
    }

    if (cpanel->selected == 0)
	return;

    unselect_item (cpanel);
    cpanel->selected--;
    if (cpanel->selected < cpanel->top_file){
	/* Scroll window half screen */
	cpanel->top_file -= ITEMS (cpanel)/2;
	if (cpanel->top_file < 0) cpanel->top_file = 0;
	paint_dir (cpanel);
    } 
    select_item (cpanel);
    panel_refresh (cpanel);
}

static void move_home (void)
{
    if (cpanel->view_type == view_tree){
	tree_move_to_top (1);
	paint_panel (cpanel);
	return;
    }

    if (cpanel->selected == 0)
	return;
    unselect_item (cpanel);
    cpanel->top_file = 0;
    cpanel->selected = 0;
    paint_dir (cpanel);
    select_item (cpanel);
    panel_refresh (cpanel);
}

static void move_end (void)
{
    if (cpanel->view_type == view_tree){
	tree_move_to_bottom (1);
	paint_panel (cpanel);
	return;
    }

    if (cpanel->selected == cpanel->count-1)
	return;
    unselect_item (cpanel);
    cpanel->selected = cpanel->count-1;
    paint_dir (cpanel);
    select_item (cpanel);
    panel_refresh (cpanel);
}

static void prev_page (void)
{
    if (cpanel->view_type == view_tree){
	tree_move_backward (tree_lines - 1);
	paint_panel (cpanel);
	return;
    }

    unselect_item (cpanel);
    cpanel->selected -= ITEMS (cpanel);
    cpanel->top_file -= ITEMS (cpanel);

    /* This keeps the selection in a reasonable place */
    if (cpanel->selected < 0)
	cpanel->selected = 0;
    if (cpanel->top_file < 0)
	cpanel->top_file = 0;
    select_item (cpanel);
    paint_dir (cpanel);
    panel_refresh (cpanel);
}

static void next_page (void)
{
    if (cpanel->view_type == view_tree){
	tree_move_forward (tree_lines - 1);
	paint_panel (cpanel);
	return;
    }

    unselect_item (cpanel);
    cpanel->selected += ITEMS (cpanel);
    cpanel->top_file += ITEMS (cpanel);

    /* This keeps the selection in it's relative position */
    if (cpanel->selected >= cpanel->count)
	cpanel->selected = cpanel->count - 1;
    if (cpanel->top_file >= cpanel->count)
	cpanel->top_file = cpanel->count - 1;
    select_item (cpanel);
    paint_dir (cpanel);
    panel_refresh (cpanel);
}

/* This routine marks a file or a directory */
static void do_mark_file (int do_move)
{
    /*
     * Only '..' can't be marked, '.' isn't visible.
     */
    if (strcmp (selection->fname, "..")){
        selection->f.marked = selection->f.marked ? 0 : 1;
        set_attr  (cpanel, 1, selection->f.marked);
        repaint_file (cpanel, cpanel->selected, 1);
        if (selection->f.marked){
            cpanel->marked++;
            if (S_ISDIR (selection->buf.st_mode))
                cpanel->dirs_marked++;
            cpanel->total += selection->buf.st_size;
            set_colors (cpanel);
            if (cpanel->marked == 1)
	        mvwprintw (cpanel->win_file, cpanel->lines+3, 1,
		           "%-*s", cpanel->cols, "", cpanel->cols);
        } else {
            if (S_ISDIR(selection->buf.st_mode))
                cpanel->dirs_marked--;
            cpanel->marked--;
            cpanel->total -= selection->buf.st_size;
        }
    }
    if (mark_moves_down && do_move)
	move_down ();
    display_mini_info (cpanel);
    panel_refresh (cpanel);
}

static void mark_file (void)
{
    if (is_view_special (cpanel->view_type))
	return;
    
    do_mark_file (1);
}

/*
 * Touch window and refresh window functions
 */

/* The extra parameter is used because some routines need to refresh */
/* or touch the panel windows in a specified order to avoid excesive */
/* redrawings */
   
static void only_refresh_screen (int refresh_panels)
{
    if (refresh_panels){
	panel_refresh (other_panel);
	panel_refresh (current_panel);
    }
    if (menubar_visible && menubar_win)
	wrefresh (menubar_win);
    if (console_flag && output_lines && output_win)
	wrefresh (output_win);
    if (command_prompt && cmdline_win)
	wrefresh (cmdline_win);
    if (keybar_visible && fkeys)
	wrefresh (fkeys);
}

static void only_touchwin (int touch_panels)
{
    if (touch_panels){
	touchwin (current_panel->win_file);
	touchwin (other_panel->win_file);
    }
    if (menubar_visible && menubar_win)
	touchwin (menubar_win);
    if (console_flag && output_lines && output_win)
	touchwin (output_win);
    if (command_prompt && cmdline_win)
	touchwin (cmdline_win);
    if (keybar_visible && fkeys)
	touchwin (fkeys);
}

/* This routine untouches the first line on both panels in order */
/* to avoid the refreshing the menu bar */
void untouch_bar (void)
{
    Panel *long_panel = 0;	/* To avoid flicker: only repaint that one */
    
    only_touchwin (1);
    
    #ifdef BUGGY_CURSES
	/* On Solaris the provided wtouchln in curses doesn't work */
    #else
	wtouchln (cpanel->win_file, 0, 1, 0);
        if (!horizontal_split)
	    wtouchln (opanel->win_file, 0, 1, 0);
    #endif
    
    if (opanel->view_type == view_long)
	long_panel = opanel;
    if (cpanel->view_type == view_long)
	long_panel = cpanel;
    if (long_panel)
	panel_refresh (long_panel);
    else {
	panel_refresh (opanel);
	panel_refresh (cpanel);
    }

    /* We already refreshed the panels in the correct order, don't
       need to refresh them again */
    only_refresh_screen (0);
}

void clr_scr (void)
{
    wstandend (clean_screen);
    wclear (clean_screen);
    touchwin (clean_screen);
    wrefresh (clean_screen);
}

void repaint_screen (int clear)
{
    if (clear){
	clr_scr ();
    }
    touchwin (other_panel->win_file);
    panel_refresh (other_panel);
    touchwin (current_panel->win_file);
    panel_refresh (current_panel);

    only_touchwin (0);
    only_refresh_screen (0);
}

static void repaint_screen_cmd (void)
{
    repaint_screen (RP_CLEAR);
}

void refresh_screen (void *unused)
{
    only_touchwin (1);
    only_refresh_screen (1);
}

static void help_cmd (void)
{
    interactive_display (LIBDIR "mc.hlp", "[main]");
}

int do_cd (char *new_dir)
{
    int  ret;
    DIR  *dirp;
    char *directory;
    char temp [MAXPATHLEN];

    while (*new_dir == ' ')
	new_dir++;
    
    if (!strcmp (new_dir, "-")){
	strcpy (temp, cpanel->lwd);
	new_dir = temp;
    }
    strcpy (cpanel->lwd, cpanel->cwd);
    
    directory = *new_dir ? new_dir : home_dir;

    /* If it requires expansion */

    directory = tilde_expand (directory);
    
    dirp = opendir (directory);
    if (!dirp){
	free (directory);
	return 0;
    }
    closedir (dirp);

    if (auto_mount)
	try_auto_mount (directory);

    ret = chdir (directory);
    free (directory);
    
    if (auto_mount)
	try_auto_umount (cpanel->cwd);

    if (ret == -1)
	return 0;
    
    clean_dir (&cpanel->dir, cpanel->count);
    cpanel->count = do_load_dir (&cpanel->dir, cpanel->sort_type,
				 cpanel->reverse, cpanel->filter);
    cpanel->top_file = 0;
    cpanel->selected = 0;
    cpanel->marked = 0;
    cpanel->dirs_marked = 0;
    cpanel->total = 0;
    get_current_wd (cpanel->cwd, sizeof (cpanel->cwd)-2);
    if (cpanel->view_type == view_info
	|| opanel->view_type == view_info)
	my_statfs (&myfs_stats, cpanel->cwd);
    return 1;
}

static void action (void)
{
    char *dir;

    if (cpanel->view_type == view_quick)
	return;
    if (cpanel->view_type == view_tree){
	dir = cpanel->cwd;
	change_panel ();
	if (do_cd (dir)){
	    cd_try_to_select (cpanel);
	    paint_panel (cpanel);
	    select_item (cpanel);
	} else
	    message (1, " Error ", " Can't chdir to \"%s\" \n %s ",
		     dir, unix_error_string (errno));
	change_panel ();
	return;
    }
    if (S_ISDIR (selection->buf.st_mode) || link_isdir (selection)){
	if (!do_cd (selection->fname))
	    return;
	
	cd_try_to_select (cpanel);

	paint_panel (cpanel);
	select_item (cpanel);
	return;
    }
    if (is_exe (selection->buf.st_mode) && if_link_is_exe (selection)){
	char *tmp = copy_strings ("./", selection->fname, 0);
	execute (tmp);
	free (tmp);
	return;
    }
    regex_command (selection->fname);
}

#ifdef HAVE_SUBSHELL_SUPPORT
static void set_new_prompt (void)
{
    /* Should be nothing, but there seems to be a problem */
    prompt = strip_ctrl_codes (subshell_prompt);
    input_set_prompt (cmdline, COLS, prompt);
}

int load_prompt (int fd)
{
    read_subshell_prompt (QUIETLY);
    if (command_prompt){
	prompt = strip_ctrl_codes (subshell_prompt);
	input_set_prompt (cmdline, COLS, prompt);
	if (refresh_list && refresh_list->refresh_fn == refresh_screen)
	    wrefresh (cmdline_win);
    }
    update_prompt = 1;
    return 0;
}
#endif

static void enter (void)
{
    if (command_prompt && strlen (cmdline->buffer)){

	/* FIXME: any initial whitespace should be removed at this point */

	if (strncmp (cmdline->buffer, "cd ", 3) == 0 ||
	    strcmp (cmdline->buffer, "cd") == 0){
	    if (cpanel->view_type == view_quick)
		change_panel ();
	    if (cmdline->buffer [2] == 0)
		cmdline->buffer [3] = 0;

	    /* FIXME: any final whitespace should be removed here
	       (to see why, try "cd fred "). */

	    if (cpanel->view_type == view_tree){
		if (cmdline->buffer [0] == 0)
		    tree_chdir (home_dir);
		else if (strcmp (cmdline->buffer+3, "..") == 0){
		    char *dir = cpanel->cwd;
		    int len = strlen (dir);
		    while (len && dir [--len] != '/');
		    dir [len] = 0;
		    if (len)
			tree_chdir (dir);
		    else
			tree_chdir ("/");
		} else if (cmdline->buffer [3] == '/'){
		    tree_chdir (cmdline->buffer+3);
		} else {
		    char *old = cpanel->cwd;
		    char *new;
		    new = copy_strings (old, "/", cmdline->buffer+3, 0);
		    tree_chdir (new);
		    free (new);
		}
	    } else if (do_cd (&cmdline->buffer [3])){
		cd_try_to_select (cpanel);
	    } else {
		message (1, " Error ", " Can't chdir to '%s' \n %s ",
			 &cmdline->buffer [3], unix_error_string (errno));
		return;
	    }
	    paint_panel (cpanel);
	    if (cpanel->view_type != view_tree)
		select_item (cpanel);
	    new_input (cmdline);
	} else {
	    char *command, *s;
	    int i, j;

	    command = xmalloc (strlen (cmdline->buffer) + 1, "main, enter");
	    command [0] = 0;
	    for (i = j = 0; i < strlen (cmdline->buffer); i ++){
		if (cmdline->buffer [i] == '%'){
		    i ++;
		    s = expand_format (cmdline->buffer [i]);
		    command = realloc (command, strlen (command) + strlen (s)
				       + strlen (cmdline->buffer) - i + 1);
		    strcat (command, s);
		    free (s);
		    j = strlen (command);
		} else {
		    command [j] = cmdline->buffer [i];
		    j ++;
		}
		command [j] = 0;
	    }
	    execute (command);
	    free (command);
	    
	    #ifdef HAVE_SUBSHELL_SUPPORT
		if (quit == SUBSHELL_EXIT)
		    return;
		if (use_subshell)
		    set_new_prompt ();
	    #endif

	    new_input (cmdline);
	}
    }
    else
	action ();
}

/* Incremental search of a file name in the panel */
static void do_search (int c_code)
{
    int l, i;
    int wrapped = 0;
    int found;

    l = strlen (search_buffer);
    if (l && (c_code == 8 || c_code == 0177 || c_code == KEY_BACKSPACE))
	search_buffer [--l] = 0;
    else {
	if (c_code && l < sizeof (search_buffer)){
	    search_buffer [l] = c_code;
	    search_buffer [l+1] = 0;
	    l++;
	}
    }
    
    if (cpanel->view_type == view_tree){
	if (!search_tree (search_buffer))
	    search_buffer [--l] = 0;
	paint_panel (cpanel);
	return;
    }
    
    found = 0;
    for (i = cpanel->selected; !wrapped || i != cpanel->selected; i++){
	if (i >= cpanel->count){
	    i = 0;
	    if (wrapped)
		break;
	    wrapped = 1;
	}
	if (strncmp (cpanel->dir.list [i].fname, search_buffer, l) == 0){
	    cpanel->selected = i;
	    #if 0
		cpanel->top_file = cpanel->selected - cpanel->lines/2; 
	    #endif
	    select_item (cpanel);
	    found = 1;
	    break;
	}
    }
    if (!found)
	search_buffer [--l] = 0;
    
    paint_panel (cpanel);
}

static void start_search (void)
{
    searching = 1;
    if (was_searching){
	move_down ();
	do_search (0);
    } else {
	search_buffer [0] = 0;
	display_mini_info (cpanel);
	wrefresh (cpanel->win_file);
    }
}

static int mouse_marking = 0;

static void mouse_toggle_mark (void)
{
    do_mark_file (0);
    mouse_marking = selection->f.marked;
}

static void mouse_set_mark (void)
{
    if (mouse_marking && !selection->f.marked)
	do_mark_file (0);
    else if (!mouse_marking && selection->f.marked)
	do_mark_file (0);
}

static inline int mark_if_marking (Gpm_Event *event)
{
    if (event->buttons & GPM_B_RIGHT){
	if (event->type & GPM_DOWN)
	    mouse_toggle_mark ();
	else
	    mouse_set_mark ();
	return 1;
    }
    return 0;
}
    
int panel_event (Gpm_Event *event, Panel *panel)
{
    int index, longlist = 0;
    int xdisp = 0; /* Displacement of event->x if in long directory listing
                      whenever the event->x is used, we use a value of
                      event->x + xdisp instead. This is necessary
                      for correct operation of split full cols panels*/

    /* Special case: the long directory listing */
    if ((cpanel->view_type == view_long
	|| cpanel->cols == cpanel->full_cols)
	&& !horizontal_split){
	    if (panel != cpanel){
	    	panel = cpanel;
	    	longlist = 1;
	    	if (panel == &left_panel)
	    	    xdisp = first_panel_size;
	    } else if (panel == &right_panel)
		    xdisp = first_panel_size;	    
    }	    

    if ((event->type & (GPM_DOWN|GPM_DRAG))){

	/* This breaks autorepeating if we get into the other panel */
	/* We do this only if not in a long directory listing*/
	if (!longlist){ 
    	    if (horizontal_split){
    	        if ((panel == &left_panel && event->y > first_panel_size)
    	            || (panel == &right_panel && event->y < 0))
    	            return MOU_NORMAL;
            } else {
                if ((panel == &left_panel && event->x > first_panel_size)
                    || (panel == &right_panel && event->x < 0))
                    return MOU_NORMAL;
            }
        }
        
	if (panel != current_panel)
	    change_panel ();

    	if (panel->view_type == view_quick){
	    int result;
	    if (view_event(event,&result))
	        paint_panel (cpanel);
	    return result;
        }
        
	if (event->y <= 0){
	    mark_if_marking (event);
	    if (mouse_move_pages)
		prev_page ();
	    else
		move_up ();
	    return MOU_REPEAT;
	}

	if (!((panel->view_type == view_tree ||
	       panel->top_file + event->y <= panel->count) &&
	      event->y <= panel->lines)){
	    mark_if_marking (event);
	    if (mouse_move_pages)
		next_page ();
	    else
		move_down ();
	    return MOU_REPEAT;
	} else {
	    if (cpanel->view_type == view_tree){
		tree_event (event->y - 1);
		paint_panel (cpanel);
		return MOU_NORMAL;
	    }
	    index = panel->top_file + event->y - 1;
	    if (panel->view_type == view_brief
		|| (cpanel->view_type == view_user && panel->split)){
		if (event->x + xdisp > (panel->cols/2))
		    index += panel->lines;
	    }
	}

	if (index >= panel->count)
	    index = panel->count - 1;
	
	if (index != panel->selected){
	    unselect_item (panel);
	    panel->selected = index;
	    select_item (panel);
	}
	if (!mark_if_marking (event))
	    panel_refresh (panel);
    } else if ((event->type & (GPM_UP|GPM_DOUBLE)) == (GPM_UP|GPM_DOUBLE)){
    	if (panel->view_type != view_quick)
            if (event->y > 0 && event->y <= panel->lines)
	        action ();
    }
    return MOU_NORMAL;
}

int menu_bar_event (Gpm_Event *event, void *x)
{
    if (event->type != GPM_DOWN)
	return MOU_NORMAL;
    menu_mouse_cmd (event->x / 13);
    return MOU_ENDLOOP;
}

/* If the key pressed is not bound to any command, this function is called */
static void default_key (int c_code)
{
    if (searching){
	do_search (c_code);
    } else {
	if (command_prompt) {
	    handle_char (cmdline, c_code);
	} else {
	    start_search ();
	    do_search (c_code);
	}
    }
}

/* Changes the selection by lines (may be negative) */
static void move_selection (Panel *panel, int lines)
{
    int new_pos;
    int adjust = 0;

    new_pos = panel->selected + lines;
    if (new_pos >= panel->count)
	new_pos = panel->count-1;

    if (new_pos < 0)
	new_pos = 0;

    unselect_item (panel);
    panel->selected = new_pos;

    if (panel->selected - panel->top_file >= ITEMS (panel)){
	panel->top_file += lines;
	adjust = 1;
    }
    
    if (panel->selected - panel->top_file < 0){
	panel->top_file += lines;
	adjust = 1;
    }
    
    if (adjust){
	if (panel->top_file > panel->selected)
	    panel->top_file = panel->selected;
	if (panel->top_file < 0)
	    panel->top_file = 0;
	paint_dir (panel);
    }
    select_item (panel);
    panel_refresh (panel);
}

/* Used to emulate Lynx's entering leaving a directory with the arrow keys */
static void maybe_cd (int char_code, int move_up_dir)
{
    if (navigate_with_arrows){
	if (!cmdline->buffer [0]){
	    if (!move_up_dir){
		if (do_cd ("..")){
		    cd_try_to_select (cpanel);
		    paint_panel (cpanel);
		    select_item (cpanel);
		}
		return;
	    }
	    if (S_ISDIR (selection->buf.st_mode) || link_isdir (selection)){
		action ();
		return;
	    }
	}
    }
    default_key (char_code);
}

static void move_left (int c_code)
{
    if (cpanel->view_type == view_brief
	|| (cpanel->view_type == view_user && cpanel->split))
	move_selection (cpanel, -cpanel->lines);
    else if (cpanel->view_type == view_tree /*&& tree_navigation_flag*/){
	tree_move_to_parent ();
	paint_panel (cpanel);
    } else {
	maybe_cd (c_code, 0);
    }
}

static void move_right (int c_code)
{
    if (cpanel->view_type == view_brief
	|| (cpanel->view_type == view_user && cpanel->split))
	move_selection (cpanel, cpanel->lines);
    else if (cpanel->view_type == view_tree /*&& tree_navigation_flag*/){
	tree_move_to_child ();
	paint_panel (cpanel);
    } else
	maybe_cd (c_code, 1);
}

static void key_f2_cmd (void)
{
    if (cpanel->view_type == view_tree){
	tree_rescan_cmd ();
	paint_panel (cpanel);
    } else if (cpanel->view_type == view_quick)
	toggle_wrap_mode ();
    else
	user_menu_cmd ();
}

static void view_cmd (void)
{
    static char *viewer = 0;

    /* Directories are viewed by changing to them */
    if (S_ISDIR (selection->buf.st_mode)){
	if (confirm_view_dir && (cpanel->marked || cpanel->dirs_marked)){
	    if (query_dialog (" CD ", "Files tagged, want to cd?",
			      0, 2, " Yes ", " No ") == 1){
		return;
	    }
	}
	action ();
    } else if (use_internal_view){
	view (selection->fname);
	repaint_screen (RP_NOCLEAR);
	if (opanel->view_type == view_quick)
	    paint_quick_view_panel (opanel);
    } else {
	if (!viewer){
	    viewer = getenv ("PAGER");
	    if (!viewer)
		viewer = "view";
	}
	execute_internal (viewer, selection->fname);
    }
}

static void key_f3_cmd (void)
{
    if (cpanel->view_type == view_tree){
	tree_forget_cmd ();
	paint_panel (cpanel);
    } else if (cpanel->view_type != view_quick)
	view_cmd ();
}

static void do_edit (const char *what)
{
    static char *editor = 0;

    if (!editor){
	editor = getenv ("EDITOR");
	if (!editor)
	    editor = "vi";
    }
    execute_internal (editor, what);
}

static void edit_cmd (void)
{
    searching = 0;		/* FIXME: We shouldn't need this here */
    do_edit (selection->fname);
}

static void key_f4_cmd (void)
{
    if (cpanel->view_type == view_tree){
	tree_navigation_flag = 1 - tree_navigation_flag;
	set_navig_label (fkeys);
	paint_panel (cpanel);
    } else if (cpanel->view_type == view_quick)
	toggle_hex_mode ();
    else
	edit_cmd ();
}

/* Clears all files in the panel, used only when one file was marked */

void unmark_file (Panel *panel)
{
    int i;

    if (panel->view_type == view_tree)
	return;
    if (!panel->marked)
	return;
    for (i = 0; i < panel->count; i++)
	panel->dir.list [i].f.marked = 0;
    panel->dirs_marked = 0;
    panel->marked = 0;
    panel->total = 0;
}

static void copy_cmd (void)
{
    save_cwds_stat ();
    if (panel_operate (OP_COPY)){
	update_panels (UP_OPTIMIZE, UP_KEEPSEL, 0);
	repaint_screen (RP_NOCLEAR);
    }
}

static void key_f5_cmd (void)
{
    if (cpanel->view_type == view_quick)
	goto_line ();
    else
	copy_cmd ();
}

static void mask_ren_cmd (void)
{
    save_cwds_stat ();
    mask_rename (0);

    update_panels (UP_OPTIMIZE, UP_KEEPSEL, 0);
    repaint_screen (RP_NOCLEAR);
}

static void ren_cmd (void)
{
    save_cwds_stat ();
    if (panel_operate (OP_MOVE)){
	update_panels (UP_OPTIMIZE, UP_KEEPSEL, 0);
	repaint_screen (RP_NOCLEAR);
    }
}

static void key_f6_cmd (void)
{
    if (cpanel->view_type == view_quick)
	regexp_search ();
    else
	ren_cmd ();
}

static void errno_report (char *s)
{
    message (1, " Error ", " %s: %s ", s, unix_error_string (errno));
}

static void do_link (int symbolic_link)
{
    struct stat s;
    char *dest, *src;
    int  stat_r;

    if (!symbolic_link){
	stat_r = stat (selection->fname, &s);
	if (stat_r != 0){
	    message (1, " Error ", " Couldn't stat %s \n %s ",
		     selection->fname, unix_error_string (errno));
	    return;
	}
	if (!S_ISREG (s.st_mode))
	    return;
    }
    
    if (!symbolic_link){
	dest = input_expand_dialog (" Link ", " Link file to: ", "");
	if (!dest)
	    return;
	save_cwds_stat ();
	if (-1 == link (selection->fname, dest))
	    errno_report ("link");
    } else {
	dest = input_expand_dialog (" Symlink ", " Existing file: ",
				    selection->fname);
	if (!dest)
	    return;
	src = input_expand_dialog (" Symlink ", " New symlinked file: ", "");
	
	if (src){
	    save_cwds_stat ();
	    if (-1 == symlink (dest, src))
		errno_report ("symlink");
	    free (src);
	}
    }
    free (dest);
    update_panels (UP_OPTIMIZE, UP_KEEPSEL, 0);
    repaint_screen (RP_NOCLEAR);
}

static void link_cmd (void)
{
    do_link (0);
}

static void symlink_cmd (void)
{
    do_link (1);
}

void mkdir_cmd (void)
{
    char *dir = input_expand_dialog (" Mkdir ", "Enter directory name:" , "");
    if (!dir)
	return;

    save_cwds_stat ();
    if (my_mkdir (dir, 0777) == 0){
	update_panels (UP_OPTIMIZE, dir, 0);
	repaint_screen (RP_NOCLEAR);
	if (cpanel->view_type != view_tree)
	    select_item (cpanel);
	free (dir);
	return;
    }
    free (dir);
    message (1, " Error ", "  %s  ", unix_error_string (errno));
}

static void key_f7_cmd (void)
{
    if (cpanel->view_type == view_quick)
	normal_search ();
    else
	mkdir_cmd ();
}

static void delete_cmd (void)
{
    save_cwds_stat ();

    if (panel_operate (OP_DELETE)){
	update_panels (UP_OPTIMIZE, UP_KEEPSEL, 0);
	repaint_screen (RP_NOCLEAR);
    }
}

static void key_f8_cmd (void)
{
    if (cpanel->view_type != view_quick)
	delete_cmd ();
}

static void tree_cmd (void)
{
    char *new_dir;

    new_dir = tree (cpanel->cwd);
    if (cpanel->view_type == view_tree){
	paint_panel (cpanel);
	return;
    }
    if (!new_dir || !do_cd (new_dir))
	return;
    cd_try_to_select (cpanel);
    paint_panel (cpanel);
    select_item (cpanel);
}

static void find_cmd (void)
{
    do_find ();
}

static void swap_cmd (void)
{
    Panel temp1, *temp2;

    temp1 = left_panel;
    left_panel = right_panel;
    right_panel = temp1;

    temp2 = current_panel;
    current_panel = other_panel;
    other_panel = temp2;

    first_panel_size = COLS - first_panel_size;
    mvwin (left_panel.small_frame, menubar_visible, 0);
    mvwin (right_panel.small_frame,
	   menubar_visible + (horizontal_split ? first_panel_size : 0),
	   horizontal_split ? 0 : first_panel_size);

    paint_panel (other_panel);
    paint_panel (current_panel);
}

static void view_other_cmd (void)
{
    static int message_flag = TRUE;
    #ifdef HAVE_SUBSHELL_SUPPORT
    char *new_dir = NULL;
    #endif

    if (!xterm_flag && !console_flag && !use_subshell){
	if (message_flag)
	    message (1, " Error ", " Not an xterm or Linux console; \n"
				   " the panels cannot be toggled. ");
	message_flag = FALSE;
    } else {
	if (use_mouse_p)
	    shut_mouse ();
	if (clear_before_exec){
	    /* FIXME: isn't there a simpler way of deleting
	       the screen?  Using raw terminfo, perhaps? */
	    wstandend (clean_screen);
	    werase (clean_screen);
	    touchwin (clean_screen);
	    wrefresh (clean_screen);
	}
	reset_shell_mode ();
	endwin ();
	if (!status_using_ncurses)
	    do_exit_ca_mode ();  /* Done automatically by ncurses' endwin() */
	raw ();
	noecho ();
	if (console_flag)
	    restore_console ();

	#ifdef HAVE_SUBSHELL_SUPPORT
	if (use_subshell){
	    if (invoke_subshell (NULL, VISIBLY, &new_dir))
		return;  /* User did `exit' or `logout': quit MC quietly */
	} else
	#endif
	    mi_getch ();

	if (console_flag)
	    handle_console (CONSOLE_SAVE);
	do_enter_ca_mode ();
	reset_prog_mode ();
	if (use_mouse_p)
	    init_mouse ();

	#ifdef HAVE_SUBSHELL_SUPPORT
	if (use_subshell){
	    set_new_prompt ();
	    if (new_dir)
		do_possible_cd (new_dir);
	    if (console_flag && output_lines)
		show_console_contents (output_win,
				       LINES-keybar_visible-output_lines-1,
				       LINES-keybar_visible-1);
	}
	#endif

	repaint_screen (0);
    }
}

static void compare_dir (Panel *panel, Panel *other)
{
    int i, j;

    /* No marks by default */
    panel->marked = 0;
    panel->total = 0;
    panel->dirs_marked = 0;
    
    /* Handle all files in the panel */
    for (i = 0; i < panel->count; i++){
	file_entry *source = &panel->dir.list[i];

	/* Default: unmarked */
	source->f.marked = 0;
	/* Skip directories */
	if (S_ISDIR (source->buf.st_mode))
	    continue;

	/* Search the corresponding entry from the other panel */
	for (j = 0; j < other->count; j++){
	    if (strcmp (source->fname,
			other->dir.list[j].fname) == 0)
		break;
	}
	if (j >= other->count)
	    /* Not found -> mark */
	    source->f.marked = 1;
	else {
	    /* Found */
	    file_entry *target = &other->dir.list[j];

	    /* Mark if newer (or same date with different size) */
	    if (source->buf.st_mtime > target->buf.st_mtime)
		source->f.marked = 1;
	    else if (source->buf.st_mtime == target->buf.st_mtime
		     && source->buf.st_size != target->buf.st_size)
		source->f.marked = 1;
	}

	/* If marked increase counts */
	if (source->f.marked){
	    panel->marked++;
	    panel->total += source->buf.st_size;
	}
    } /* for (i ...) */
}

static void compare_dirs (void)
{
    compare_dir (cpanel, opanel);
    compare_dir (opanel, cpanel);
    paint_panel (cpanel);
    paint_panel (opanel);
}

extern void assign_text (Input *in, char *text);

static void history_cmd (void)
{
    Listbox *listbox;
    Hist *current;

    if (!cmdline->history){
	message (1, " Error ", " The command history is empty ");
	return;
    }
    if (cmdline->need_push){
	push_history (cmdline, cmdline->buffer);
	cmdline->need_push = 0;
    }
    free (cmdline->history->text);
    cmdline->history->text = strdup (cmdline->buffer);
    
    current = cmdline->history;
    while (current->prev)
	current = current->prev;
    listbox = create_listbox_window (60, 10, " Command history ",
				     "[Command Menu]");
    while (current){
	LISTBOX_APPEND_TEXT (listbox, 0, current->text,
			     current);
	current = current->next;
    }
    run_dlg (listbox->dlg);
    if (listbox->dlg->ret_value == B_CANCEL)
	current = NULL;
    else
	current = listbox->list->current->data;
    destroy_dlg (listbox->dlg);
    delwin (listbox->w);
    free (listbox);
    do_refresh ();

    if (!current)
	return;
    cmdline->history = current;
    assign_text (cmdline, cmdline->history->text);
    update_input (cmdline);
}

static void quick_chdir (void)
{
    char *target;

    target = hotlist_cmd ();
    if (!target)
	return;

    if (cpanel->view_type == view_tree)
	tree_chdir (target);
    else {
	if (!do_cd (target))
	    return;
	cd_try_to_select (cpanel);
    }
    paint_panel (cpanel);
    select_item (cpanel);
}

/* Check if the file exists */
/* If not copy the default */
static int check_for_default(char *default_file, char *file)
{
    struct stat s;
    if (stat (file, &s)){
	if (stat (default_file, &s)){
	    return -1;
	}
	create_op_win (OP_COPY);
	copy_file_file (default_file, file);
	destroy_op_win ();
    }
    return 0;
}

static void ext_cmd (void)
{
    char *buffer;
    int  dir;

    dir = 0;
    if (geteuid () == 0){
	dir = query_dialog ("Extension file edit",
			    " Which extension file you want to edit? ", 0, 2,
			    " User ", " System Wide ");
    }
    if (dir == 0){
	buffer = copy_strings (home_dir, "/.mc.ext", 0);
	check_for_default (LIBDIR "mc.ext", buffer);
	do_edit (buffer);
	free (buffer);
    } else if (dir == 1)
	do_edit (LIBDIR "mc.ext");
}

static void menu_edit_cmd (void)
{
#define MC_MENU "mc.menu"

    char *buffer, *home_file;
    int dir = 0;
    
    if (geteuid () == 0)
	dir = query_dialog ("Menu file edit",
			    " Which menu file will you edit? ", 0, 2,
			      " User ", " System Wide ");

    home_file = copy_strings (home_dir, "/.mc.menu", 0);

    switch (dir){
    case 0:
	buffer = strdup (home_file);
	check_for_default (LIBDIR MC_MENU, buffer);
	break;

    case 1:
	buffer = copy_strings (LIBDIR, MC_MENU, 0);
	break;

    default:
	free (home_file);
	return;
    }
    do_edit (buffer);
    free (buffer);
    free (home_file);
}

static void select_cmd (void)
{
    char *reg_exp, *reg_exp_t;
    int i;
    int c;
    int dirflag = 0;

    reg_exp = input_dialog (" Select ", "", easy_patterns ? "*" : ".");
    if (!reg_exp)
	return;
    
    reg_exp_t = reg_exp;

    /* Check if they specified a directory */
    if (*reg_exp_t == '/'){
        dirflag = 1;
        reg_exp_t++;
    }
    if (reg_exp_t [strlen(reg_exp_t) - 1] == '/'){
        dirflag = 1;
        reg_exp_t [strlen(reg_exp_t) - 1] = 0;
    }

    for (i = 0; i < cpanel->count; i++){
	if (S_ISDIR (cpanel->dir.list [i].buf.st_mode)){
	    if (!dirflag)
                continue;
            if (!strcmp( cpanel->dir.list [i].fname, ".."))
                continue;
        } else {
            if (dirflag)
                continue;
	}
	c = regexp_match (reg_exp_t, cpanel->dir.list [i].fname, match_file);
	if (c == -1){
	    message (1, " Error ", "  Malformed regular expression  ");
	    free (reg_exp);
	    return;
	}
	if (c){
	    if (!cpanel->dir.list [i].f.marked){
		cpanel->marked++;
		if (S_ISDIR (cpanel->dir.list[i].buf.st_mode))
		    cpanel->dirs_marked++;
		cpanel->total += cpanel->dir.list [i].buf.st_size;
	    }
	    cpanel->dir.list [i].f.marked = 1;
	}
    }
    paint_panel (cpanel);
    panel_refresh (cpanel);
    panel_refresh (cpanel);
    free (reg_exp);
}

static void unselect_cmd (void)
{
    char *reg_exp, *reg_exp_t;
    int i;
    int c;
    int dirflag = 0;

    reg_exp = input_dialog (" Unselect ","", easy_patterns ? "*" : ".");
    if (!reg_exp)
	return;
    
    reg_exp_t = reg_exp;
    
    /* Check if they specified directory matching */
    if (*reg_exp_t == '/'){
        dirflag = 1;
	reg_exp_t ++;
    }
    if (reg_exp_t [strlen(reg_exp_t) - 1] == '/'){
        dirflag = 1;
        reg_exp_t [strlen(reg_exp_t) - 1] = 0;
    }
    for (i = 0; i < cpanel->count; i++){
	if (S_ISDIR (cpanel->dir.list [i].buf.st_mode)){
	    if (!dirflag)
	        continue;
	    if (!strcmp( cpanel->dir.list [i].fname, ".."))
	        continue;
        } else {
            if (dirflag)
                continue;
        }
	c = regexp_match (reg_exp_t, cpanel->dir.list [i].fname, match_file);
	if (c == -1){
	    message (1, " Error ", "  Malformed regular expression  ");
	    free (reg_exp);
	    return;
	}
	if (c){
	    if (cpanel->dir.list [i].f.marked){
		if (S_ISDIR (cpanel->dir.list[i].buf.st_mode))
		    cpanel->dirs_marked--;
		cpanel->marked--;
		cpanel->total -= cpanel->dir.list [i].buf.st_size;
	    }
	    cpanel->dir.list [i].f.marked = 0;
	}
    }
    paint_panel (cpanel);
    panel_refresh (cpanel);
    panel_refresh (cpanel);
    free (reg_exp);
}

static void reverse_selection (void)
{
    file_entry *file;
    int i;

    for (i = 0; i < cpanel->count; i++){
	file = &cpanel->dir.list [i];
	if (S_ISDIR (file->buf.st_mode))
	    continue;
	if (file->f.marked){
	    file->f.marked = 0;
	    cpanel->marked--;
	    cpanel->total-=file->buf.st_size;
	} else {
	    file->f.marked = 1;
	    cpanel->marked++;
	    cpanel->total+=file->buf.st_size;
	}
    }
    paint_panel (cpanel);
    panel_refresh (cpanel);
    panel_refresh (cpanel);
}

static void tree_leave (Panel *p)
{
    char *dir;
    int change = 0;
    
    dir = p->cwd;
    if (cpanel != p){
	change_panel ();
	change = 1;
    }
    if (do_cd (dir)){
	cd_try_to_select (cpanel);
	paint_panel (cpanel);
	select_item (cpanel);
    } else
	message (1, " Error ", " Can't chdir to \"%s\" \n %s ",
		 dir, unix_error_string (errno));
    if (change)
	change_panel ();
    else {
	/* A little hackish... */
	p->view_type = view_full;
	change_labels ();
	p->view_type = view_tree;
    }
}

void paint_info_panel (Panel *panel)
{
    Panel *other;
    WINDOW *w = panel->win_file;
    struct stat buf;

    if (! is_idle ())
	return;
    set_colors (panel);
    werase (w);
    wclr (w);
    box (w, ACS_VLINE, ACS_HLINE);

    set_attr (panel, 0, 1);
    mvwprintw (w, 1, 3, "Midnight Commander %s", VERSION);
    set_attr (panel, 0, 0);
    wmove (w, 2, 1);
    whline (w, ACS_HLINE|NORMAL_COLOR, panel->cols);
    if (cpanel == panel)
	other = opanel;
    else
	other = cpanel;
    if (other->view_type == view_tree && other->view_type == view_info
	&& other->view_type == view_quick)
	return;
    buf = other->dir.list [other->selected].buf;

    /* Print only lines which fit */
    switch (panel->lines + 1 + 2*show_mini_info){
	/* Note: all cases are fall-throughs */
	
    default:

    case 16:
	if (myfs_stats.nfree >0 || myfs_stats.nodes > 0)
	    mvwprintw (w, 16, 3, "Free nodes %d (%d%%) of %d",
		       myfs_stats.nfree,
		       myfs_stats.total
		       ? 100 * myfs_stats.nfree / myfs_stats.nodes : 0,
		       myfs_stats.nodes);
	else
	    mvwprintw (w, 16, 3, "No node information");
    case 15:
	if (myfs_stats.avail > 0 || myfs_stats.total > 0){
	    mvwprintw (w, 15, 3, "Free space ");
	    wprint_bytesize (w, myfs_stats.avail, 1);
	    wprintw (w, " (%d%%) of ",
		     myfs_stats.total
		     ? 100 * myfs_stats.avail / myfs_stats.total : 0);
	    wprint_bytesize (w, myfs_stats.total, 1);
	} else
	    mvwprintw (w, 15, 3, "No space information");

    case 14:
	mvwprintw (w, 14, 3, "Type:      %s ", myfs_stats.typename);
	if (myfs_stats.type != 0xffff && myfs_stats.type != 0xffffffff)
	    wprintw (w, " (%Xh)", myfs_stats.type);

    case 13:
	mvwprintw (w, 13, 3, "Device:    %s",
		   name_trunc (myfs_stats.device, panel->cols - 13));
    case 12:
	mvwprintw (w, 12, 2, "Filesystem: %s",
		   name_trunc (myfs_stats.mpoint, panel->cols - 13));

    case 11:
	mvwprintw (w, 11, 3, "Accessed:  %s", file_date (buf.st_atime));
    case 10:
	mvwprintw (w, 10, 3, "Modified:  %s", file_date (buf.st_mtime));
    case 9:
	mvwprintw (w, 9, 3, "Created:   %s", file_date (buf.st_ctime));

    case 8:
#ifdef HAVE_ST_RDEV
	if (buf.st_rdev)
	    mvwprintw (w, 8, 3, "Inode dev: major: %d, minor: %d",
		       buf.st_rdev >> 8, buf.st_rdev & 0xff);
	else
#endif
	{
	    mvwprintw (w, 8, 3, "Size:      ");
	    wprint_bytesize (w, buf.st_size, 0);
	    wprintw (w, " (%d blocks)", buf.st_blocks);
	}
    case 7:
	mvwprintw (w,  7, 3, "Owner:     %s/%s", get_owner (buf.st_uid),
		   get_group (buf.st_gid));
    case 6:
	mvwprintw (w,  6, 3, "Links:     %d", buf.st_nlink);
    case 5:
	mvwprintw (w,  5, 3, "Mode:      %s (%o)",
		   string_perm (buf.st_mode), buf.st_mode & 07777);
    case 4:
	mvwprintw (w,  4, 3, "Location:  %Xh:%Xh", buf.st_dev, buf.st_ino);
    case 3:
	mvwprintw (w,  3, 2, "File:       %s",
		   name_trunc (other->dir.list [other->selected].fname,
			       panel->cols - 13));
    case 2:
    case 1:
    case 0:
	;
    } /* switch */
}

void change_view (Panel *p, int view_type);

/* Switches panel p to info mode */
static void switch_to_info (Panel *p, Panel *o)
{
    if (p->view_type == view_tree)
	tree_leave (p);

    if (!is_view_file_listing(o->view_type))
	return;

    /*if (p->view_type == view_info)
	change_view (p, view_full);
    else */{
	my_statfs (&myfs_stats, o->cwd);
	p->view_type = view_info;
	info_frame (p);
	paint_panel (p);
	if (p == cpanel)
	    change_panel ();
    }
}

static void info_cmd_no_menu (void)
{
    switch_to_info (opanel, cpanel);
}

void paint_quick_view_panel (Panel *panel)
{
    Panel *other;
    WINDOW *w = panel->win_file;
    char *filename;

    if (!is_idle ())
	return;
    set_colors (panel);
    if (cpanel == panel)
	other = opanel;
    else
	other = cpanel;
    
    if (is_view_file_listing (other->view_type)
	&& S_ISREG (other->dir.list [other->selected].buf.st_mode))
    {
	box (w, ACS_VLINE, ACS_HLINE);
	filename = other->dir.list [other->selected].fname;
	if (panel->active)
	    wattron (panel->win_file, REVERSE_COLOR);
	mvwaddstr (panel->win_file, 0, 1, "Quick view");
	if (panel->active)
	    wstandend (panel->win_file);

	view_init (panel->win_file, fkeys, panel->cols + 1,
		   panel->lines + 2 + 2*show_mini_info, filename, 1);
	view_update ();
    } else {
    	/* The blank panel shouldn't ever be current */
    	if (panel == current_panel)
    	    change_panel ();
	werase (w);
	wclr (w);
	box (w, ACS_VLINE, ACS_HLINE);
	view_init_windows (panel->win_file, fkeys);
	view_update (); /* is this one needed ? */
    }
}

static int quick_view_key (int key)
{
    int result = view_check_key (key, 1-command_prompt);
    paint_panel (cpanel);
    return result;
}

static void quick_view_cmd (void)
{
    Panel *p = MENU_PANEL;
    Panel *o = OTHER_PANEL;

    if (o->view_type == view_quick || o->view_type == view_info
	|| o->view_type == view_tree)
	return;
    if (p->view_type == view_tree)
	tree_leave (p);

    p->view_type = view_quick;
    p->split = 0;
    chdir (o->cwd);
    paint_panel (p);
}

static void tree_view_cmd (void)
{
    Panel *p = MENU_PANEL;
    Panel *o = OTHER_PANEL;

    if (o->view_type == view_tree || o->view_type == view_quick
	|| o->view_type == view_info)
	return;
    if (p->view_type == view_tree)
	tree_leave (p);
    
    if (tree_init (p->cwd, p->lines) == -1)
	return;
    p->view_type = view_tree;
    clean_dir (&p->dir, p->count);
    p->count = set_zero_dir (&p->dir);
    change_labels ();
    paint_panel (p);
}

void change_view (Panel *p, int view_type)
{
   char  *error;

   if (view_type != view_tree)
	if (p->view_type == view_tree)
	    tree_leave (p);

   if (view_type != view_quick && p->view_type == view_quick)
       view_done ();
   
   if (view_type != view_quick && view_type != view_tree &&
       view_type != view_info)
       chdir (p->cwd);

    if (view_type == view_tree){
	tree_view_cmd ();
	return;
    }

    if (view_type == view_info){
	switch_to_info (p, p == cpanel ? opanel : cpanel);
	return;
    }

    if (view_type == view_quick){
	quick_view_cmd ();
	return;
    }

    p->view_type = view_type;

    error = set_panel_format (p, panel_format (p));
    if (error){
	message (1, " Error ", error);
	free (error);
	p->view_type = view_full;
	set_panel_format (p, panel_format (p));
    }
   if (p != cpanel && cpanel->view_type != view_quick
       && cpanel->view_type != view_info){
       chdir (cpanel->cwd);
   }
}    

/* Panel sorting related routines */
static void do_re_sort (Panel *panel)
{
    char *filename = strdup (selection->fname);
    int  i;
    
    unselect_item (panel);
    do_sort (&panel->dir, panel->sort_type, panel->count-1, panel->reverse);
    panel->selected = -1;
    for (i = panel->count; i; i--){
	if (!strcmp (panel->dir.list [i-1].fname, filename)){
	    panel->selected = i-1;
	    break;
	}
    }
    free (filename);
    cpanel->top_file = cpanel->selected - cpanel->lines/2;
    if (cpanel->top_file < 0)
	cpanel->top_file = 0;
    select_item (panel);
    paint_dir (panel);
    panel_refresh (panel);
}

static void reread_cmd (void)
{
    int flag;

    flag = strcmp (cpanel->cwd, opanel->cwd) ? UP_ONLY_CURRENT : 0;
	
    update_panels (UP_RELOAD|flag, UP_KEEPSEL, 0);
    repaint_screen (RP_NOCLEAR);
}

static void filter_cmd (void)
{
    char *reg_exp;
    char *x;
    Panel *p = MENU_PANEL;

    x = p->filter ? p->filter : easy_patterns ? "*" : ".";
	
    reg_exp = input_dialog (" Filter ", "", x);
    if (!reg_exp)
	return;
    if (p->filter){
	free (p->filter);
	p->filter = 0;
    }
    if (!(reg_exp [0] == '*' && reg_exp [1] == 0))
	p->filter = reg_exp;
    reread_cmd ();
}

static void display_cmd (void)
{
    Panel *p = MENU_PANEL;
    int   view_type;
     
    view_type = display_box (p);

    if (view_type == -1)
	return;

    change_view (p, view_type);
}

static void sort_cmd (void)
{
    Panel  *p = MENU_PANEL;
    sortfn *sort_order;
    
    sort_order = sort_box (p->sort_type, &p->reverse);

    if (sort_order == 0)
	return;

    p->sort_type = sort_order;

    /* The directory is already sorted, we have to load the unsorted stuff */
    if (sort_order == (sortfn *) unsorted){
	char *current_file;
	
	current_file = strdup (cpanel->dir.list [cpanel->selected].fname);
	panel_reload (cpanel);
	try_to_select (cpanel, current_file);
	free (current_file);
    }
    do_re_sort (p);
}

static menu_entry PanelMenu [] = {
    { " Display mode...",	'D',        display_cmd },
    { " Sort order...",	        'S',        sort_cmd },
    { " Filter...",	        'F',        filter_cmd },
    { "", ' ', 0 },
    { " Rescan    C-r",         XCTRL('R'), reread_cmd }
};

static menu_entry RightMenu [] = {
    { " Display mode...",	'D',        display_cmd },
    { " Sort order...",	        'S',        sort_cmd },
    { " Filter...",	        'F',        filter_cmd },
    { "", ' ', 0 },
    { " Rescan    C-r",         XCTRL('R'), reread_cmd }
};

/*    "Advanced chown  ", 'A',   chown_advanced_cmd, */

static menu_entry FileMenu [] = {
    { "User menu       F2",     KEY_F(2), user_menu_cmd },
    { "View            F3",     KEY_F(3), view_cmd },
    { "Edit            F4",     KEY_F(4), edit_cmd },
    { "Copy            F5",     KEY_F(5), copy_cmd },
    { "Link            C-x l",  'L',      link_cmd },
    { "SymLink         C-x s",  'S',      symlink_cmd },
    { "Chmod           C-x c",  'C',      chmod_cmd },
    { "chOwn           C-x o",  'O',      chown_cmd },
    { "Rename/Move     F6",     KEY_F(6), ren_cmd },
    { "Mask rename     C-x r",  'M',      mask_ren_cmd },
    { "Mkdir           F7",     KEY_F(7), mkdir_cmd },
    { "Delete          F8",     KEY_F(8), delete_cmd },
    { "", ' ', 0 },
    { "Select group     + ",    '+',	  select_cmd },
    { "Unselect group   \\",    '\\',     unselect_cmd },
    { "Reverse selection  ",    'R',	  reverse_selection },
    { "", ' ', 0 },
    { "Quit            F10",    KEY_F(10), (callfn) quit_cmd }
};

static menu_entry CmdMenu [] = {
    { "Directory tree",		   'D',	         tree_cmd },
    { "Find file",		   'F',	         find_cmd },
    { "Swap panels          C-u",  XCTRL('u'),   swap_cmd },
    { "Switch panels on/off C-o",  XCTRL('o'),   view_other_cmd },
    { "Compare directories",	   'C',	         compare_dirs },
    { "", ' ', 0 },
    { "command History",           'H',	         history_cmd },
    { "Directory hotlist    C-\\",  XCTRL('\\'), quick_chdir },
    { "", ' ', 0 }, 
    { "Extension file edit",	   'E',	         ext_cmd },
    { "Menu file edit",		   'M',	         menu_edit_cmd }
};

/* Must keep in sync with the constants in menu_cmd */
static menu_entry OptMenu [] = {
    { "Configuration...",    'c', configure_box },
    { "Layout...",           'l', layout_cmd },
    { "Save setup",          's', save_setup_cmd }
};

#define menu_entries(x) sizeof(x)/sizeof(menu_entry)

static Menu *MenuBar [5];
static Menu *MenuBarEmpty [5];

static void init_menu (void)
{
    int i;
    
    MenuBar [0] = create_menu (" Left ", PanelMenu, menu_entries (PanelMenu));
    MenuBar [1] = create_menu (" File ", FileMenu, menu_entries (FileMenu));
    MenuBar [2] = create_menu (" Command ", CmdMenu, menu_entries (CmdMenu));
    MenuBar [3] = create_menu (" Options ", OptMenu, menu_entries (OptMenu));
    MenuBar [4] = create_menu (" Right ", RightMenu, menu_entries (PanelMenu));

    for (i = 0; i < 5; i++)
	MenuBarEmpty [i] = create_menu (MenuBar [i]->name, 0, 0);
}

static void done_menu (void)
{
    int i;

    for (i = 0; i < 5; i++){
	free (MenuBar [i]);
	free (MenuBarEmpty [i]);
    }
}

/* This variable is shared between the following three functions */
static int last_menu_selection;

/* All the drop_menu mess is so that the user could select from the
   capital letters on the MenuBar.  Suggested by Torben */

enum {
    menu_select_current,
    menu_select_last,
    menu_select_pos
};

static void menu_display_cmd (int select_type, int drop_menu, int item)
{
    int first_selected;
    int bar_result = 0;
    WINDOW *bar_window;

    if (select_type == menu_select_last)
	first_selected = last_menu_selection;
    else if (select_type == menu_select_current)
	first_selected = current_panel == &right_panel ? 4 : 0;
    else
	first_selected = item;

    push_frame (0, 0, 0);

    bar_result = first_selected;
    
    bar_window = newwin (1, COLS, 0, 0);
    if (!drop_menu){
	bar_result = run_bar (bar_window, 0, 0, COLS, 12, 5, first_selected,
			      MenuBarEmpty, BAR_FILL|BAR_NO_REFRESH,
			      A_BOLD, SELECTED_COLOR);
	
    }
    if (bar_result >= 0){
	bar_result = run_bar (bar_window, 0, 0, COLS, 12, 5, bar_result,
			      MenuBar, BAR_FILL, A_BOLD, SELECTED_COLOR);
	if (bar_result >= 0)
	    last_menu_selection = bar_result;
    } else
	refresh_screen (NULL);
    delwin (bar_window);
    pop_frame ();
}

static void menu_cmd (void)
{
    menu_display_cmd (menu_select_current, drop_menus, 0);
}

static void menu_last_selected_cmd (void)
{
    menu_display_cmd (menu_select_last, drop_menus, 0);
}

static void menu_mouse_cmd (int menu_item)
{
    menu_item = menu_item > 4 ? 4 : menu_item;
    menu_display_cmd (menu_select_pos, 1, menu_item);
}

/* Flag toggling functions */

void toggle_advanced_chfns (void)
{
    advanced_chfns = !advanced_chfns;
}

void toggle_drop_menus (void)
{
    drop_menus = !drop_menus;
}

void toggle_auto_mount (void)
{
    auto_mount = !auto_mount;
}

void toggle_eight_bit (void)
{
    eight_bit_clean = !eight_bit_clean;
    meta (stdscr, eight_bit_clean);
}

void toggle_confirm_delete (void)
{
    confirm_delete = !confirm_delete;
}

void toggle_fast_reload (void)
{
    fast_reload = !fast_reload;
    if (fast_reload_w == 0 && fast_reload){
	message (0, " Information ",
		 " Using the fast reload option may not reflect the exact \n"
		 " directory contents. In this cases you'll need to do a  \n"
		 " manual reload of the directory. See the man page for   \n"
		 " the details.                                           ");
	fast_reload_w = 1;
    }
}

void toggle_mix_all_files (void)
{
    mix_all_files = !mix_all_files;
    update_panels (UP_RELOAD, UP_KEEPSEL, 0);
}

void toggle_internal (void)
{
    use_internal_view = !use_internal_view;
}

void toggle_auto_menu (void)
{
    auto_menu = !auto_menu;
}

void toggle_show_backup (void)
{
    show_backups = !show_backups;
    update_panels (UP_RELOAD, UP_KEEPSEL, 0);
}

void toggle_show_hidden (void)
{
    show_dot_files = !show_dot_files;
    update_panels (UP_RELOAD, UP_KEEPSEL, 0);
}

void toggle_verbose (void)
{
    verbose = !verbose;
}

void toggle_auto_save (void)
{
    auto_save_setup = !auto_save_setup;
}

void toggle_mark_move_down (void)
{
    mark_moves_down = !mark_moves_down;
}

void toggle_show_mini_status (void)
{
    if (show_mini_info){
	left_panel.lines += 2;
	right_panel.lines +=2;
    } else {
	left_panel.lines -= 2;
	right_panel.lines -= 2;
    }
    show_mini_info = !show_mini_info;
    paint_panel (cpanel);
    paint_panel (opanel);
}

void toggle_easy_patterns (void)
{
    easy_patterns = !easy_patterns;
}

void toggle_align_extensions (void)
{
    align_extensions = !align_extensions;
}

static void dump_marked (void)
{
    int i;
    
    system ("clear");
    move (1, 1);
    for (i = 1; i < 8; i++){
	attron (COLOR_PAIR (i));
	printw ("Este es el color %d\n", i);
    }
    xgetch ();
    fprintf (stderr, "Hay %d archivos marcados\n", cpanel->marked);
    for (i = 0; i < cpanel->count; i++){
	if (cpanel->dir.list [i].f.marked)
	    fprintf (stderr, "%s\n", cpanel->dir.list [i].fname);
    }
    xgetch ();
    mouse_log ("MARCA-F12", __FILE__, __LINE__);
    repaint_screen (RP_NOCLEAR);
}

/* Select a file name in a panel */

static void goto_top_file (void)
{
    if (is_view_special (cpanel->view_type))
	return;
    
    unselect_item (cpanel);
    cpanel->selected = cpanel->top_file;
    select_item (cpanel);
    panel_refresh (cpanel);
}

static void goto_middle_file (void)
{
    if (is_view_special (cpanel->view_type))
	return;
    
    unselect_item (cpanel);
    cpanel->selected = cpanel->top_file + (ITEMS(cpanel)/2);
    if (cpanel->selected >= cpanel->count)
	cpanel->selected = cpanel->count - 1;
    select_item (cpanel);
    panel_refresh (cpanel);
}

static void goto_bottom_file (void)
{
    if (is_view_special (cpanel->view_type))
	return;
    
    unselect_item (cpanel);
    cpanel->selected = cpanel->top_file + ITEMS(cpanel)-1;
    if (cpanel->selected >= cpanel->count)
	cpanel->selected = cpanel->count - 1;
    select_item (cpanel);
    panel_refresh (cpanel);
}

static void copy_current_pathname (void)
{
    if (cpanel->view_type == view_quick)
	return;
    if (!command_prompt)
	return;

    stuff (cmdline, cpanel->cwd, 0);
}

static void copy_other_pathname (void)
{
    if (is_view_special (opanel->view_type))
	return;
    if (!command_prompt)
	return;

    stuff (cmdline, opanel->cwd, 0);
}

/* Inserts the selected file name into the input line */
static void copy_prog_name (void)
{
    if (is_view_special (cpanel->view_type))
	return;
    if (!command_prompt)
	return;
    
    if (cpanel->view_type == view_tree)
	copy_current_pathname ();
    else
	stuff (cmdline, selection->fname, 1);
}

/* Inserts the selected file name in the other panel into the input line */
static void copy_other_prog_name (void)
{
    char *k = opanel->dir.list [opanel->selected].fname;

    if (is_view_special (opanel->view_type))
	return;
    if (!command_prompt)
	return;

    if (cpanel->view_type == view_tree)
	copy_other_pathname ();
    else
	stuff (cmdline, k, 1);
}

static void copy_tagged (Panel *panel)
{
    int i;

    if (panel->view_type == view_tree
	|| panel->view_type == view_info
	|| panel->view_type == view_quick)
	return;
    if (!command_prompt)
	return;
    input_disable_update (cmdline);
    if (panel->marked){
	for (i = 0; i < panel->count; i++)
	    if (panel->dir.list [i].f.marked)
		stuff (cmdline, panel->dir.list [i].fname, 1);
    } else 
	stuff (cmdline, panel->dir.list [panel->selected].fname, 1);
    input_enable_update (cmdline);
}
    
static void copy_current_tagged (void)
{
    copy_tagged (cpanel);
}

static void copy_other_tagged (void)
{
    copy_tagged (opanel);
}

static void mount_this_dir (void)
{
    try_auto_mount (selection->fname);
}

static void umount_this_dir (void)
{
    try_auto_umount (selection->fname);
}

static void suspend_cmd (void)
{
    struct sigaction sigtstp_action;
    
    save_cwds_stat ();
    pre_exec ();

    if (console_flag && !use_subshell)
	restore_console ();
    
    /* Make sure that the SIGTSTP below will suspend us directly,
       without calling ncurses' SIGTSTP handler; we *don't* want
       ncurses to redraw the screen immediately after the SIGCONT */
    sigaction (SIGTSTP, &startup_handler, &sigtstp_action);
    
    kill (getpid (), SIGTSTP);

    /* Restore previous SIGTSTP action */
    sigaction (SIGTSTP, &sigtstp_action, NULL);
    
    if (console_flag && !use_subshell)
	handle_console (CONSOLE_SAVE);
    
    post_exec ();
    update_panels (UP_OPTIMIZE, UP_KEEPSEL, 0);
    refresh_screen (NULL);
}

static void quote_next (void)
{
    quote = 1;
}

static key_map ctl_x_map [] = {
    XCTRL('c'),   (callfn) quit_cmd,
    'r',          mask_ren_cmd,
    'd',          compare_dirs,
    'p',          copy_current_pathname,
    XCTRL('p'),   copy_other_pathname,
    't',          copy_current_tagged,
    XCTRL('t'),   copy_other_tagged,
    'c',          chmod_cmd,
    'o',          chown_cmd,
    'l',          link_cmd,
    's',          symlink_cmd,
    'a',          toggle_auto_mount,
    'm',          mount_this_dir,
    'u',          umount_this_dir,
    'i',          info_cmd_no_menu,
    'h',          add2hotlist_cmd,
    0, 0
};

static void ctl_x_cmd (int ignore)
{
    int i;
    int key = mi_getch ();

    for (i = 0; i < ctl_x_map [i].key_code; i++){
	if (key == ctl_x_map [i].key_code){
	    (*ctl_x_map [i].fn)(key);
	    break;
	}
    }
}

static key_map default_map [] = {
    KEY_F(12),  dump_marked,
    KEY_F(19),  menu_last_selected_cmd,
    '+',        select_cmd,
    '\\',       unselect_cmd,
    '\t',  	change_panel,
    '\n',       enter,		/* Enter */
    
    KEY_DOWN,   move_down,
    KEY_UP, 	move_up,
    KEY_LEFT,   move_left,
    KEY_RIGHT,  move_right,
    KEY_IC,     mark_file,
    KEY_HOME,	move_home,
    KEY_C1,     move_end,
    KEY_END,    move_end,
    KEY_A1,     move_home,
    KEY_NPAGE,  next_page,
    KEY_PPAGE,  prev_page,
    XCTRL('l'), repaint_screen_cmd, /* C-l */

    /* Copy useful information to the command line */
    ALT('\n'),  copy_prog_name,
    ALT('\t'),  copy_other_prog_name,
    ALT('a'),   copy_current_pathname,
    ALT('A'),   copy_other_pathname,

    /* To quickly move in the panel */
    ALT('g'),   goto_top_file,
    ALT('h'),   goto_middle_file,
    ALT('r'),   goto_middle_file, /* M-r like emacs */
    ALT('j'),   goto_bottom_file,

    /* To access the directory hotlist */
    XCTRL('\\'), quick_chdir,
    
    /* Emacs-like bindings */
    XCTRL('v'), next_page,      /* C-v like emacs */
    ALT('v'),   prev_page,	/* M-v like emacs */
    XCTRL('p'), move_up,	/* C-p like emacs */
    XCTRL('n'), move_down,	/* C-n like emacs */
    XCTRL('q'), quote_next,	/* C-q like emacs */
    XCTRL('s'), start_search,	/* C-s like emacs */
    ALT('s'),   start_search,	/* M-s not like emacs */

    /* Suspend */
    XCTRL('z'), suspend_cmd,

    XCTRL('t'), mark_file,

    /* Panel refresh */
    XCTRL('r'), reread_cmd,

    /* Swap panels */
    XCTRL('u'), swap_cmd,

    /* View output */
    XCTRL('o'), view_other_cmd,
    
    /* Control-X keybindings */
    XCTRL('x'), ctl_x_cmd,
    
    /* Default key handling */
    0,          default_key
};

void init_labels (void)
{
    fkeys = push_fkey (keybar_visible);
    define_label (fkeys, 1, "Help", help_cmd);
    define_label (fkeys, 2, "Menu", key_f2_cmd);
    define_label (fkeys, 3, "View", key_f3_cmd);
    define_label (fkeys, 4, "Edit", key_f4_cmd);
    define_label (fkeys, 5, "Copy", key_f5_cmd);
    define_label (fkeys, 6, "RenMov", key_f6_cmd);
    define_label (fkeys, 7, "Mkdir", key_f7_cmd);
    define_label (fkeys, 8, "Delete", key_f8_cmd);
    define_label (fkeys, 9, "PullDn", menu_cmd);
    define_label_quit (fkeys, 10, "Quit", quit_cmd);
    if (cpanel && opanel)
	change_labels ();

    /* workaround ncurses 1.8.5 bug */
    wmove (fkeys, 0, 0);
    waddch (fkeys, '1');
    if (keybar_visible && fkeys)
	wrefresh (fkeys);
}

static void save_setup_cmd (void)
{
    save_setup ();
    sync_profiles ();
    message (0, " Setup ", " Setup saved to ~/.mc.ini ");
}

static int do_nc (void)
{
    int key;
    int i;

    push_refresh (refresh_screen, 0, REFRESH_COVERS_ALL);

    if (COLS < 70 || LINES < 22){
	endwin ();
	fprintf (stderr, "Screen too small: you need at least 70x22\n");
	return 0;
    }

    /* Call all the inits */
    init_uid_gid_cache ();

    if (baudrate () < 9600 || force_slow)
	verbose = 0;
    if (use_mouse_p)
	init_mouse ();
    init_key ();
    load_tree ();
    init_panels ();
    init_menu ();

#if VERSION_3
#ifdef SIGWINCH
    signal (SIGWINCH, flag_winch);
#endif
#endif
    
    if (auto_menu)
	user_menu_cmd ();

    /* Main program loop */
    for (quit = 0; !quit;){
	remove_dash ();
	/* we need to change the layout here, since it can't be done   */
	/* on an inner loop of the program because of the setup frames */
	if (layout_do_change){
	    layout_change ();
	    refresh_screen (0);
	}
#ifdef VERSION_3
	if (winch_flag)
	    change_screen_size ();
#endif
	
	if (was_searching != searching){
	    display_mini_info (cpanel);
	    wrefresh (cpanel->win_file);
	}

	/* This is needed for ncurses 1.8.7, ugh :-( */
	/* Also SysV curses benefits from this */
	if (command_prompt){
	    leaveok (cmdline_win, FALSE);
	    wrefresh (cmdline_win);
	}
	
	key = mi_getch ();

	if (cpanel->view_type == view_quick && quick_view_key (key))
	    continue;
	was_searching = searching;
	if (quote){
	    default_key (key);
	    quote = 0;
	    continue;
	}
	/* This function should set the searching variable to 0 before */
	/* invoking the appropiate routine */
	if (check_fkeys (key)){
	    searching = 0;
	    continue;
	}
	else {
	    for (i = 0; default_map [i].key_code; i++){
		if (key == default_map [i].key_code){
		    searching = 0;
		    (*default_map [i].fn)(key);
		    break;
		}
	    }
	}

	if (default_map [i].key_code || key == -1)
	    continue;

	/* Default key handling */
	(*default_map [i].fn)(key);
    }

    /* Shutdown the program */
    pop_frame ();
    destroy_input (cmdline, IN_NORMAL);
    done_menu ();
    
    /* Setup shutdown
     *
     * We sync the profiles since the hotlist may have changed, while
     * we only change the setup data if we have the auto save feature set
     */
    if (auto_save_setup)
	save_setup ();
    sync_profiles ();
    done_setup ();

    /* Panels shutdown */
    save_tree ();
    destroy_tree ();
    done_panels ();

    free_profiles ();

    if (quit != SUBSHELL_EXIT)
	clr_scr ();
    
    keypad (cmdline_win, FALSE);
    keypad (stdscr, FALSE);
    reset_shell_mode ();
    noraw ();
    if (use_mouse_p)
	shut_mouse ();
    pop_refresh ();
    return 1;
}

static void version (void)
{
    printf ("Midnight Commander %s, with mouse support on xterm%s,\n%s%s.\n",
	    VERSION, status_mouse_support ? " and the Linux console" : "",
	    status_using_ncurses ? "using the ncurses library"
				 : "using some unknown curses library",
	    status_using_old_tools ? " and compiled with old tools" : "");
}

#define CONTROL_FILE "/tmp/mc.%d.control"
char control_file [sizeof (CONTROL_FILE) + 8];
char *shell;

static void handle_args (int argc, char *argv [])
{
    extern int optind;
    extern char *optarg;
    char   *this_dir = 0;
    int    c;
    char   *termvalue;
    
    while ((c = getopt (argc, argv, "VbsdPcfC:hxu")) != -1){
	switch (c){
	case 'V':
	    version ();
	    exit (0);
	    
	case 'b':
	    disable_colors = 1;
	    break;

	case 'c':
	    disable_colors = 0;
	    break;

	case 'f':
	    printf (
		    "Help file:            " LIBDIR "mc.hlp" "\n"
	            "Global defaults file: " LIBDIR "mc.ini" "\n"
		    "Global user menu:     " LIBDIR "mc.menu" "\n"
		    "Extensions file:      " LIBDIR "mc.ext" "\n");
	    exit (1);

	case 'P':
	    print_last_wd = 1;
	    break;
	    
	case 's':
	    force_slow = 1;	/* Currently it does nothing */
	    break;

	case 'd':
	    use_mouse_p = NO_MOUSE;
	    break;

	case 'C':
	    configure_colors_string (optarg);
	    break;

	case 'x':
	    force_xterm = 1;
	    break;

	case 'u':
	    #ifdef HAVE_SUBSHELL_SUPPORT
		use_subshell = 0;
	    #endif
	    break;
	    
	case '?':
	case 'h':
	default:
	    version ();
	    printf ("Usage is:\n"
		    "mc [-bCcdfhPsV?] [other_panel_dir]\n"
		    "-b Force black and white\n"
		    "-C <keyword>=<fore>,<back> Color usage\n"
		    "   keywords: normal, selected, marked, markselect, "
                        "errors,menu, reverse,\n"
		    "   dnormal, dfocus, dhotnormal, dhotfocus\n"
		    "   colors: black, red, green, yellow, blue, magenta, "
		        "cyan, white\n"
		    "-c Force color, only available if terminfo allows it\n"
		    "-d Disable mouse support\n"
		    "-f Print configured paths\n"
		    "-P At exit, print last working directory\n"
		    "-s Disables verbose operation (for slow terminals)\n"
		    "-V Print program version\n"
		    "-u Disable the concurrent subshell mode\n"
                    "-x Force xterm mouse support and screen save/restore\n"
		    );
	    exit (1);
	}
    }

    termvalue = getenv ("TERM");
    if (!termvalue){
	fprintf (stderr, "The TERM environment variable is unset!\n");
	termvalue = "";
    }
    if (force_xterm || strncmp (termvalue, "xterm", 5) == 0){
	use_mouse_p = XTERM_MOUSE;
	xterm_flag = 1;
    }

    sprintf (control_file, CONTROL_FILE, getpid ());
    my_putenv ("MC_CONTROL_FILE", control_file);

    shell = getenv ("SHELL");
    if (!shell || !*shell)
	shell = strdup (getpwuid (geteuid ()) -> pw_shell);
    if (!shell || !*shell)
	shell = "/bin/sh";

    /* sets the current dir and the other dir */
    for (; optind < argc; optind++){
	if (this_dir){
	    other_dir = strdup (argv [optind]);
	} else {
	    this_dir = argv [optind];
	    get_current_wd (right_panel.cwd, sizeof (right_panel.cwd)-2);
	    other_dir = strdup (right_panel.cwd);
	}
    }
    if (this_dir)
	chdir (this_dir);
}

static void sigchld_handler_no_subshell (int sig)
{
    if (!console_flag)
	return;
    /* FIXME: if it were true that after the call to handle_console(..INIT)
       the value of console_flag never changed, we could simply not install
       this handler at all if (!console_flag && !use_subshell). */

    if (wait (NULL) == cons_saver_pid){
	handle_console (CONSOLE_DONE);
	handle_console (CONSOLE_INIT);
    }
}

void init_sigchld (void)
{
    struct sigaction sigchld_action;

    sigchld_action.sa_handler =
    #ifdef HAVE_SUBSHELL_SUPPORT
	use_subshell ? sigchld_handler :
    #endif
	sigchld_handler_no_subshell;

    sigemptyset (&sigchld_action.sa_mask);

    #ifdef SA_RESTART
        sigchld_action.sa_flags = SA_RESTART;
    #else
        sigchld_action.sa_flags = 0;
    #endif
    sigaction (SIGCHLD, &sigchld_action, NULL);
}	

int main (int argc, char *argv [])
{
    int stdout_fd = 0;

    /* Backward compatibility: Gives up privileges in case someone
       installed the mc as setuid */
    setuid (getuid ());

    handle_args (argc, argv);
    
    /* Used to report the last working directory at program end */
    if (print_last_wd){
	stdout_fd = dup (1);
	close (1);
	open ("/dev/tty", O_RDWR);
    }

    /* Must be done before installing the SIGCHLD handler [[FIXME]] */
    handle_console (CONSOLE_INIT);

#ifdef HAVE_SUBSHELL_SUPPORT
    subshell_get_console_attributes ();
#endif
    
    /* Install the SIGCHLD handler; must be done before init_subshell() */
    if (!use_subshell)
	init_sigchld ();
    
    /* This variable is used by the subshell */
    home_dir = getenv ("HOME");
    home_dir = home_dir ? home_dir : "/";

    /* We need this, since ncurses endwin () doesn't restore the signals */
    save_stop_handler ();

    load_setup ();

    /* Must be done before init_subshell, to set up the terminal size: */
    initscr ();

#ifdef HAVE_SUBSHELL_SUPPORT
    /* Done here to ensure that the subshell doesn't  */
    /* inherit the file descriptors opened below, etc */
 
    if (use_subshell){
	init_subshell ();  
	if (!use_subshell){
	    fputs ("Couldn't start up the subshell; using the old "
		   "command execution method.\nPress enter... ", stderr);
	    getchar ();
	} else
	    add_select_channel (subshell_pty, load_prompt);
    }
#endif

    /* Must be done after init_subshell, as it changes the terminal modes: */
    init_curses ();

    /* Also done after init_subshell, to save any shell init file messages */
    if (console_flag)
	handle_console (CONSOLE_SAVE);

#ifdef HAVE_SUBSHELL_SUPPORT
    if (use_subshell){
	prompt = strip_ctrl_codes (subshell_prompt);
	if (!prompt)
	    prompt = "";
    } else
#endif
	prompt = (geteuid () == 0) ? "# " : "$ ";

    init_my_statfs ();
    meta (stdscr, eight_bit_clean);
    cmdline = create_input (strlen (prompt), 0, cmdline_win, 0,
			    COLS-strlen (prompt), "");

    push_frame (0, 0, 0);
    create_panels ();
    if (do_nc ()){
	destroy_panels ();
	endwin ();
	pop_frame ();

	if (console_flag && quit != SUBSHELL_EXIT)
	    restore_console ();
    }

    signal (SIGCHLD, SIG_DFL);  /* Disable the SIGCHLD handler */

    if (console_flag)
	handle_console (CONSOLE_DONE);

    putchar ('\r');  /* Hack to make shell's prompt start at left of screen */

    if (print_last_wd)
	write (stdout_fd, cpanel->cwd, strlen (cpanel->cwd));

    mad_finalize (__FILE__, __LINE__);
    return 0;
}
