;;; t-mouse.el --- mouse support within the text terminal

;;; Copyright (C) 1994 Alessandro Rubini <rubini@ipvvis.unipv.it>

;; Maintainer: Alessandro Rubini
;; Keywords: mouse linux

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.

;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.

;;; You should have received a copy of the GNU General Public License
;;; along with GNU Emacs; see the file COPYING.  If not, write to
;;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;; This package provides access to mouse event as reported by the
;; gpm-Linux package. It uses the program "mev" to get mouse events.
;; It tries to reproduce the functionality offered by emacs under X.
;; The "gpm" server runs under Linux, so this package is rather
;; Linux-dependent.

;; Tested only under emacs-19.25

;;; Code:

(defvar t-mouse-process nil 
  "Embeds the process which passes mouse events to emacs. It is used
by 't-mouse.el")


;; get the number of the current virtual console
(defun t-mouse-tty ()
  (let (tty
	(buffer (generate-new-buffer "*t-mouse*")))
    (call-process "ps" nil buffer nil
		  "-h" (format "%s" (emacs-pid)))
    (save-excursion
      (set-buffer buffer)
      (goto-char 0)
      (or
       (re-search-forward "p \\([0-9a-f]\\)" nil t)
       (re-search-forward "v0\\([0-9a-f]\\)"))
      (setq tty (buffer-substring (match-beginning 1) (match-end 1))))
    (kill-buffer buffer)
    tty
))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; this is the firing function
(defun t-mouse-run ()
  (interactive)
  (let ((process-connection-type nil))
    ;; remove any existing mouse process
    (and (boundp 't-mouse-process)
	 (processp t-mouse-process)
	 (delete-process t-mouse-process))
    (setq t-mouse-process 
	  ;; use only press events
	  (start-process "t-mouse" nil "mev" "-E" "-C" (t-mouse-tty) "-M" "leftAlt"
			 "-e" "press,drag,release,hard" "-d" "move"))
    (set-process-filter t-mouse-process 't-mouse-process-filter)
    (process-kill-without-query t-mouse-process)
    t-mouse-process
))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; this is the filter
(defun t-mouse-process-filter (proc string)
  (let* ((event (t-mouse-make-event string))
	 (cmd (lookup-key t-mouse-keymap (car event))))
    ;;(insert (format "%s ===>%s" (car event) cmd))
    (and cmd (funcall cmd (nth 1 event)))
))



;;; This fun is partly Copyright (C) 1994 Per Abrahamsen <abraham@iesd.auc.dk>
(defun t-mouse-make-event (string)
  (let* ((string (car (read-from-string string)))
	 (mouse (vector (car string)))
	 (point (nth 1 string))					
	 (x (car point))
	 (y (cdr point))
	 (window (window-at x y))
	 (where (coordinates-in-window-p point window))
	 (pos (if (consp where)
		  (save-excursion
		    (select-window window)
		    (goto-char (window-start window))
		    (move-to-window-line  (cdr where))
		    (move-to-column (+ (car where) (current-column)
				       (max 0 (1- (window-hscroll)))))
		    (point))
		where)))
    ;; manage the scroll-bar

    ;; doesn't work with horizontally-split windows
    ;;(if (or t-mouse-scroll (>= x (1- (window-width window))))
    ;;    (setq pos 'scroll-bar))

 

    ;; does work with horizontally-split windows
    (if (or
	 t-mouse-scroll
	 (eq pos 'vertical-line)
	 (and (consp where) (>= (car where) (1- (window-width window)))))
        (setq pos 'scroll-bar))
	
		
    ;; make the event
    (or (integerp pos) (setq mouse (vector pos (car string))))
    (list mouse (list window pos point))
))


;; Restore normal mouse behaviour outside Emacs.
(defun t-mouse-suspend ()
  (process-send-string t-mouse-process "push -e 0 -d any\n"))
(defun t-mouse-resume ()
  (process-send-string t-mouse-process "pop\n"))

(add-hook 'suspend-hook 't-mouse-suspend)
(add-hook 'suspend-resume-hook 't-mouse-resume)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; make your keymap


(defvar t-mouse-keymap nil "Keymap used by t-mouse")
(defvar t-mouse-scroll nil "The last position when scrolling")
(defvar t-mouse-word-syntax "w_" "The syntax set to select with double-click")

(setq t-mouse-keymap
      '(keymap
	(down-mouse-1 .      tm-goto)
	(drag-mouse-1 .      tm-drag)
	(mouse-1      .      tm-regn)
	(down-mouse-2 .      tm-yank)
	(down-mouse-3 .      tm-copy)
	(double-mouse-1 .    tm-word)
	(triple-mouse-1 .    tm-line)
	(double-mouse-3 .    tm-kill)
	(M-down-mouse-1 .    tm-scroll-up)
	(M-down-mouse-2 .    tm-scroll-jump)
	(M-down-mouse-3 .    tm-scroll-down)
	(mode-line keymap
		   (mouse-1 . tm-win-select)
		   (mouse-2 . tm-win-single)
		   (mouse-3 . tm-win-delete))
	(scroll-bar keymap
		    (down-mouse-1   . tm-scroll-up)
		    (down-mouse-3   . tm-scroll-down)
		    (down-mouse-2   . tm-scroll-jump)
		    (drag-mouse-2   . tm-scroll-drag)
		    (M-drag-mouse-2 . tm-scroll-drag)
		    (mouse-1        . tm-scroll-done)
		    (double-mouse-1 . tm-scroll-done)
		    (triple-mouse-1 . tm-scroll-done)
		    (mouse-2        . tm-scroll-done)
		    (double-mouse-2 . tm-scroll-done)
		    (triple-mouse-2 . tm-scroll-done)
		    (mouse-3        . tm-scroll-done)
		    (double-mouse-3 . tm-scroll-done)
		    (triple-mouse-3 . tm-scroll-done)
		    (M-mouse-1        . tm-scroll-done)
		    (M-double-mouse-1 . tm-scroll-done)
		    (M-triple-mouse-1 . tm-scroll-done)
		    (M-mouse-2        . tm-scroll-done)
		    (M-double-mouse-2 . tm-scroll-done)
		    (M-triple-mouse-2 . tm-scroll-done)
		    (M-mouse-3        . tm-scroll-done)
		    (M-double-mouse-3 . tm-scroll-done)
		    (M-triple-mouse-3 . tm-scroll-done))))

;;; first, the scrollbar

(defun tm-scroll-up (event)
  ;;(message "up")
  (setq t-mouse-scroll t)
  (scroll-up (cdr (nth 2 event)))
)

(defun tm-scroll-down (event)
  ;;(message "down")
  (setq t-mouse-scroll t)
  (scroll-down (cdr (nth 2 event)))
)

(defun tm-scroll-jump (event)
  ;;(message "jump")
  (let* ((lines (+ (count-lines (point-min) (point-max)) (window-height)))
	 (target (/ (* lines (cdr (nth 2 event))) (window-height)))
	 (step (/ lines (window-height))))
;;    (goto-line target)
;;    (recenter)
    (setq t-mouse-scroll (cons step  (cdr (nth 2 event))))
))

(defun tm-scroll-drag (event)
  ;;(message "drag")
  (let* ((step (car t-mouse-scroll))
	 (steplet (/ (* step (car (nth 2 event))) (window-width)))
	 (y (cdr (nth 2 event))))
    (scroll-down (* steplet (- (cdr t-mouse-scroll) y)))
    (setq t-mouse-scroll (cons step y))
))

(defun tm-scroll-done (event)
  ;;(message "done")
  (setq t-mouse-scroll nil))

;;; and then conventional events

(defun tm-goto (event)
  (let ((current (nth 1 event)))
    (goto-char current)
    (push-mark (point) 'nomsg 'activate)))

(defun tm-drag (event)
  (goto-char (nth 1 event)))

(defun tm-regn (event)
  (let ((current (nth 1 event)))
    (or
     (equal current (mark))
     (progn
       (goto-char (mark)) 
       (sit-for 0.5) 
       (goto-char current) 
       (tm-copy event)))))

(defun tm-word (event)
  (let* ((current (nth 1 event))
	(previous (mark))
	(backward (> previous current)))
    (pop-mark)
    ;; update previous and set mark
    (goto-char previous)
    (if backward (skip-syntax-forward  t-mouse-word-syntax)
      (skip-syntax-backward t-mouse-word-syntax))
    (setq previous (point))
    (push-mark (point) 'nomsg 'activate)
    (sit-for 0.5)
    ;; update current
    (goto-char current)
    (if backward (skip-syntax-backward t-mouse-word-syntax) 
      (skip-syntax-forward  t-mouse-word-syntax))
    (setq current (point))
    (kill-ring-save (mark) current)))
  
(defun tm-line (event)
  (let* ((current (nth 1 event))
	(previous (mark))
	(backward (> previous current)))
    (tm-drag event)
    (pop-mark)
    ;; update previous and set mark
    (goto-char previous)
    (if backward (end-of-line) (beginning-of-line))
    (setq previous (point))
    (push-mark (point) 'nomsg 'activate)
    (sit-for 0.5)
    ;; update current
    (goto-char current)
    (if backward (beginning-of-line) (end-of-line))
    (setq current (point))
    (kill-ring-save (mark) current)))


(defun tm-yank (event)
  (let ((current (nth 1 event)))
    (or
     mouse-yank-at-point
     (goto-char current))
    (yank)))

(defun tm-copy (event)
  (let ((current (nth 1 event)))
    (kill-ring-save (mark) current)
    (goto-char current)))

(defun tm-kill (event)
  (let ((current (nth 1 event)))
    (kill-region (mark) current)))

(defun tm-win-select (event)
  (select-window (car event)))

(defun tm-win-single (event)
  (select-window (car event))
  (delete-other-windows))

(defun tm-win-delete (event)
  (select-window (car event))
  (if (one-window-p 'no-mini)
      (kill-buffer (window-buffer (car event)))
    (delete-window)))


(t-mouse-run)
(provide 't-mouse)

;(setq debug-on-error t)

;;; t-mouse.el ends here






