/**
 * @license
 * Copyright 2023 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { __decorate, __metadata } from '../../../../tslib/tslib.js';
import '../../elevation/elevation.js';
import { html, LitElement } from '../../../../lit/index.js';
import { property, query, state } from '../../../../lit/decorators.js';
import { classMap } from '../../../../lit/directives/class-map.js';
import { createThrottle, msFromTimeCSSValue } from '../../motion/animation.js';
// This is required for decorators.
// tslint:disable:no-new-decorators
/**
 * Default close action.
 */
export const CLOSE_ACTION = 'close';
const OPENING_TRANSITION_PROP = '--_opening-transition-duration';
const CLOSING_TRANSITION_PROP = '--_closing-transition-duration';
/**
 * A dialog component.
 */
export class Dialog extends LitElement {
    constructor() {
        super(...arguments);
        /**
         * Opens the dialog when set to `true` and closes it when set to `false`.
         */
        this.open = false;
        /**
         * Setting fullscreen displays the dialog fullscreen on small screens.
         * This can be customized via the `fullscreenBreakpoint` property.
         * When showing fullscreen, the header will take up less vertical space, and
         * the dialog will have a `showing-fullscreen`attribute, allowing content to
         * be styled in this state.
         *
         * Dialogs can be sized by setting:
         *
         * * --md-dialog-container-min-block-size
         * * --md-dialog-container-max-block-size
         * * --md-dialog-container-min-inline-size
         * * --md-dialog-container-max-inline-size
         *
         * These are typically configured via media queries and are independent of the
         * fullscreen setting.
         */
        this.fullscreen = false;
        /**
         * A media query string specifying the breakpoint at which the dialog
         * should be shown fullscreen. Note, this only applies when the `fullscreen`
         * property is set.
         *
         * By default, the dialog is shown fullscreen on screens less than 600px wide
         * or 400px tall.
         */
        this.fullscreenBreakpoint = '(max-width: 600px), (max-height: 400px)';
        /**
         * Hides the dialog footer, making any content slotted into the footer
         * inaccessible.
         */
        this.footerHidden = false;
        /**
         * Renders footer content in a vertically stacked alignment rather than the
         * normal horizontal alignment.
         */
        this.stacked = false;
        /**
         * When the dialog is closed it disptaches `closing` and `closed` events.
         * These events have an action property which has a default value of
         * the value of this property. Specific actions have explicit values but when
         * a value is not specified, the default is used. For example, clicking the
         * scrim, pressing escape, or clicking a button with an action attribute set
         * produce an explicit action.
         *
         * Defaults to `close`.
         */
        this.defaultAction = CLOSE_ACTION;
        /**
         * The name of an attribute which can be placed on any element slotted into
         * the dialog. If an element has an action attribute set, clicking it will
         * close the dialog and the `closing` and `closed` events dispatched will
         * have their action property set the the value of this attribute on the
         * clicked element.The default valus is `dialogAction`. For example,
         *
         *   <md-dialog>
         *    Content
         *     <md-filled-button slot="footer"dialogAction="buy">
         *       Buy
         *     </md-filled-button>
         *   </md-dialog>
         */
        this.actionAttribute = 'dialogAction';
        /**
         * When the dialog is opened, it will focus the first element which has
         * an attribute name matching this property. The default value is
         * `dialogFocus`. For example:
         *
         *  <md-dialog>
         *    <md-filled-text-field
         *      label="Enter some text"
         *      dialogFocus
         *    >
         *    </md-filled-text-field>
         *  </md-dialog>
         */
        this.focusAttribute = 'dialogFocus';
        /**
         * Clicking on the scrim surrounding the dialog closes the dialog.
         * The `closing` and `closed` events this produces have an `action` property
         * which is the value of this property and defaults to `close`.
         */
        this.scrimClickAction = CLOSE_ACTION;
        /**
         * Pressing the `escape` key while the dialog is open closes the dialog.
         * The `closing` and `closed` events this produces have an `action` property
         * which is the value of this property and defaults to `close`.
         */
        this.escapeKeyAction = CLOSE_ACTION;
        /**
         * When opened, the dialog is displayed modeless or non-modal. This
         * allows users to interact with content outside the dialog without
         * closing the dialog and does not display the scrim around the dialog.
         */
        this.modeless = false;
        /**
         * Set to make the dialog position draggable.
         */
        this.draggable = false;
        this.throttle = createThrottle();
        /**
         * Private properties that reflect for styling manually in `updated`.
         */
        this.showingFullscreen = false;
        this.showingOpen = false;
        this.opening = false;
        this.closing = false;
        /**
         * Transition kind. Supported options include: grow, shrink, grow-down,
         * grow-up, grow-left, and grow-right.
         *
         * Defaults to grow-down.
         */
        this.transition = 'grow-down';
        this.dragging = false;
        this.dragMargin = 8;
        this.fullscreenQueryListener = undefined;
    }
    static setDocumentScrollingDisabled(disabled = false) {
        let { preventedScrollingCount, scrollbarTester } = Dialog;
        Dialog.preventedScrollingCount = preventedScrollingCount =
            Math.max(preventedScrollingCount + (Number(disabled) || -1), 0);
        const shouldPrevent = Boolean(preventedScrollingCount);
        const { style } = document.body;
        if (shouldPrevent && style.overflow === 'hidden') {
            return;
        }
        if (scrollbarTester === undefined) {
            Dialog.scrollbarTester = scrollbarTester = document.createElement('div');
            scrollbarTester.style.cssText =
                `position: absolute; height: 0; width: 100%; visibility: hidden; pointer-events: none;`;
        }
        // Appends an element to see if its offsetWidth changes when overflow is
        // altered. If it does, then add end inline padding to restore the width to
        // avoid a visible layout shift.
        document.body.append(scrollbarTester);
        const { offsetWidth } = scrollbarTester;
        style.overflow = shouldPrevent ? 'hidden' : '';
        const scrollbarWidth = scrollbarTester.offsetWidth - offsetWidth;
        scrollbarTester.remove();
        style.paddingInlineEnd = shouldPrevent ? `${scrollbarWidth}px` : '';
    }
    /**
     * Opens and shows the dialog. This is equivalent to setting the `open`
     * property to true.
     */
    show() {
        this.open = true;
    }
    /**
     * Closes the dialog. This is equivalent to setting the `open`
     * property to false.
     */
    close(action = '') {
        this.currentAction = action;
        this.open = false;
    }
    /**
     * Opens and shows the dialog if it is closed; otherwise closes it.
     */
    toggleShow() {
        if (this.open) {
            this.close(this.currentAction);
        }
        else {
            this.show();
        }
    }
    getContentScrollInfo() {
        if (!this.hasUpdated) {
            return { isScrollable: false, isAtScrollTop: true, isAtScrollBottom: true };
        }
        const { scrollTop, scrollHeight, offsetHeight, clientHeight } = this.contentElement;
        return {
            isScrollable: scrollHeight > offsetHeight,
            isAtScrollTop: scrollTop === 0,
            isAtScrollBottom: Math.abs(Math.round(scrollHeight - scrollTop) - clientHeight) <= 2
        };
    }
    render() {
        const { isScrollable, isAtScrollTop, isAtScrollBottom } = this.getContentScrollInfo();
        return html `
    <dialog
      @close=${this.handleDialogDismiss}
      @cancel=${this.handleDialogDismiss}
      @click=${this.handleDialogClick}
      class="dialog ${classMap({
            'stacked': this.stacked,
            'scrollable': isScrollable,
            'scroll-divider-header': !isAtScrollTop,
            'scroll-divider-footer': !isAtScrollBottom,
            'footerHidden': this.footerHidden
        })}"
      aria-labelledby="header"
      aria-describedby="content"
    >
      <div class="container ${classMap({
            'dragging': this.dragging
        })}"
        @pointermove=${this.handlePointerMove}
        @pointerup=${this.handleDragEnd}
      >
        <header class="header">
          <slot name="header">
            <slot name="headline-prefix"></slot>
            <slot name="headline"></slot>
            <slot name="headline-suffix"></slot>
          </slot>
        </header>
        <section class="content" @scroll=${this.handleContentScroll}>
          <slot></slot>
        </section>
        <footer class="footer">
          <slot name="footer"></slot>
        </footer>
        <md-elevation surface></md-elevation>
      </div>
    </dialog>`;
    }
    willUpdate(changed) {
        if (changed.has('open')) {
            this.opening = this.open;
            // only closing if was opened previously...
            this.closing = !this.open && changed.get('open');
        }
        if (changed.has('fullscreen') || changed.has('fullscreenBreakpoint')) {
            this.updateFullscreen();
        }
    }
    firstUpdated() {
        // Update when content size changes to show/hide scroll dividers.
        new ResizeObserver(() => {
            if (this.showingOpen) {
                this.requestUpdate();
            }
        }).observe(this.contentElement);
    }
    updated(changed) {
        if (changed.get('draggable') && !this.draggable) {
            this.style.removeProperty('--_container-drag-inline-start');
            this.style.removeProperty('--_container-drag-block-start');
        }
        // Reflect internal state to facilitate styling.
        this.reflectStateProp(changed, 'opening', this.opening);
        this.reflectStateProp(changed, 'closing', this.closing);
        this.reflectStateProp(changed, 'showingFullscreen', this.showingFullscreen, 'showing-fullscreen');
        this.reflectStateProp(changed, 'showingOpen', this.showingOpen, 'showing-open');
        if (!changed.has('open')) {
            return;
        }
        // prevent body scrolling early only when opening to avoid layout
        // shift when closing.
        if (!this.modeless && this.open) {
            this.constructor
                .setDocumentScrollingDisabled(this.open);
        }
        if (this.open) {
            this.contentElement.scrollTop = 0;
            if (this.modeless) {
                this.dialogElement.show();
            }
            else {
                // Note, native focus handling fails when focused element is in an
                // overflow: auto container.
                this.dialogElement.showModal();
            }
        }
        // Avoids dispatching initial state.
        const shouldDispatchAction = changed.get('open') !== undefined;
        this.performTransition(shouldDispatchAction);
    }
    /**
     * Internal state is reflected here as attributes to effect styling. This
     * could be done via internal classes, but it's published on the host
     * to facilitate the (currently undocumented) possibility of customizing
     * styling of user content based on these states.
     * Note, in the future this could be done with `:state(...)` when browser
     * support improves.
     */
    reflectStateProp(changed, key, value, attribute) {
        attribute ?? (attribute = key);
        if (!changed.has(key)) {
            return;
        }
        if (value) {
            this.setAttribute(attribute, '');
        }
        else {
            this.removeAttribute(attribute);
        }
    }
    async performTransition(shouldDispatchAction) {
        // TODO: pause here only to avoid a double update warning.
        await this.updateComplete;
        this.showingOpen = this.open;
        if (shouldDispatchAction) {
            this.dispatchActionEvent(this.open ? 'opening' : 'closing');
        }
        // Compute desired transition duration.
        const duration = msFromTimeCSSValue(getComputedStyle(this).getPropertyValue(this.open ? OPENING_TRANSITION_PROP : CLOSING_TRANSITION_PROP));
        await new Promise(r => {
            setTimeout(r, duration);
        });
        this.opening = false;
        this.closing = false;
        if (!this.open && this.dialogElement.open) {
            // Closing the dialog triggers an asynchronous `close` event.
            // It's important to wait for this event to fire since it changes the
            // state of `open` to false.
            // Without waiting, this element's `closed` event can be called before
            // the dialog's `close` event, which is problematic since the user
            // can set `open` in the `closed` event.
            // The timing of the event appears to vary via browser and does *not*
            // seem to resolve by "task" timing; therefore an explicit promise is
            // used.
            const closedPromise = new Promise(resolve => {
                this.dialogClosedResolver = resolve;
            });
            this.dialogElement.close(this.currentAction || this.defaultAction);
            await closedPromise;
            // enable scrolling late to avoid layout shift when closing
            if (!this.modeless) {
                this.constructor
                    .setDocumentScrollingDisabled(this.open);
            }
        }
        // Focus initial element.
        if (this.open) {
            this.focus();
        }
        if (shouldDispatchAction) {
            this.dispatchActionEvent(this.open ? 'opened' : 'closed');
        }
        this.currentAction = undefined;
    }
    dispatchActionEvent(type) {
        const detail = { action: this.open ? 'none' : this.currentAction };
        this.dispatchEvent(new CustomEvent(type, { detail, bubbles: true }));
    }
    updateFullscreen() {
        if (this.fullscreenQuery !== undefined) {
            this.fullscreenQuery.removeEventListener('change', this.fullscreenQueryListener);
            this.fullscreenQuery = this.fullscreenQueryListener = undefined;
        }
        if (!this.fullscreen) {
            this.showingFullscreen = false;
            return;
        }
        this.fullscreenQuery = window.matchMedia(this.fullscreenBreakpoint);
        this.fullscreenQuery.addEventListener('change', (this.fullscreenQueryListener = (event) => {
            this.showingFullscreen = event.matches;
        }));
        this.showingFullscreen = this.fullscreenQuery.matches;
    }
    // handles native close/cancel events and we just ensure
    // internal state is in sync.
    handleDialogDismiss(e) {
        if (e.type === 'cancel') {
            this.currentAction = this.escapeKeyAction;
        }
        this.dialogClosedResolver?.();
        this.dialogClosedResolver = undefined;
        this.open = false;
        this.opening = false;
        this.closing = false;
    }
    handleDialogClick(e) {
        if (!this.open) {
            return;
        }
        this.currentAction =
            e.target.getAttribute(this.actionAttribute) ??
                (!this.modeless && !e.composedPath().includes(this.containerElement) ?
                    this.scrimClickAction :
                    '');
        if (this.currentAction !== '') {
            this.close(this.currentAction);
        }
    }
    /* This allows the dividers to dynamically show based on scrolling. */
    handleContentScroll() {
        this.throttle('scroll', () => {
            this.requestUpdate();
        });
    }
    getFocusElement() {
        const selector = `[${this.focusAttribute}]`;
        const slotted = [this.footerSlot, this.contentSlot].flatMap(slot => slot.assignedElements({ flatten: true }));
        for (const el of slotted) {
            const focusEl = el.matches(selector) ? el : el.querySelector(selector);
            if (focusEl) {
                return focusEl;
            }
        }
        return null;
    }
    focus() {
        this.getFocusElement()?.focus();
    }
    blur() {
        this.getFocusElement()?.blur();
    }
    canStartDrag(e) {
        if (this.draggable === false || e.defaultPrevented || !(e.buttons & 1) ||
            !e.composedPath().includes(this.headerElement)) {
            return false;
        }
        return true;
    }
    handlePointerMove(e) {
        if (!this.dragging && !this.canStartDrag(e)) {
            return;
        }
        const { top, left, height, width } = this.containerElement.getBoundingClientRect();
        if (!this.dragging) {
            this.containerElement.setPointerCapture(e.pointerId);
            this.dragging = true;
            const { x, y } = e;
            this.dragInfo = [x, y, top, left];
        }
        const [sx, sy, st, sl] = this.dragInfo ?? [0, 0, 0, 0];
        const dx = e.x - sx;
        const dy = e.y - sy;
        const ml = window.innerWidth - width - this.dragMargin;
        const mt = window.innerHeight - height - this.dragMargin;
        const l = Math.max(this.dragMargin, Math.min(ml, dx + sl));
        const t = Math.max(this.dragMargin, Math.min(mt, dy + st));
        this.style.setProperty('--_container-drag-inline-start', `${l}px`);
        this.style.setProperty('--_container-drag-block-start', `${t}px`);
    }
    handleDragEnd(e) {
        if (!this.dragging) {
            return;
        }
        this.containerElement.releasePointerCapture(e.pointerId);
        this.dragging = false;
        this.dragInfo = undefined;
    }
}
Dialog.preventedScrollingCount = 0;
__decorate([
    property({ type: Boolean }),
    __metadata("design:type", Object)
], Dialog.prototype, "open", void 0);
__decorate([
    property({ type: Boolean }),
    __metadata("design:type", Object)
], Dialog.prototype, "fullscreen", void 0);
__decorate([
    property({ type: String }),
    __metadata("design:type", Object)
], Dialog.prototype, "fullscreenBreakpoint", void 0);
__decorate([
    property({ type: Boolean }),
    __metadata("design:type", Object)
], Dialog.prototype, "footerHidden", void 0);
__decorate([
    property({ type: Boolean }),
    __metadata("design:type", Object)
], Dialog.prototype, "stacked", void 0);
__decorate([
    property(),
    __metadata("design:type", Object)
], Dialog.prototype, "defaultAction", void 0);
__decorate([
    property(),
    __metadata("design:type", Object)
], Dialog.prototype, "actionAttribute", void 0);
__decorate([
    property(),
    __metadata("design:type", Object)
], Dialog.prototype, "focusAttribute", void 0);
__decorate([
    property({ type: String }),
    __metadata("design:type", Object)
], Dialog.prototype, "scrimClickAction", void 0);
__decorate([
    property({ type: String }),
    __metadata("design:type", Object)
], Dialog.prototype, "escapeKeyAction", void 0);
__decorate([
    property({ type: Boolean, reflect: true }),
    __metadata("design:type", Object)
], Dialog.prototype, "modeless", void 0);
__decorate([
    property({ type: Boolean }),
    __metadata("design:type", Object)
], Dialog.prototype, "draggable", void 0);
__decorate([
    query('.dialog', true),
    __metadata("design:type", HTMLDialogElement)
], Dialog.prototype, "dialogElement", void 0);
__decorate([
    query('slot[name=footer]', true),
    __metadata("design:type", HTMLSlotElement)
], Dialog.prototype, "footerSlot", void 0);
__decorate([
    query('slot:not([name])', true),
    __metadata("design:type", HTMLSlotElement)
], Dialog.prototype, "contentSlot", void 0);
__decorate([
    query(`.content`, true),
    __metadata("design:type", HTMLDivElement)
], Dialog.prototype, "contentElement", void 0);
__decorate([
    query(`.container`, true),
    __metadata("design:type", HTMLDivElement)
], Dialog.prototype, "containerElement", void 0);
__decorate([
    query(`.header`, true),
    __metadata("design:type", HTMLDivElement)
], Dialog.prototype, "headerElement", void 0);
__decorate([
    state(),
    __metadata("design:type", Object)
], Dialog.prototype, "showingFullscreen", void 0);
__decorate([
    state(),
    __metadata("design:type", Object)
], Dialog.prototype, "showingOpen", void 0);
__decorate([
    state(),
    __metadata("design:type", Object)
], Dialog.prototype, "opening", void 0);
__decorate([
    state(),
    __metadata("design:type", Object)
], Dialog.prototype, "closing", void 0);
__decorate([
    property({ reflect: true }),
    __metadata("design:type", Object)
], Dialog.prototype, "transition", void 0);
__decorate([
    state(),
    __metadata("design:type", Object)
], Dialog.prototype, "dragging", void 0);
//# sourceMappingURL=dialog.js.map