// Copyright 2021 The Tint Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#ifndef TOOLS_SRC_CMD_REMOTE_COMPILE_SOCKET_H_
#define TOOLS_SRC_CMD_REMOTE_COMPILE_SOCKET_H_

#include <atomic>
#include <memory>

/// Socket provides an OS abstraction to a TCP socket.
class Socket {
  public:
    /// Connects to the given TCP address and port.
    /// @param address the target socket address
    /// @param port the target socket port
    /// @param timeout_ms the timeout for the connection attempt.
    ///        If timeout_ms is non-zero and no connection was made before timeout_ms milliseconds,
    ///        then nullptr is returned.
    /// @returns the connected Socket, or nullptr on failure
    static std::shared_ptr<Socket> Connect(const char* address,
                                           const char* port,
                                           uint32_t timeout_ms);

    /// Begins listening for connections on the given TCP address and port.
    /// Call Accept() on the returned Socket to block and wait for a connection.
    /// @param address the socket address to listen on. Use "localhost" for connections from only
    ///        this machine, or an empty string to allow connections from any incoming address.
    /// @param port the socket port to listen on
    /// @returns the Socket that listens for connections
    static std::shared_ptr<Socket> Listen(const char* address, const char* port);

    /// Attempts to read at most `n` bytes into buffer, returning the actual number of bytes read.
    /// read() will block until the socket is closed or at least one byte is read.
    /// @param buffer the output buffer. Must be at least `n` bytes in size.
    /// @param n the maximum number of bytes to read
    /// @return the number of bytes read, or 0 if the socket was closed or errored
    virtual size_t Read(void* buffer, size_t n) = 0;

    /// Writes `n` bytes from buffer into the socket.
    /// @param buffer the source data buffer. Must be at least `n` bytes in size.
    /// @param n the number of bytes to read from `buffer`
    /// @returns true on success, or false if there was an error or the socket was
    /// closed.
    virtual bool Write(const void* buffer, size_t n) = 0;

    /// @returns true if the socket has not been closed.
    virtual bool IsOpen() = 0;

    /// Closes the socket.
    virtual void Close() = 0;

    /// Blocks for a connection to be made to the listening port, or for the Socket to be closed.
    /// @returns a pointer to the next established incoming connection
    virtual std::shared_ptr<Socket> Accept() = 0;
};

#endif  // TOOLS_SRC_CMD_REMOTE_COMPILE_SOCKET_H_
