#if 0
#pragma makedep unix
#endif

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/epoll.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <stddef.h>
#include <string.h>
#include <time.h>

#include "dvhp.h"

static int enable_log = 0;
#define debug(...)  do{if(enable_log) printf(__VA_ARGS__);}while(0)

#ifndef DV_SOCKET
#define DV_SOCKET "/tmp/dv"
#endif

#define PACKET_HEADER_SIZE sizeof(int)

enum DV_STATUS {
    DV_GET_PACKET_HEADER = 0,
    DV_GET_PACKET_BODY,
    DV_GET_SIGN_REQ,
    DV_GET_UNSUPPORT,
    DV_HANDLE_PACKET,
    DV_HANDLE_COMPLE,
};

#define KEY_SIZE   1024
#define EV_COUNT   128
#define DV_TIMEOUT -1
#define DV_NEW(fd, is, os) new_dv_data(fd, is, os)
#define DV_DATA(x) ((dv_data_t*)(x.ptr))
#define DV_DATA_FD(x) DV_DATA(x)->fd

extern int isEnableVar();

typedef struct dv_buffer_s {
    union {
        int  pos;
        int  handled_size;
    };
    union {
        int  need_handle_size;
        int  available_size;
    };

    int  capacity;
    char data[0];
} dv_buffer_t;

typedef struct dv_data_s {
    int fd;
    dv_buffer_t* in_buf;
    dv_buffer_t* out_buf;
    void*        user;
    int          status;
} dv_data_t;

dv_buffer_t* __new_dv_buffer(int sz) {
    dv_buffer_t* dv_buf = (dv_buffer_t*)calloc(1, sizeof(dv_buffer_t) + sz);
    if (!dv_buf) goto _err;
    dv_buf->capacity = sz;

    return dv_buf;
_err:
    perror("[dv] __new_dv_buffer fail");
    exit(EXIT_FAILURE);
    return 0;
}
#define __free_dv_buffer(x) free(x)

dv_data_t* new_dv_data(int fd, int in_sz, int out_sz) {
    dv_data_t* dv_data = (dv_data_t*)calloc(1, sizeof(dv_data_t));
    if (!dv_data) goto _err;

    dv_data->fd = fd;

    if (in_sz>0) dv_data->in_buf = __new_dv_buffer(in_sz);

    if (out_sz>0) dv_data->out_buf = __new_dv_buffer(out_sz);

    return dv_data;
_err:
    perror("[dv] new_dv_data fail");
    exit(EXIT_FAILURE);
    return 0;
}

void del_dv_data(dv_data_t* dv_data) {
    if (!dv_data) return;

    if (dv_data->in_buf) __free_dv_buffer(dv_data->in_buf);
    if (dv_data->out_buf) __free_dv_buffer(dv_data->out_buf);

    free(dv_data);
}

void dv_buffer_pop(dv_buffer_t* buf, int len) {
    memmove(buf->data, buf->data+len, buf->capacity-len);
    buf->pos -= len;
}

void dv_buffer_append(dv_buffer_t* buf, char* data, int len) {
    memcpy(buf->data+buf->need_handle_size, data, len);
    buf->need_handle_size += len;
}

void setnonblocking(int fd) {
    int flag = fcntl(fd, F_GETFL, 0);
    if (flag < 0) {
        perror("[dv] fcntl F_GETFL:");
        return;
    }
    if (fcntl(fd, F_SETFL, flag | O_NONBLOCK) < 0) {
        perror("[dv] fcntl F_SETFL:");
    }
}

int epoll_add(int efd, dv_data_t* data) {
    struct epoll_event ev;
    ev.events = EPOLLIN;
    ev.data.ptr = data;
    if (-1 == epoll_ctl(efd, EPOLL_CTL_ADD, data->fd, &ev)) {
        perror("[dv] epoll_ctl: sock");
        exit(EXIT_FAILURE);
        return 1;
    }
    return 0;
}

void epoll_write(int efd, dv_data_t* data, int enable) {
    struct epoll_event ev;
    ev.events = EPOLLIN | (enable ? EPOLLOUT : 0);
    ev.data.ptr = data;
    epoll_ctl(efd, EPOLL_CTL_MOD, data->fd, &ev);
}

void epoll_del(int efd, dv_data_t* data) {
    epoll_ctl(efd, EPOLL_CTL_DEL, data->fd, 0);
    close(data->fd);
    del_dv_data(data);
}

void handle_read(int ep_fd, dv_data_t* dv_data) {
    dv_buffer_t* in_buf = dv_data->in_buf;
    dv_buffer_t* out_buf = dv_data->out_buf;

    // 只要当前满足处理条件就会一直处理
    for(;;) {
        switch (dv_data->status) {
            case DV_GET_PACKET_HEADER: {
                if (in_buf->handled_size < PACKET_HEADER_SIZE) return;
                int bdsz = ((int*)in_buf->data)[0];
                if (bdsz<=0) {
                    epoll_del(ep_fd, dv_data);
                    return;
                }

                debug("[dv] get packet header: {pksz:%d, {bdsz:%d}}\n", (int)PACKET_HEADER_SIZE+bdsz, bdsz);

                // dv_data->user = __new_dv_buffer(bdsz);
                dv_buffer_pop(in_buf, PACKET_HEADER_SIZE);
                if (isEnableVar() && is_sign_req(in_buf->data, bdsz)) {
                    dv_data->status = DV_GET_SIGN_REQ;
                } else {
                    dv_data->status = DV_GET_UNSUPPORT;
                }
            } break;
            // don't support other than sign_req data
            case DV_GET_UNSUPPORT: {
                char data[64];
                dv_data->status = DV_HANDLE_PACKET;
                // just send random data as reply
                create_sign_req(data, 64);
                dv_buffer_append(out_buf, data,  64);
                dv_buffer_pop(in_buf, in_buf->handled_size);
                dv_data->status = DV_HANDLE_COMPLE;
                epoll_write(ep_fd, dv_data, 1);
                dv_data->status = DV_GET_PACKET_HEADER;
            }
            break;
            case DV_GET_SIGN_REQ: {
                dv_data->status = DV_HANDLE_PACKET;
                void* origin = malloc( in_buf->handled_size);
                memcpy(origin, in_buf->data, in_buf->handled_size);
                dv_sign(in_buf->data, in_buf->handled_size, in_buf->data, in_buf->handled_size);
                dv_buffer_append(out_buf, in_buf->data,  in_buf->handled_size);
                dv_buffer_pop(in_buf, in_buf->handled_size);
                dv_data->status = DV_HANDLE_COMPLE;
                epoll_write(ep_fd, dv_data, 1);
                dv_data->status = DV_GET_PACKET_HEADER;
            }
            break;
            case DV_GET_PACKET_BODY: {
                #if 0
                dv_buffer_t* user_buf = (dv_buffer_t*)dv_data->user;
                if (in_buf->handled_size < user_buf->capacity) return;
                dv_buffer_append(user_buf, in_buf->data, in_buf->handled_size);
                dv_buffer_pop(in_buf, in_buf->handled_size);

                // 开始处理数据
                dv_data->status = DV_HANDLE_PACKET;

                {
                    char hex[3] = {0};
                    char chain[256+1] = {0};
                    unsigned char sig[SIG_LENGTH];

                    if (user_buf->capacity < 128) {
                        for (int i=0; i<user_buf->capacity; i++) {
                            sprintf(hex,  "%02x", user_buf->data[i]);
                            strcat(chain, hex);
                        }
                        debug("[dv] get packet body: {bdsz: %d %s}\n", user_buf->capacity, chain);
                    }
                    else {
                        debug("[dv] get packet body: {bdsz: %d ??}\n", user_buf->capacity);
                    }

                    if (!isEnableVar() || !dv_sign((const unsigned char*)user_buf->data, user_buf->capacity, sig, SIG_LENGTH)) {
                        // sig 随机
                        srand(time(NULL) + rand());
                        for (int i = 0; i  < SIG_LENGTH; i++)
                        {
                            sig[i] = rand()%255+0;
                        }
                    }

                    for (int i = 0; i < SIG_LENGTH-2; i++ ) {
                        sig[i] ^= sig[SIG_LENGTH-1];
                    }

                    // 写出
                    dv_buffer_append(out_buf, (char*)sig, SIG_LENGTH);
                }

                dv_data->status = DV_HANDLE_COMPLE;
                __free_dv_buffer(user_buf);

                // 全双工模式，需要cli协议配合
                epoll_write(ep_fd, dv_data, 1);

                // 重置
                dv_data->status = DV_GET_PACKET_HEADER;
                #endif
            } break;
            default: {
                return;
            }
        }
    }
}

void handle_write(int ep_fd, dv_data_t* dv_data) {
    dv_buffer_t* out_buf = dv_data->out_buf;
    if (out_buf->need_handle_size == 0) {
        dv_buffer_pop(out_buf, out_buf->handled_size);
        debug("\n");

        // 关闭写
        epoll_write(ep_fd, dv_data, 0);
    }
}

int srv_loop () {
    int srv_fd, ep_fd, nfds;

    struct sockaddr_un srv_addr;
    int srv_len;

    // 声明epoll_event结构体的变量, ev用于注册事件, 数组用于回传要处理的事件
    struct epoll_event events[EV_COUNT];

    enable_log = !!getenv("DV");

    // 删除原有server_socket对象
    unlink (DV_SOCKET);

    // 创建通信协议为AF_UNIX, SCK_STREAM 数据方式 的 socket，并设为非阻塞模式
    srv_fd = socket (AF_UNIX, SOCK_STREAM, 0);
    setnonblocking(srv_fd);

    srv_addr.sun_family = AF_UNIX;
    strcpy(srv_addr.sun_path, DV_SOCKET);

    debug("[dv] sock path: %s\n", DV_SOCKET);
    srv_len = sizeof (srv_addr);

    // 绑定并监听
    if (0 != bind(srv_fd, (struct sockaddr *)&srv_addr, srv_len) ||
        0 != listen(srv_fd, EV_COUNT)) {
        perror("[dv] listen fail");
        exit(EXIT_FAILURE);
    }

    debug ("[dv] dv start...\n");

    // 生成用于处理accept的epoll专用的文件描述符
    ep_fd = epoll_create(EV_COUNT);
    if (-1 == ep_fd) {
        perror("[dv] epoll_create fail");
        exit(EXIT_FAILURE);
    }

    //注册 srv_fd 到 epoll 事件
    epoll_add(ep_fd, DV_NEW(srv_fd, 0, 0));

    for (;;) {
        nfds = epoll_wait(ep_fd, events, EV_COUNT, DV_TIMEOUT);
        if (-1 == nfds && errno != ETIMEDOUT && errno != EINTR) {
            perror("[dv] epoll_wait fail");
            exit(EXIT_FAILURE);
        }
        for (int n = 0; n < nfds; ++n) {
            dv_data_t* dv_data = DV_DATA(events[n].data);
            int fd = dv_data->fd;
            if (fd == srv_fd) {
                fd = accept(srv_fd, 0, 0);
                if (-1 == fd) {
                    perror("[dv] accept fail");
                    continue;
                }
                setnonblocking(fd);
                epoll_add(ep_fd, DV_NEW(fd, KEY_SIZE, KEY_SIZE));
            }
            else {
                dv_buffer_t* dv_buf;
                int len;
                ssize_t (*action)(int,  void *, size_t);
                void (*handler)(int, dv_data_t*);
                if (events[n].events & EPOLLIN) {
                    action = &read;
                    handler = handle_read;
                    dv_buf = dv_data->in_buf;
                    dv_buf->available_size = dv_buf->capacity-dv_buf->pos;
                    if (dv_buf->available_size < 0) {
                        perror("[dv] in_buf too small\n");
                        epoll_del(ep_fd, dv_data);
                        continue;
                    }
                }
                else if (events[n].events & EPOLLOUT) {
                    action = (ssize_t (*)(int,  void *, size_t))&write;
                    handler = handle_write;
                    dv_buf = dv_data->out_buf;
                }
                else {
                    perror("[dv] why here?\n");
                    epoll_del(ep_fd, dv_data);
                    continue;
                }

                len = (int)action(fd, dv_buf->data+dv_buf->pos, dv_buf->need_handle_size);
                if (len > 0) {
                    debug("[dv] %s data sz: %d\n", events[n].events & EPOLLIN?"read":"write",len);
                    dv_buf->pos+=len;

                    // for write
                    dv_buf->need_handle_size-=len;
                    handler(ep_fd, dv_data);
                }
                else if (len < 0) {
                    switch(errno) {
                    case EINTR:
                    case EAGAIN:
                        debug("[dv] try again\n");
                        break;
                    default:
                        epoll_del(ep_fd, dv_data);
                        debug("[dv] err: %s\n", strerror(errno));
                    }
                }
                else if (len == 0) {
                    epoll_del(ep_fd, dv_data);
                    debug("[dv] close cli fd\n");
                }
            }
        }
    }
    return 0;
}
