/*************************************************************************\
 *
 * Package:        ProbDist
 * File:           finv.c
 * Environment:    ANSI C
 *
 * Copyright (c) 2002 Pierre L'Ecuyer, DIRO, Université de Montréal.
 * e-mail: lecuyer@iro.umontreal.ca
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted without a fee for private, research,
 * academic, or other non-commercial purposes.
 * Any use of this software in a commercial environment requires a
 * written licence from the copyright owner.
 *
 * Any changes made to this package must be clearly identified as such.
 *
 * In scientific publications which used this software, a reference to it
 * would be appreciated.
 *
 * Redistributions of source code must retain this copyright notice
 * and the following disclaimer.
 *
 * THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
\*************************************************************************/

#include "finv.h"
#include "fdist.h"
#include "wdist.h"

#include "num.h"
#include "num2.h"
#include "util.h"
#include "gdef.h"

#include <limits.h>
#include <float.h>
#include <math.h>
#include <stdio.h>


double fdist_belog(double);
void fdist_CalcB4 (double, double *, double *, double *, double *);

#define TRACE0(x) printf ("***   " #x "     ");


extern const double fdist_XINF;
extern const double fdist_XBIG;
extern const double fdist_XBIGM;

/* EpsArray[j]: Epsilon required for j decimal degits of precision */
static const double EpsArray[] = {
   0.5, 0.5E-1, 0.5E-2, 0.5E-3, 0.5E-4, 0.5E-5, 0.5E-6, 0.5E-7, 0.5E-8,
   0.5E-9, 0.5E-10, 0.5E-11, 0.5E-12, 0.5E-13, 0.5E-14, 0.5E-15, 0.5E-16,
   0.5E-17, 0.5E-18, 0.5E-19, 0.5E-20, 0.5E-21, 0.5E-22, 0.5E-23, 0.5E-24,
   0.5E-25, 0.5E-26, 0.5E-27, 0.5E-28, 0.5E-29, 0.5E-30, 0.5E-31, 0.5E-32,
   0.5E-33, 0.5E-34, 0.5E-35
};




/*=========================================================================*/

double finv_Expon (double u)
{
   util_Assert (u >= 0.0 && u <= 1.0, "finv_Expon:   u not in [0, 1]");
   if (u >= 1.0) {
      util_Warning (1, "finv_Expon:   u = 1");
      return fdist_XBIGM;
   }
   if (u <= 0.0)
      return 0.0;
   return -num2_log1p (-u);
}


/*=========================================================================*/

double finv_Weibull (double c, double u)
{
   double t;

   util_Assert (c > 0.0, "finv_Weibull:   c <= 0");
   util_Assert (u >= 0.0 && u <= 1.0, "finv_Weibull:   u not in [0, 1]");
   if (u <= 0.0)
      return 0.0;
   if (u >= 1.0) {
      util_Warning (1, "finv_Weibull:   u = 1");
      return fdist_XINF;
   }

   t = -num2_log1p (-u);
   if (log10 (t) >= c * DBL_MAX_10_EXP) {
      util_Warning (1, "finv_Weibull:   u --> 1");
      return fdist_XINF;
   }
   
   return pow (t, 1.0 / c);
}


/*=========================================================================*/

double finv_ExtremeValue (double u)
{

   util_Assert (u >= 0.0 && u <= 1.0, "finv_ExtremeValue:  u not in [0, 1]");
   if (u >= 1.0) {
      util_Warning (1, "finv_ExtremeValue:   u = 1");
      return fdist_XBIG;
   }
   if (u <= 0.0) {
      util_Warning (1, "finv_ExtremeValue:   u = 0");
      return -fdist_XBIG;
   }
   return -log (-log (u));
}


/*=========================================================================*/

double finv_Logistic (double u)
{
   util_Assert (u >= 0.0 && u <= 1.0, "finv_Logistic:   u not in [0, 1]");
   if (u >= 1.0) {
      util_Warning (1, "finv_Logistic:   u = 1");
      return fdist_XBIGM;
   }
   if (u <= 0.0) {
      util_Warning (1, "finv_Logistic:   u = 0");
      return -fdist_XBIGM;
   }
   return log(u / (1.0 - u));
}


/*=========================================================================*/

double finv_Pareto (double c, double u)
{
   double t;
   util_Assert (c > 0.0, "finv_Pareto:   c <= 0");
   util_Assert (u >= 0.0 && u <= 1.0, "finv_Pareto:   u not in [0, 1]");
   if (u <= 0.0)
      return 1.0;
   t = -num2_log1p (-u);

   if ((u >= 1.0) || t >= c * DBL_MAX_EXP) {
      util_Warning (1, "finv_Pareto:   u = 1  or  t >= c * DBL_MAX_EXP");
      return fdist_XINF;
   }
   return pow (1.0 / (1 - u), 1.0 / c);
}


/**************************************************************************/

double finv_Normal3 (double y)
/*
 * Returns Inverse of the cdf of the normal distribution
 * As in P.Bratley, B.L.Fox, and L.E.Schrage. A Guide to Simulation
 * Springer-Verlag, New York, second edition, 1987.
 * in Figure L27
 */
{
   static const double plim = 1.0e-18;
   static const double p0 = -0.322232431088;
   static const double p1 = -1.0;
   static const double p2 = -0.342242088547;
   static const double p3 = -0.0204231210245;
   static const double p4 = -0.453642210148e-4;

   static const double q0 = 0.099348462606;
   static const double q1 = 0.588581570495;
   static const double q2 = 0.531103462366;
   static const double q3 = 0.10353775285;
   static const double q4 = 0.38560700634e-2;

   double p;
   double vtemp, Y;

   p = y;
   if (p > 0.5)
      p = 1.0 - p;

   if (p >= plim) {
      Y = sqrt (-log (p * p));
      vtemp = Y + ((((Y * p4 + p3) * Y + p2) * Y + p1) * Y + p0) /
         ((((Y * q4 + q3) * Y + q2) * Y + q1) * Y + q0);
      if (y < 0.5)
         vtemp = -vtemp;
      return vtemp;
   } else {
      vtemp = 8.0;
      if (y < 0.0) {
         printf ("Error: Negative Value in finv_Normal3\n");
         exit (EXIT_FAILURE);
      }
      if (y < 0.5)
         vtemp = -vtemp;
      return vtemp;
   }
}


/**************************************************************************/

static const double InvNormal1P1[7] = {
   0.160304955844066229311E2,
   -0.90784959262960326650E2,
   0.18644914861620987391E3,
   -0.16900142734642382420E3,
   0.6545466284794487048E2,
   -0.864213011587247794E1,
   0.1760587821390590
};

static const double InvNormal1Q1[7] = {
   0.147806470715138316110E2,
   -0.91374167024260313396E2,
   0.21015790486205317714E3,
   -0.22210254121855132366E3,
   0.10760453916055123830E3,
   -0.206010730328265443E2,
   0.1E1
};

static const double InvNormal1P2[8] = {
   -0.152389263440726128E-1,
   0.3444556924136125216,
   -0.29344398672542478687E1,
   0.11763505705217827302E2,
   -0.22655292823101104193E2,
   0.19121334396580330163E2,
   -0.5478927619598318769E1,
   0.237516689024448000
};

static const double InvNormal1Q2[8] = {
   -0.108465169602059954E-1,
   0.2610628885843078511,
   -0.24068318104393757995E1,
   0.10695129973387014469E2,
   -0.23716715521596581025E2,
   0.24640158943917284883E2,
   -0.10014376349783070835E2,
   0.1E1
};

static const double InvNormal1P3[11] = {
   0.56451977709864482298E-4,
   0.53504147487893013765E-2,
   0.12969550099727352403,
   0.10426158549298266122E1,
   0.28302677901754489974E1,
   0.26255672879448072726E1,
   0.20789742630174917228E1,
   0.72718806231556811306,
   0.66816807711804989575E-1,
   -0.17791004575111759979E-1,
   0.22419563223346345828E-2
};

static const double InvNormal1Q3[9] = {
   0.56451699862760651514E-4,
   0.53505587067930653953E-2,
   0.12986615416911646934,
   0.10542932232626491195E1,
   0.30379331173522206237E1,
   0.37631168536405028901E1,
   0.38782858277042011263E1,
   0.20372431817412177929E1,
   0.1E1
};

double finv_Normal1 (double x)
/*
 * Returns the inverse of the cdf of the normal distribution.
 * Rational approximations giving 16 decimals of precision.
 * The authors also give an approximation with 23 decimals of
 * precision.
 * J.M. Blair, C.A. Edwards, J.H. Johnson, "Rational Chebyshev
 * approximations for the Inverse of the Error Function", in
 * Mathematics of Computation, Vol. 30, 136, pp 827, (1976)
 */
{
   int i, negatif;
   double z, v, numer, denom;
  
   util_Assert (x >= 0.0, "finv_Normal1:   u < 0");
   util_Assert (x <= 1.0, "finv_Normal1:   u > 1");
   if (x <= 0.0) {
      util_Warning (1, "finv_Normal1:   u = 0");
      return -fdist_XBIG;
   }

   /* Transform x as argument of InvErf */
   z = x;
   x = 2.0 * x - 1.0;
   if (x >= 1.0) {
      util_Warning (1, "finv_Normal1:   u = 1");
      return fdist_XBIG;
   }
   if (x < 0.0) {
      x = -x;
      negatif = 1;
   } else {
      negatif = 0;
   }

   if (x <= 0.75) {
      v = x * x - 0.5625;
      numer = denom = 0.0;
      /* Evaluation of the 2 polynomials by Horner */
      for (i = 6; i >= 0; i--) {
         numer = numer * v + InvNormal1P1[i];
         denom = denom * v + InvNormal1Q1[i];
      }
      z = x * numer / denom;

   } else if (x <= 0.9375) {
      v = x * x - 0.87890625;
      numer = denom = 0.0;
      for (i = 7; i >= 0; i--) {
         numer = numer * v + InvNormal1P2[i];
         denom = denom * v + InvNormal1Q2[i];
      }
      z = x * numer / denom;
   } else {
      if (z > 1.0e-1)
         v = 1.0 / sqrt (-log (1.0 - x));
      else
         v = 1.0 / sqrt (-log (2.0*z));
      numer = denom = 0.0;
      for (i = 10; i >= 0; i--)
         numer = numer * v + InvNormal1P3[i];
      for (i = 8; i >= 0; i--)
         denom = denom * v + InvNormal1Q3[i];
      z = (1.0 / v) * numer / denom;
   }

   if (negatif)
      return -(z * num_Rac2);
   else
      return z * num_Rac2;
}


/*=========================================================================*/

typedef union {
  float u;
  int i;
} Uvalue;


double finv_Normal2 (double u)
{
   static const double A[1025] = {
     6.33795775455,     6.33321372178,     6.32860807635,
     6.32413288056,     6.31978086301,     6.31554534594,
     6.31142018224,     6.30739970055,     6.30347865735,
     6.29965219486,     6.29591580408,     6.29226529211,
     6.28869675323,     6.28520654322,     6.28179125646,
     6.27844770553,     6.27517290294,     6.27196404468,
      6.2688184955,     6.26573377562,     6.26270754867,
     6.25973761088,     6.25682188116,     6.25395839218,
     6.25114528219,     6.24838078761,      6.2456632362,
     6.24299104087,       6.240362694,     6.23777676216,
     6.23523188139,     6.23272675268,     6.23026013799,
     6.22543778081,     6.22075598811,         6.2162067,
     6.21178253304,     6.20747670647,     6.20328297819,
     6.19919558917,     6.19520921468,     6.19131892165,
     6.18752013112,     6.18380858505,       6.180180317,
     6.17663162601,      6.1731590534,     6.16975936206,
     6.16642951782,     6.16316667288,     6.15996815078,
      6.1568314329,     6.15375414629,     6.15073405259,
     6.14776903807,     6.14485710448,     6.14199636082,
     6.13918501574,     6.13642137069,     6.13370381356,
     6.13103081295,     6.12840091282,     6.12581272765,
     6.12326493791,     6.12075628597,     6.11585165173,
     6.11108986339,     6.10646273321,     6.10196276044,
     6.09758305644,     6.09331727974,     6.08915957943,
     6.08510454578,     6.08114716689,     6.07728279052,
     6.07350709041,     6.06981603653,     6.06620586849,
     6.06267307204,     6.05921435799,     6.05582664334,
     6.05250703436,     6.04925281142,     6.04606141524,
      6.0429304345,     6.03985759465,     6.03684074773,
     6.03387786311,     6.03096701912,     6.02810639533,
     6.02529426559,      6.0225289916,     6.01980901702,
     6.01713286212,     6.01449911878,     6.01190644597,
     6.00935356553,     6.00436236049,     5.99951639563,
     5.99480734912,     5.99022759726,     5.98577013833,
     5.98142852661,      5.9771968149,     5.97306950434,
     5.96904150032,     5.96510807378,     5.96126482692,
     5.95750766299,     5.95383275929,     5.95023654327,
     5.94671567111,     5.94326700856,     5.93988761376,
     5.93657472175,      5.9333257306,     5.93013818875,
     5.92700978364,     5.92393833142,     5.92092176747,
     5.91795813793,     5.91504559188,     5.91218237419,
     5.90936681905,     5.90659734398,     5.90387244436,
     5.90119068838,     5.89855071241,     5.89595121674,
     5.89086876372,     5.88593406746,     5.88113866541,
     5.87647480476,     5.87193536504,     5.86751379108,
     5.86320403456,     5.85900050299,     5.85489801495,
     5.85089176064,     5.84697726716,     5.84315036764,
     5.83940717394,     5.83574405227,     5.83215760142,
     5.82864463328,     5.82520215541,     5.82182735527,
     5.81851758606,     5.81527035394,     5.81208330646,
     5.80895422198,     5.80588100021,     5.80286165344,
     5.79989429868,     5.79697715037,     5.79410851379,
     5.79128677896,     5.78851041509,     5.78577796547,
     5.78308804272,     5.78043932448,      5.7752605134,
     5.77023210866,     5.76534549972,     5.76059279799,
     5.75596675813,     5.75146070984,     5.74706849845,
     5.74278443301,     5.73860324073,     5.73452002692,
     5.73053023962,     5.72662963823,     5.72281426575,
     5.71908042393,     5.71542465116,      5.7118437027,
     5.70833453283,     5.70489427893,     5.70152024702,
     5.69820989876,     5.69496083963,     5.69177080831,
     5.68863766688,     5.68555939205,     5.68253406704,
     5.67955987426,     5.67663508853,     5.67375807094,
     5.67092726314,     5.66814118219,      5.6653984157,
     5.66269761746,     5.65741684752,     5.65228927845,
     5.64730614385,     5.64245941209,      5.6377417063,
     5.63314623489,     5.62866673113,     5.62429740032,
      5.6200328734,     5.61586816622,     5.61179864349,
     5.60781998693,     5.60392816693,      5.6001194173,
      5.5963902128,     5.59273724893,     5.58915742386,
     5.58564782214,     5.58220570003,     5.57882847227,
     5.57551370004,     5.57225908009,     5.56906243489,
     5.56592170358,     5.56283493377,     5.55980027406,
     5.55681596716,     5.55388034364,     5.55099181609,
     5.54814887388,     5.54535007824,      5.5425940578,
     5.53720516928,     5.53197243466,     5.52688692307,
     5.52194045196,      5.5171255056,     5.51243516442,
     5.50786304341,     5.50340323832,     5.49905027849,
      5.4947990853,     5.49064493552,      5.4865834288,
     5.48261045887,     5.47872218783,      5.4749150232,
     5.47118559738,     5.46753074927,     5.46394750763,
     5.46043307616,     5.45698482001,     5.45360025356,
     5.45027702931,     5.44701292782,     5.44380584855,
     5.44065380149,     5.43755489952,     5.43450735142,
     5.43150945548,     5.42855959358,     5.42565622584,
     5.42279788561,     5.41998317492,     5.41447936842,
     5.40913484272,     5.40394049304,     5.39888797725,
     5.39396963282,     5.38917840481,     5.38450778312,
     5.37995174765,     5.37550472027,     5.37116152244,
     5.36691733786,     5.36276767939,     5.35870835965,
     5.35473546482,      5.3508453313,     5.34703452473,
     5.34329982125,     5.33963819058,      5.3360467808,
     5.33252290459,     5.32906402674,     5.32566775291,
     5.32233181932,     5.31905408341,     5.31583251534,
     5.31266519015,     5.30955028069,     5.30648605108,
     5.30347085068,     5.30050310862,     5.29758132874,
     5.29470408485,     5.28907782499,     5.28361416415,
     5.27830381332,      5.2731382613,        5.26810969,
     5.26321090103,     5.25843525172,      5.2537765992,
     5.24922925139,     5.24478792384,     5.24044770161,
     5.23620400551,     5.23205256217,     5.22798937729,
     5.22401071182,      5.2201130606,     5.21629313323,
     5.21254783681,     5.20887426039,       5.205269661,
     5.20173145085,     5.19825718588,     5.19484455528,
     5.19149137189,      5.1881955636,     5.18495516531,
     5.18176831175,     5.17863323075,     5.17554823716,
     5.17251172724,     5.16952217346,     5.16657811973,
     5.16082101937,     5.15523004763,     5.14979571985,
     5.14450934531,     5.13936294072,     5.13434915534,
     5.12946120568,     5.12469281843,      5.1200381805,
     5.11549189488,     5.11104894181,     5.10670464433,
     5.10245463765,     5.09829484194,     5.09422143797,
     5.09023084532,     5.08631970285,     5.08248485106,
      5.0787233163,     5.07503229637,      5.0714091476,
     5.06785137307,     5.06435661191,     5.06092262957,
     5.05754730896,     5.05422864229,     5.05096472372,
      5.0477537425,     5.04459397676,     5.04148378777,
     5.03842161464,     5.03540596946,     5.02950864968,
     5.02378122248,     5.01821399503,     5.01279808548,
     5.00752533468,      5.0023882297,     4.99737983708,
     4.99249374467,     4.98772401045,     4.98306511755,
     4.97851193459,     4.97405968039,     4.96970389282,
     4.96544040091,     4.96126529999,     4.95717492946,
     4.95316585275,     4.94923483945,     4.94537884901,
     4.94159501614,     4.93788063759,     4.93423316006,
     4.93065016935,     4.92712938035,     4.92366862801,
     4.92026585904,     4.91691912431,     4.91362657193,
     4.91038644087,     4.90719705507,     4.90405681805,
     4.90096420796,     4.89491612703,      4.8890419637,
     4.88333180298,     4.87777655906,     4.87236788511,
     4.86709809501,     4.86196009527,     4.85694732544,
     4.85205370587,      4.8472735917,     4.84260173228,
      4.8380332352,      4.8335635343,     4.82918836126,
     4.82490372013,      4.8207058646,     4.81659127758,
     4.81255665279,     4.80859887824,      4.8047150212,
     4.80090231471,     4.79715814524,     4.79348004153,
     4.78986566443,     4.78631279763,     4.78281933914,
     4.77938329358,     4.77600276506,     4.77267595063,
     4.76940113431,     4.76617668159,     4.76300103427,
     4.75679027709,      4.7507577546,      4.7448933141,
     4.73918765149,     4.73363221901,     4.72821914523,
     4.72294116527,     4.71779155991,     4.71276410212,
     4.70785300996,     4.70305290507,     4.69835877584,
      4.6937659447,       4.689270039,     4.68486696501,
     4.68055288459,     4.67632419431,     4.67217750662,
     4.66810963292,      4.6641175682,     4.66019847719,
     4.65634968177,     4.65256864953,     4.64885298336,
     4.64520041195,     4.64160878112,     4.63807604585,
     4.63460026304,     4.63117958481,     4.62781225236,
      4.6244965903,      4.6212310015,     4.61484401763,
     4.60863991397,     4.60260828265,     4.59673958525,
     4.59102505831,     4.58545663137,     4.58002685553,
     4.57472884104,     4.56955620253,     4.56450301082,
     4.55956375032,     4.55473328135,     4.55000680661,
     4.54537984133,     4.54084818664,      4.5364079057,
     4.53205530226,     4.52778690138,     4.52359943209,
     4.51948981163,     4.51545513127,     4.51149264343,
      4.5075997499,     4.50377399122,     4.50001303691,
     4.49631467662,     4.49267681191,      4.4890974489,
     4.48557469132,     4.48210673427,     4.47869185842,
     4.47532842465,     4.46874969883,     4.46235887005,
     4.45614525563,     4.45009906476,     4.44421130165,
     4.43847368137,     4.43287855675,     4.42741885424,
     4.42208801786,     4.41687995966,     4.41178901614,
      4.4068099095,     4.40193771332,     4.39716782193,
     4.39249592307,      4.3879179734,     4.38343017655,
     4.37902896328,     4.37471097372,     4.37047304122,
     4.36631217778,      4.3622255608,     4.35821052107,
     4.35426453179,     4.35038519863,     4.34657025053,
     4.34281753142,     4.33912499252,     4.33549068532,
     4.33191275513,     4.32838943511,     4.32491904083,
     4.31813067359,     4.31153564248,      4.3051229678,
     4.29888258592,      4.2928052498,     4.28688244266,
     4.28110630283,     4.27546955801,     4.26996546758,
     4.26458777186,     4.25933064724,     4.25418866648,
     4.24915676339,      4.2442302014,     4.23940454548,
       4.234675637,     4.23003957111,     4.22549267647,
     4.22103149691,      4.2166527749,     4.21235343653,
     4.20813057793,     4.20398145293,     4.19990346176,
     4.19589414084,     4.19195115337,     4.18807228072,
     4.18425541464,     4.18049854994,     4.17679977794,
     4.17315728028,     4.16956932335,     4.16255048996,
     4.15573091512,     4.14909929681,     4.14264527528,
     4.13635933075,      4.1302326947,     4.12425727258,
     4.11842557618,     4.11273066434,      4.1071660908,
     4.10172585798,     4.09640437621,      4.0911964274,
     4.08609713269,     4.08110192358,     4.07620651603,
     4.07140688728,      4.0666992549,     4.06208005798,
     4.05754594009,     4.05309373387,     4.04872044701,
     4.04442324957,     4.04019946237,     4.03604654644,
     4.03196209338,     4.02794381649,      4.0239895427,
     4.02009720515,      4.0162648363,     4.01249056171,
     4.00877259417,     4.00149883606,     3.99443081604,
     3.98755688078,     3.98086634771,     3.97434939969,
     3.96799699364,     3.96180078097,     3.95575303793,
     3.94984660457,     3.94407483096,     3.93843152965,
     3.93291093367,     3.92750765903,      3.9222166715,
     3.91703325678,     3.91195299388,     3.90697173117,
     3.90208556486,     3.89729081955,     3.89258403075,
     3.88796192889,     3.88342142501,     3.87895959754,
     3.87457368047,     3.87026105237,     3.86601922652,
     3.86184584173,     3.85773865405,     3.85369552908,
     3.84971443491,     3.84579343567,      3.8419306855,
     3.83437296653,     3.82702811235,     3.81988408487,
     3.81292984787,     3.80615525838,     3.79955097242,
     3.79310836293,     3.78681944792,      3.7806768275,
     3.77467362835,     3.76880345477,     3.76306034518,
     3.75743873361,     3.75193341525,     3.74653951576,
     3.74125246377,     3.73606796613,     3.73098198575,
     3.72599072151,     3.72109059012,     3.71627820978,
     3.71155038522,     3.70690409424,     3.70233647531,
     3.69784481637,     3.69342654456,     3.68907921677,
     3.68480051108,     3.68058821879,     3.67644023716,
     3.67235456274,     3.66832928512,     3.66045271041,
     3.65279688532,     3.64534934809,     3.63809867295,
     3.63103435798,     3.62414672762,     3.61742684789,
      3.6108664521,     3.60445787576,     3.59819399923,
       3.592068197,     3.58607429286,     3.58020651993,
     3.57445948514,     3.56882813748,      3.5633077395,
      3.5578938418,     3.55258226001,     3.54736905409,
     3.54225050954,     3.53722312046,     3.53228357414,
     3.52742873704,     3.52265564207,     3.51796147693,
     3.51334357346,     3.50879939795,     3.50432654219,
     3.49992271527,     3.49558573607,      3.4913135263,
     3.48710410411,     3.47886614237,     3.47085771144,
     3.46306588016,     3.45547879117,     3.44808554464,
     3.44087609744,      3.4338411752,     3.42697219544,
     3.42026120015,     3.41370079626,     3.40728410324,
     3.40100470653,     3.39485661617,     3.38883422999,
     3.38293230069,     3.37714590628,     3.37147042368,
     3.36590150482,     3.36043505519,     3.35506721435,
     3.34979433839,     3.34461298392,     3.33951989355,
     3.33451198267,      3.3295863274,     3.32474015352,
      3.3199708264,     3.31527584174,     3.31065281706,
      3.3060994839,      3.3016136806,     3.29719334569,
     3.28854129969,     3.28012863599,     3.27194189999,
     3.26396875282,     3.25619785067,     3.24861874006,
     3.24122176661,     3.23399799517,     3.22693913976,
     3.22003750184,     3.21328591581,     3.20667770063,
      3.2002066169,     3.19386682855,     3.18765286865,
     3.18155960874,     3.17558223129,     3.16971620498,
      3.1639572623,     3.15830137938,     3.15274475767,
     3.14728380734,     3.14191513221,     3.13663551594,
     3.13144190952,     3.12633141979,     3.12130129892,
     3.11634893481,     3.11147184224,     3.10666765471,
     3.10193411699,      3.0972690782,     3.08813637782,
     3.07925420286,     3.07060851058,      3.0621864209,
      3.0539760908,     3.04596660532,     3.03814788252,
     3.03051059033,     3.02304607359,     3.01574628977,
     3.00860375216,     3.00161147949,      2.9947629512,
     2.98805206756,     2.98147311398,     2.97502072908,
     2.96868987601,     2.96247581659,     2.95637408809,
     2.95038048206,     2.94449102535,     2.93870196271,
     2.93300974113,     2.92741099539,     2.92190253507,
      2.9164813325,     2.91114451188,     2.90588933915,
     2.90071321284,     2.89561365554,      2.8905883061,
     2.88563491243,      2.8759354898,     2.86649931119,
     2.85731166746,     2.84835906508,     2.83962909503,
     2.83111031887,     2.82279216952,      2.8146648644,
     2.80671932916,     2.79894713055,     2.79134041701,
     2.78389186607,     2.77659463754,     2.76944233182,
     2.76242895259,     2.75554887345,     2.74879680789,
     2.74216778225,     2.73565711142,     2.72926037674,
     2.72297340607,     2.71679225565,     2.71071319364,
     2.70473268509,     2.69884737824,       2.693054092,
     2.68734980447,     2.68173164244,      2.6761968717,
     2.67074288817,     2.66536720972,     2.66006746862,
     2.64968685818,     2.63958415075,      2.6297438751,
     2.62015183589,     2.61079497623,     2.60166125833,
     2.59273955939,      2.5840195806,      2.5754917671,
      2.5671472376,     2.55897772201,     2.55097550632,
     2.54313338341,     2.53544460937,     2.52790286427,
     2.52050221719,     2.51323709464,     2.50610225225,
     2.49909274914,      2.4922039248,      2.4854313781,
      2.4787709483,     2.47221869765,      2.4657708957,
     2.45942400483,     2.45317466714,     2.44701969236,
     2.44095604678,     2.43498084314,     2.42909133123,
     2.42328488933,     2.41755901624,     2.40633953091,
     2.39541501107,     2.38476910497,     2.37438680605,
     2.36425430827,     2.35435888041,     2.34468875653,
     2.33523304007,      2.3259816196,     2.31692509462,
     2.30805470992,      2.2993622974,     2.29084022432,
     2.28248134715,     2.27427897031,     2.26622680921,
     2.25831895701,     2.25054985471,     2.24291426415,
     2.23540724356,     2.22802412548,     2.22076049665,
      2.2136121797,     2.20657521654,     2.19964585312,
     2.19282052555,     2.18609584736,     2.17946859779,
      2.1729357111,      2.1664942667,     2.16014147999,
     2.15387469406,     2.14158908915,     2.12961846912,
     2.11794545368,     2.10655408816,     2.09542969053,
     2.08455871841,     2.07392865314,     2.06352789832,
     2.05334569055,     2.04337202094,     2.03359756554,
     2.02401362372,     2.01461206336,     2.00538527192,
     1.99632611278,     1.98742788593,     1.97868429286,
     1.97008940476,      1.9616376339,     1.95332370775,
     1.94514264545,     1.93708973655,     1.92916052156,
     1.92135077429,     1.91365648573,     1.90607384923,
     1.89859924703,     1.89122923782,     1.88396054536,
     1.87679004798,     1.86971476892,     1.86273186742,
     1.84903246517,     1.83567153691,     1.82263048664,
     1.80989223848,     1.79744107395,     1.78526249044,
     1.77334307781,     1.76167041036,     1.75023295184,
     1.73901997173,     1.72802147122,     1.71722811751,
     1.70663118536,     1.69622250501,     1.68599441547,
     1.67593972277,     1.66605166239,     1.65632386534,
     1.64675032767,     1.63732538277,     1.62804367632,
     1.61890014354,     1.60988998842,     1.60100866489,
     1.59225185952,      1.5836154758,     1.57509561963,
     1.56668858607,     1.55839084718,     1.55019904079,
     1.54210996014,     1.53412054435,     1.51842914115,
     1.50310294313,     1.48812189602,     1.47346757795,
     1.45912302502,     1.44507257982,      1.4313017591,
       1.417797138,     1.40454624816,       1.391537488,
     1.37876004322,     1.36620381637,     1.35385936408,
     1.34171784108,     1.32977095018,      1.3180108973,
     1.30643035113,     1.29502240671,     1.28378055261,
     1.27269864119,     1.26177086164,     1.25099171546,
     1.24035599423,     1.22985875922,     1.21949532285,
     1.20926123171,     1.19915225099,      1.1891643502,
     1.17929369001,     1.16953661021,     1.15988961853,
     1.15034938038,     1.13157655839,     1.11319427716,
     1.09518065276,     1.07751556704,     1.06018047944,
     1.04315826332,     1.02643306314,     1.00999016925,
    0.993815907861,    0.977897543941,    0.962223195295,
    0.946781756301,    0.931562830007,    0.916556667533,
     0.90175411383,    0.887146559019,    0.872725894627,
    0.858484474142,    0.844415077375,    0.830510878205,
    0.816765415315,    0.803172565598,    0.789726519943,
    0.776421761148,    0.763253043733,    0.750215375468,
    0.737304000439,    0.724514383492,    0.711842195939,
    0.699283302383,    0.686833748575,    0.674489750196,
    0.650104070648,    0.626099012346,    0.602449453164,
    0.579132162256,    0.556125593619,    0.533409706241,
    0.510965806738,    0.488776411115,    0.466825122853,
    0.445096524986,    0.423576084201,    0.402250065322,
    0.381105454764,     0.36012989179,    0.339311606539,
    0.318639363964,     0.29810241293,    0.277690439822,
    0.257393526101,    0.237202109329,     0.21710694721,
    0.197099084294,    0.177169820992,     0.15731068461,
    0.137513402144,    0.117769874579,   0.0980721524887,
   0.0784124127331,   0.0587829360689,   0.0391760855031,
   0.0195842852301,                 0
};

   static const double  B[1025] = {
     468043598.745,     454185281.982,     441133386.786,
     428819269.757,     417181876.023,     406166717.813,
     395725013.783,     385812960.329,     376391111.851,
     367423851.404,     358878936.738,     350727109.464,
     342941757.343,     335498621.458,      328375541.45,
     321552233.174,     315010094.053,     308732032.185,
     302702315.899,     296906440.935,     291331012.915,
     285963643.058,     280792855.461,     275808004.446,
     270999200.737,     266357245.389,     261873570.517,
     257540186.041,     253349631.735,     249294933.976,
     245369566.664,     241567415.856,     237882747.698,
     230844652.301,     224215968.814,     217961855.044,
     212051320.023,     206456705.678,     201153250.091,
     196118717.706,     191333084.832,     186778271.013,
     182437908.646,     178297144.629,     174342468.981,
      170561566.22,     166943186.067,     163477030.605,
     160153655.481,     156964383.183,     153901226.667,
     150956821.959,     148124368.489,     145397576.172,
     142770618.331,     140238089.759,     137794969.238,
     135436586.005,     133158589.665,     130956923.161,
     128827798.432,     126767674.455,     124773237.414,
     122841382.741,     120969198.842,     117393074.498,
     114024901.787,     110846987.361,     107843593.262,
     105000673.785,      102305653.75,     99747240.7669,
     97315265.5629,     95000545.5991,      92794768.101,
     90690389.3539,     88680547.6828,     86758987.9953,
     84919996.1313,     83158341.5649,     81469227.2427,
     79848245.5433,     78291339.5027,     76794768.5853,
     75355078.3902,      73969073.775,     72633794.9552,
     71346496.2016,     70104626.8117,     68905814.0771,
     67747848.0069,     66628667.5985,     65546348.4768,
     64499091.7445,     63485213.9074,     62503137.7568,
     61551384.1017,     59733373.2481,     58021039.3817,
     56405393.2293,     54878438.8023,     53433039.7483,
     52062806.7313,     50762002.0764,     49525458.6675,
     48348510.6706,     47226934.1194,     46156895.7613,
     45134908.8531,     44157794.8309,     43222649.9599,
     42326816.2279,      41467855.862,     40643528.9565,
     39851773.7738,     39090689.3553,     38358520.1316,
     37653642.2677,     36974551.5205,     36319852.4159,
      35688248.581,     35078534.0907,     34489585.7054,
     33920355.8956,      33369866.562,     32837203.3696,
     32321510.6295,     31821986.6654,     31337879.6134,
     30413135.3113,     29542122.4706,     28720271.6567,
     27943518.2501,     27208234.5323,     26511172.4564,
     25849415.1891,     25220335.8956,     24621562.5335,
     24050947.6575,     23506542.4223,     22986574.1168,
     22489426.6833,     22013623.7674,     21557813.9244,
     21120757.6673,     20701316.0956,     20298440.8828,
     19911165.4376,     19538597.0812,     19179910.1071,
     18834339.6081,     18501175.9757,     18179759.9851,
     17869478.3966,     17569760.0097,     17280072.1174,
     16999917.3132,     16728830.6117,     16466376.8456,
     16212148.3111,     15965762.6321,     15495105.5155,
     15051783.6023,     14633472.8806,     14238106.0377,
     13863837.9303,     13509016.4872,     13172158.0713,
     12851926.5229,     12547115.2589,     12256631.9188,
     11979485.1457,     11714773.1625,     11461673.8654,
     11219436.2047,     10987372.6619,      10764852.663,
     10551296.7957,     10346171.7177,     10148985.6613,
     9959284.45525,     9776647.99501,     9600687.10337,
     9431040.73252,     9267373.46466,     9109373.27455,
     8956749.52272,     8809231.15191,     8666565.06313,
     8528514.65085,     8394858.47935,     8265389.08465,
     8139911.88833,     7900214.37602,      7674431.7041,
        7461381.19,      7260010.7574,     7069381.37083,
     6888652.23304,      6717068.2507,     6553949.37303,
      6398681.4844,     6250708.59292,     6109526.10467,
     5974675.01146,     5845736.85046,     5722329.31864,
      5604102.4449,     5490735.23866,     5381932.74725,
       5277423.465,     5176957.04587,     5080302.27894,
     4987245.29223,     4897587.95517,     4811146.45477,
     4727750.02357,     4647239.80112,     4569467.81256,
     4494296.05084,     4421595.65018,     4351246.14058,
     4283134.77418,     4217155.91547,     4153210.48844,
     4031053.45636,     3915984.28239,      3807400.7263,
     3704767.03978,     3607605.02544,      3515486.5018,
     3428026.92286,     3344879.95083,     3265732.81994,
     3190302.35977,     3118331.57125,     3049586.66763,
     2983854.50841,      2920940.3665,     2860665.97936,
     2802867.84264,     2747395.71194,     2694111.28361,
     2642887.03006,     2593605.16898,     2546156.74866,
     2500440.83456,     2456363.78429,     2413838.59984,
     2372784.34781,     2333125.63928,     2294792.16242,
     2257718.26156,     2221842.55759,     2187107.60483,
     2153459.58052,     2120848.00323,     2058547.45994,
     1999859.79463,     1944478.13147,       1892129.468,
     1842570.12107,     1795581.88886,     1750968.80087,
     1708554.35336,     1668179.14769,     1629698.86458,
      1592982.5198,     1557910.95684,     1524375.53953,
     1492277.01457,     1461524.51863,     1432034.70898,
     1403731.00015,     1376542.89185,     1350405.37546,
     1325258.40889,     1301046.45049,     1277718.04459,
     1255225.45204,     1233524.32016,     1212573.38728,
     1192334.21771,     1172770.96356,     1153850.15023,
     1135540.48296,     1117812.67195,     1100639.27416,
     1083994.54978,     1052195.90814,     1022240.22693,
     993971.022577,     967249.079973,     941950.131258,
     917962.899577,     895187.442482,     873533.742731,
     852920.504418,     833274.120331,     814527.782755,
     796620.715006,     779497.504963,     763107.525152,
     747404.426526,     732345.695244,     717892.263476,
     704008.166703,     690660.241153,     677817.855988,
     665452.675682,     653538.448664,     642050.818932,
     630967.157747,      620266.41297,     609928.973904,
     599936.549831,     590272.060629,     580919.538101,
     571864.036815,     563091.553384,     554588.953291,
     538344.816665,     523041.548206,      508599.29108,
     494946.998456,     482021.249542,     469765.251724,
     458127.995525,     447063.535752,     436530.377359,
     426490.948641,     416911.147609,     407759.949924,
     399009.068882,     390632.659531,     382607.060391,
     374910.567309,     367523.234875,     360426.701557,
     353604.035312,     347039.596927,      340718.91876,
     334628.596889,      328756.19497,     323090.158348,
     317619.737168,     312334.917401,     307226.358847,
     302285.339317,     297503.704266,     292873.821293,
     288388.538937,     284041.149331,     275735.234396,
     267910.070584,     260524.871062,     253543.347354,
     246933.104947,      240665.13389,     234713.377391,
     229054.364815,     223666.898142,        218531.783,
     213631.597053,     208950.489813,     204474.009027,
     200188.949594,     196083.221672,     192145.735205,
     188366.298503,     184735.528953,     181244.774164,
      177886.04218,     174651.939551,     171535.616247,
     168530.716557,     165631.335218,     162831.978148,
     160127.527208,     157513.208541,     154984.564055,
     152537.425701,     150167.892222,     147872.308115,
     145647.244544,     141395.994603,     137390.624505,
     133610.283442,     130036.419482,     126652.470773,
     123443.605264,     120396.500282,     117499.155002,
     114740.730226,      112111.41094,     109602.287952,
     107205.255591,     104912.922975,     102718.536798,
     100615.913924,     98599.3823607,     96663.7294285,
      94804.156116,     93016.2367777,     91295.8834589,
     89639.3142378,     88043.0250685,     86503.7646782,
     85018.5121417,     83584.4568033,      82198.980265,
     80859.6401976,     79564.1557618,     78310.3944557,
     77096.3602287,     75920.1827217,     74780.1075103,
     72601.7735874,     70549.3244291,     68612.0902638,
     66780.5772822,     65046.3097201,     63401.6967477,
     61839.9197339,     60354.8363331,     58940.8985361,
     57593.0823673,     56306.8273413,     55077.9841309,
     53902.7691783,     52777.7251933,     51699.6866696,
     50665.7496879,     49673.2453971,      48719.716661,
     47802.8974376,     46920.6945257,     46071.1713659,
     45252.5336325,     44463.1163889,     43701.3726106,
     42965.8629118,     42255.2463268,     41568.2720255,
     40903.7718523,     40260.6535941,     39637.8948977,
     39034.5377622,     38449.6835453,     37332.1592893,
     36279.1577368,     35285.2130631,     34345.4616221,
     33455.5611057,     32611.6223983,     31810.1518596,
     31048.0022185,     30322.3306142,     29630.5625994,
     28970.3611385,     28339.5998107,     27736.3395662,
     27158.8084976,     26605.3841795,     26074.5782033,
     25565.0225957,     25075.4578572,     24604.7224005,
     24151.7432007,     23715.5274976,     23295.1554155,
     22889.7733832,     22498.5882546,     22120.8620455,
     21755.9072116,      21403.082403,     21061.7886423,
     20731.4658758,     20411.5898558,     20101.6693198,
     19801.2434311,     19227.1706446,     18686.2100916,
     18175.5584758,     17692.7212135,     17235.4710025,
     16801.8128973,     16389.9547259,     15998.2819199,
     15625.3360073,     15269.7961595,      14930.463299,
     14606.2463622,     14296.1503821,     13999.2661181,
     13714.7610004,      13441.871201,     13179.8946694,
      12928.184999,     12686.1460114,     12453.2269616,
     12228.9182827,     12012.7478009,     11804.2773608,
     11603.0998103,     11408.8363005,     11221.1338643,
     11039.6632386,     10864.1169034,     10694.2073132,
     10529.6652976,     10370.2386141,      10215.690635,
     9920.35530902,     9642.03589933,     9379.29279296,
     9130.84483975,     8895.54809344,      8672.3778898,
     8460.41366569,     8258.82604172,     8066.86578413,
     7883.85433402,     7709.17565017,     7542.26915762,
      7382.6236306,     7229.77186868,     7083.28604825,
      6942.7736517,      6807.8738919,     6678.25456313,
     6553.60926028,     6433.65491683,     6318.12961989,
     6206.79066653,     6099.41283073,     5995.78681498,
     5895.71786375,     5799.02451956,     5705.53750473,
     5615.09871424,     5527.56030707,     5442.78388486,
     5360.63974837,     5281.00622313,     5128.82081305,
     4985.39281878,     4849.98187786,     4721.92907957,
     4600.64604199,     4485.60570297,     4376.33451913,
     4272.40582769,     4173.43417369,     4079.07044241,
     3988.99766684,     3902.92740311,     3820.59658617,
     3741.76479301,     3666.21185297,     3593.73575486,
      3524.1508087,     3457.28602662,     3392.98369304,
     3331.09809869,     3271.49441705,     3214.04770487,
     3158.64201089,     3105.16957952,     3053.53013775,
     3003.63025539,     2955.38276983,     2908.70626808,
     2863.52461931,     2819.76655239,     2777.36527337,
     2736.25811858,     2657.69431628,      2583.6451563,
     2513.72928085,     2447.60726343,     2384.97598811,
     2325.56391052,      2269.1270432,     2215.44553905,
     2164.32077155,     2115.57282931,     2069.03835784,
     2024.56869368,     1982.02824566,     1941.29308582,
     1902.24971907,     1864.79400552,     1828.83021394,
     1794.27018797,     1761.03260977,     1729.04234809,
     1698.22987957,     1668.53077394,     1639.88523494,
     1612.23769017,     1585.53642374,     1559.73324667,
     1534.78320059,     1510.64429085,     1487.27724571,
     1464.64529866,      1442.7139913,     1421.45099465,
     1380.81030481,     1342.50136932,     1306.32740443,
      1272.1132493,     1239.70246899,     1208.95491167,
     1179.74463906,      1151.9581651,     1125.49295061,
     1100.25611146,     1076.16330573,     1053.13777154,
     1031.10949225,     1010.01446977,     989.794089986,
     970.394566925,     951.766454485,     933.864216342,
     916.645846088,     900.072530873,     884.108352851,
     868.720023553,     853.876647032,     839.549508182,
     825.711883191,      812.33886945,     799.407232626,
     786.895268923,     774.782680783,     763.050464528,
     751.680808624,     740.657001406,     719.585090388,
     699.720033366,     680.960086838,     663.214678935,
     646.402913465,     630.452308433,     615.297727189,
     600.880468722,     587.147490089,     574.050739088,
     561.546579359,      549.59529327,     538.160650604,
     527.209533055,     516.711606313,     506.639032813,
      496.96621939,     487.669594983,     478.727414292,
     470.119583902,     461.827507957,     453.833950828,
     446.122914658,     438.679529921,     431.489957406,
     424.541300267,     417.821524944,     411.319389924,
     405.024381463,     398.926655468,      393.01698488,
     387.286711944,     376.332318284,     366.004045462,
     356.249171211,     347.020757519,     338.276876563,
     329.979957895,     322.096235284,      314.59527586,
     307.449577629,     300.634224017,     294.126586238,
     287.906065913,      281.95387174,     276.252825046,
     270.787189968,     265.542524683,      260.50555071,
     255.664037759,     251.006702025,     246.523116109,
     242.203629065,     238.039295241,     234.021810832,
     230.143457166,     226.397049911,     222.775893491,
     219.273740104,     215.884752796,      212.60347214,
     209.424786113,     206.343902815,     203.356325724,
     197.644448118,      192.25828345,      187.17044241,
     182.356537811,     177.794783117,     173.465653828,
     169.351600521,     165.436804568,     161.706969302,
     158.149140755,     154.751553204,     151.503495587,
     148.395195583,     145.417718676,     142.562879997,
     139.823167083,     137.191672013,     134.662031621,
     132.228374673,     129.885275086,     127.627710402,
     125.451024827,     123.350896269,     121.323306878,
     119.364516656,     117.471039772,     115.639623265,
     113.867227861,     112.151010651,     110.488309437,
     108.876628554,     107.313626007,     104.324987222,
     101.506311917,     98.8433301246,     96.3233343285,
     93.9349706793,     91.6680628859,     89.5134629529,
     87.4629241036,     85.5089921298,     83.6449121192,
     81.8645480786,     80.1623134152,     78.5331106002,
     76.9722786296,     75.4755471284,     74.0389961376,
     72.6590207791,     71.3323001202,     70.0557696658,
     68.8265969947,     67.6421601273,     66.5000282736,
     65.3979446612,     64.3338111858,     63.3056746605,
     62.3117144747,     61.3502314957,     60.4196380687,
     59.5184489908,     58.6452733506,     57.7988071362,
     56.9778265301,     55.4077918373,     53.9267643608,
     52.5272846644,     51.2027090519,     49.9471006653,
     48.7551376125,      47.622035089,     46.5434790675,
     45.5155695954,     44.5347721114,     43.5978754878,
     42.7019557373,     41.8443445089,     41.0226016526,
     40.2344912502,     39.4779606118,      38.751121818,
     38.0522354538,     37.3796962368,     36.7320202863,
     36.1078338186,     35.5058630854,     34.9249253981,
     34.3639211041,     33.8218263982,     33.2976868701,
     32.7906117014,     32.2997684362,     31.8243782605,
     31.3637117346,      30.917084926,     30.4838559023,
     29.6552146411,     28.8733782988,      28.134432563,
     27.4348905633,     26.7716358691,     26.1418743927,
     25.5430936131,     24.9730278502,     24.4296285677,
     23.9110388709,     23.4155715259,     22.9416899422,
     22.4879916639,     22.0531939906,     21.6361214127,
     21.2356946011,     20.8509207292,     20.4808849434,
     20.1247428261,     19.7817137183,     19.4510747894,
      19.132155759,      18.824334187,     18.5270312637,
     18.2397080369,     17.9618620255,     17.6930241733,
     17.4327561035,     17.1806476409,     16.9363145704,
     16.6993966065,     16.4695555516,     16.0298519219,
     15.6148798725,     15.2225761964,     14.8511026109,
     14.4988157995,     14.1642421305,     13.8460562183,
     13.5430626642,     13.2541804371,       12.97842946,
     12.7149190454,     12.4628378888,     12.2214453808,
     11.9900640388,     11.7680728923,     11.5549016866,
     11.3500257858,     11.1529616813,     10.9632630215,
     10.7805170944,     10.6043417028,     10.4343823832,
     10.2703099236,     10.1118181445,     9.95862191002,
     9.81045534239,      9.6670702159,     9.52823450908,
      9.3937310977,     9.26335657253,     9.13692016828,
     9.01424279167,     8.77950188724,      8.5579029016,
     8.34835265336,     8.14987690272,     7.96160453251,
     7.78275419548,     7.61262298955,     7.45057681007,
     7.29604209573,     7.14849873828,     7.00747396878,
     6.87253706654,     6.74329476426,      6.6193872446,
     6.50048464099,     6.38628396998,     6.27650643422,
     6.17089504471,     6.06921251895,     5.97123941835,
     5.87677249363,     5.78562321153,     5.69761644002,
     5.61258927241,     5.53038997362,     5.45087703381,
     5.37391831699,     5.29939029358,     5.22717734731,
     5.15717114836,     5.08927008528,     5.02337874941,
     4.89727186893,     4.77819450111,     4.66556389779,
      4.5588605707,     4.45761989234,     4.36142500459,
     4.26990080294,     4.18270881023,     4.09954278998,
     4.02012497734,      3.9442028284,     3.87154620636,
      3.8019449374,     3.73520668068,     3.67115506623,
     3.60962806223,      3.5504765392,     3.49356300388,
     3.43876047989,     3.38595151559,     3.33502730255,
     3.28588689046,      3.2384364864,     3.19258882806,
     3.14826262191,     3.10538203853,     3.06387625858,
      3.0236790634,     2.98472846526,      2.9469663728,
     2.91033828793,      2.8747930306,      2.8067613898,
     2.74251940505,     2.68175424311,     2.62418694186,
     2.56956792209,     2.51767319702,     2.46830115574,
     2.42126982168,     2.37641450585,        2.33358579,
     2.29264778675,      2.2534766331,     2.21595918157,
     2.17999185932,     2.14547967046,     2.11233532111,
     2.08047844974,      2.0498349484,     2.02033636247,
     1.99191935846,      1.9645252511,     1.93809958206,
     1.91259174386,      1.8879546434,     1.86414440028,
     1.84112007579,     1.81884342903,     1.79727869694,
     1.77639239563,     1.75615314058,     1.73653148369,
     1.71749976531,     1.68110365251,     1.64677447223,
     1.61434304574,     1.58365847132,     1.55458570812,
      1.5270035349,     1.50080281743,     1.47588503118,
     1.45216099646,     1.42954979098,     1.40797781157,
     1.38737796141,     1.36768894386,     1.34885464657,
     1.33082360294,     1.31354851955,      1.2969858605,
     1.28109548066,      1.2658403013,     1.25118602242,
     1.23710086713,     1.22355535382,     1.21052209274,
     1.19797560406,     1.18589215458,     1.17424961117,
     1.16302730872,     1.15220593111,     1.14176740369,
      1.1316947959,     1.12197223308,     1.11258481634,
     1.09476027388,     1.07811888208,     1.06256992542,
     1.04803270312,     1.03443522136,     1.02171308882,
     1.00980857969,    0.998669835485,    0.988250182301,
    0.978507544678,     0.96940394074,    0.960905045915,
    0.952979814867,    0.945600153008,    0.938740630449,
    0.932378232401,    0.926492141047,    0.921063544663,
    0.916075470469,    0.911512638216,    0.907361331987,
    0.903609288065,    0.900245597058,    0.897260618748,
    0.894645908353,    0.892394153106,    0.890499118227,
    0.888955601518,    0.887759395947,    0.886907259694,
    0.886396893265,                 0
};

   int n;
   lebool negatif;
   Uvalue uval, wval;
   double v, x;

   util_Assert (u >= 0.0, "finv_Normal2:   u < 0");
   util_Assert (u <= 1.0, "finv_Normal2:   u > 1");

   if (u >= 1.0) {
      util_Warning (1, "finv_Normal2:   u = 1");
      return fdist_XBIG;
   }
   if (u <= 0.0) {
      util_Warning (1, "finv_Normal2:   u = 0");
      return -fdist_XBIG;
   }
   if (u < 0.5) {
      negatif = TRUE;
      uval.u = 2.0*u;
   } else {
      negatif = FALSE;
      uval.u = 2.0*(1.0 - u);
   }
   wval.i = uval.i & 2147221504;
   n = (wval.i >> 18) - 3040;
   /*   util_Assert (n <= 1024, "n >= 1024");  */
   if (n < 0) {
      if (negatif) {
         util_Warning (1, "finv_Normal2:   u --> 0");
         return -fdist_XBIG;
      } else {
         util_Warning (1, "finv_Normal2:   u --> 1");
         return fdist_XBIG;
      }
   }
   v = (uval.u - wval.u) * B[n];
   x = A[n] - v * (1.414214 - v * (A[n] - 0.4714045 * (1.0 +
            2.0 * A[n] * A[n]) * v));
   if (negatif)
      return -x;
   else 
      return x;
}


/*=========================================================================*/

double finv_LogNormal (double mu, double sigma, double u)
{
   double t, v;
   util_Assert (sigma > 0.0, "finv_LogNormal:  sigma  <= 0");
   util_Assert (u >= 0.0 && u <= 1.0, "finv_LogNormal:   u not in [0, 1]");
   if (DBL_EPSILON >= 1.0 - u) {
      util_Warning (1, "finv_LogNormal:   u --> 1");
      return fdist_XINF;
   }
   if (u <= 0.0)
      return 0.0;
   t = finv_Normal1 (u);
   v = mu + sigma * t;
   if ((t >= fdist_XBIGM) || (v >= DBL_MAX_EXP * num_Ln2)) {
      util_Warning (1, "finv_LogNormal:   u --> 1");
      return fdist_XINF;
   }
   if ((t <= -fdist_XBIGM) || (v <= -DBL_MAX_EXP * num_Ln2)) {
      util_Warning (1, "finv_LogNormal:   u --> 0");
      return 0.0;
   }
   return exp (mu + sigma * t);
}


/*=========================================================================*/

double finv_JohnsonSB (double alpha, double beta, double a, double b,
                       double u)
{
   double v, t;
   util_Assert (beta > 0.0, "finv_JohnsonSB:  beta  <= 0");
   util_Assert (b > a, "finv_JohnsonSB:  b  <= a");
   util_Assert (u >= 0.0 && u <= 1.0, "finv_JohnsonSB:   u not in [0, 1]");

   if (u >= 1.0)                  /* if u == 1, in fact */
      return b;
   if (u <= 0.0)                  /* if u == 0, in fact */
      return a;

   t = finv_Normal1 (u);
   v = (t - alpha) / beta;
   if ((t >= fdist_XBIGM) || (v >= DBL_MAX_EXP * num_Ln2)) {
      util_Warning (1, "finv_JohnsonSB:   u --> 1");
      return b;
   }
   if ((t <= -fdist_XBIGM) || (v <= -DBL_MAX_EXP * num_Ln2)) {
      util_Warning (1, "finv_JohnsonSB:   u --> 0");
      return a;
   }
   v = exp (v);
   return (a + b * v) / (1 + v);
}


/*=========================================================================*/

double finv_JohnsonSU (double alpha, double beta, double u)
{
   double t, v;
   util_Assert (beta > 0.0, "finv_JohnsonSU:  beta  <= 0");
   util_Assert (u >= 0.0 && u <= 1.0, "finv_JohnsonSU:   u not in [0, 1]");
   if (u >= 1.0) {
      util_Warning (1, "finv_JohnsonSU:   u = 1");
      return fdist_XINF;
   }
   if (u <= 0.0) {
      util_Warning (1, "finv_JohnsonSU:   u = 0");
      return -fdist_XINF;
   }

   t = finv_Normal1 (u);
   if (t >= fdist_XBIGM) {
      util_Warning (1, "finv_JohnsonSU:   u --> 1");
      return fdist_XINF;
   }
   if (t <= -fdist_XBIGM) {
      util_Warning (1, "finv_JohnsonSU:   u --> 0");
      return -fdist_XINF;
   }

   v = (t - alpha) / beta;
   if (v >= num_Ln2 * DBL_MAX_EXP) {
      util_Warning (1, "finv_JohnsonSU:   v > Ln 2 * DBL_MAX_EXP");
      return fdist_XINF;
   }
   if (v <= num_Ln2 * DBL_MIN_EXP) {
      util_Warning (1, "finv_JohnsonSU:   v < Ln 2 * DBL_MIN_EXP");
      return -fdist_XINF;
   }
   return sinh (v);
}


/**************************************************************************/

double finv_ChiSquare1 (long k, double u)
/*
 * Returns an approximation of the inverse of Chi square cdf
 * with k degrees of freedom.
 * As in Figure L.24 of P.Bratley, B.L.Fox, and L.E.Schrage.
 *     A Guide to Simulation Springer-Verlag,
 *	   New York, second edition, 1987.
 */
{
   const double sqp5 = 0.70710678118654752440;
   const double dwarf = 0.1e-15;
   double Z, arg, zsq, ch, sqdf;

   util_Assert (u >= 0.0 && u <= 1.0, "finv_ChiSquare1:   u not in [0, 1]");
   if (u <= 0.0)
      return 0.0;
   if (u >= 1.0) {
      util_Warning (1, "finv_ChiSquare1:   u = 1");
      return k * fdist_XBIG;
   }
   if (k == 1) {
      Z = finv_Normal1 ((1.0 + u) / 2);
      return (Z * Z);
   } else {
      if (k == 2) {
         arg = 1.0 - u;
         if (arg < dwarf)
            arg = dwarf;
         return (-log (arg) * 2.0);
      } else {
         if (u > dwarf) {
            Z = finv_Normal1 (u);
            sqdf = sqrt ((double) k);
            zsq = Z * Z;
            ch = -(((3753.0 * zsq + 4353.0) * zsq - 289517.0) * zsq -
               289717.0) * Z * sqp5 / 9185400;
            ch = ch / sqdf + (((12.0 * zsq - 243.0) * zsq - 923.0) * zsq +
               1472.0) / 25515.0;
            ch = ch / sqdf + ((9.0 * zsq + 256.0) * zsq - 433.0) * Z * sqp5 /
               4860;
            ch = ch / sqdf - ((6.0 * zsq + 14.0) * zsq - 32.0) / 405.0;
            ch = ch / sqdf + (zsq - 7.0) * Z * sqp5 / 9;
            ch = ch / sqdf + 2.0 * (zsq - 1.0) / 3.0;
            ch = ch / sqdf + Z / sqp5;
            return (k * (ch / sqdf + 1.0));
         } else
            return 0.0;
      }
   }
}


/**************************************************************************/

double finv_ChiSquare2 (long k, double u)
/*
 * Returns an approximation of the inverse of Chi square cdf
 * with k degrees of freedom.
 * As in Figure L.23 of P.Bratley, B.L.Fox, and L.E.Schrage.
 *    A Guide to Simulation Springer-Verlag,
 *    New York, second edition, 1987.
 */
{
   const double E = 0.5e-5;       /* Precision of this approximation */
   const double AA = 0.6931471805;
   double A, XX, X, C, G, CH, Q, P1, P2, T, B, S1, S2, S3, S4, S5, S6;

   util_Assert (u >= 0.0 && u <= 1.0, "finv_ChiSquare2:   u not in [0, 1]");
   if (k < 0 || u <= 0.000002)
      return 0.0;
   if (u >= 1.0) {
      util_Warning (1, "finv_ChiSquare2:   u = 1");
      return k * fdist_XBIG;
   }
   if (u >= 0.999998)
      return (k + 4.0 * sqrt (2.0 * k));

   G = num2_LnGamma (k / 2.0);
   XX = 0.5 * k;
   C = XX - 1.0;

   if (k >= -1.24 * log (u)) {
      if (k <= 0.32) {
         CH = 0.4;
         A = num2_log1p (-u);
         Q = CH;
         P1 = 1.0 + CH * (4.67 + CH);
         P2 = CH * (6.73 + CH * (6.66 + CH));
         T = -0.5 + (4.67 + 2.0 * CH) / P1 - (6.73 +
            CH * (13.32 + 3.0 * CH)) / P2;
         CH = CH - (1.0 - exp (A + G + 0.5 * CH + C * AA) * P2 / P1) / T;
         while (fabs (Q / CH - 1.0) - 0.01 > 0) {
            Q = CH;
            P1 = 1.0 + CH * (4.67 + CH);
            P2 = CH * (6.73 + CH * (6.66 + CH));
            T = -0.5 + (4.67 + 2.0 * CH) / P1 - (6.73 + CH * (13.32 +
                  3.0 * CH)) / P2;
            CH = CH - (1.0 - exp (A + G + 0.5 * CH + C * AA) * P2 / P1) / T;
         }
      } else {
         X = finv_Normal1 (u);
         P1 = 0.222222 / k;
         CH = k * pow ((X * sqrt (P1) + 1.0 - P1), 3.0);
         if (CH > 2.2 * k + 6.0)
            CH = -2.0 * (num2_log1p (-u) - C * log (0.5 * CH) + G);
      }
   } else {
      CH = pow ((u * XX * exp (G + XX * AA)), (1.0 / XX));
      if (CH - E < 0)
         return CH;
   }

   Q = CH;
   P1 = 0.5 * CH;
   P2 = u - fdist_Gamma (XX, 5, P1);
   if (fdist_Gamma (XX, 5, P1) == -1.0)
      return -1.0;

   T = P2 * exp (XX * AA + G + P1 - C * log (CH));
   B = T / CH;
   A = 0.5 * T - B * C;
   S1 = (210.0 + A * (140.0 +
         A * (105.0 + A * (84.0 + A * (70.0 + 60.0 * A))))) / 420.0;
   S2 = (420.0 + A * (735.0 + A * (966.0 + A * (1141.0 + 1278.0 * A))))
      / 2520.0;
   S3 = (210.0 + A * (462.0 + A * (707.0 + 932.0 * A))) / 2520.0;
   S4 = (252.0 + A * (672.0 + 1182.0 * A) +
      C * (294.0 + A * (889.0 + 1740.0 * A))) / 5040.0;
   S5 = (84.0 + 264.0 * A + C * (175.0 + 606.0 * A)) / 2520.0;
   S6 = (120.0 + C * (346.0 + 127.0 * C)) / 5040.0;
   CH = CH + T * (1.0 + 0.5 * T * S1 - B * C * (S1 - B * (S2 -
            B * (S3 - B * (S4 - B * (S5 - B * S6))))));

   while (fabs (Q / CH - 1.0) > E) {
      Q = CH;
      P1 = 0.5 * CH;
      P2 = u - fdist_Gamma (XX, 5, P1);
      if (fdist_Gamma (XX, 5, P1) == -1.0)
         return -1.0;

      T = P2 * exp (XX * AA + G + P1 - C * log (CH));
      B = T / CH;
      A = 0.5 * T - B * C;
      S1 = (210.0 + A * (140.0 + A * (105.0 + A * (84.0 +
                  A * (70.0 + 60.0 * A))))) / 420.0;
      S2 = (420.0 + A * (735.0 + A * (966.0 + A * (1141.0 +
                  1278.0 * A)))) / 2520.0;
      S3 = (210.0 + A * (462.0 + A * (707.0 + 932.0 * A))) / 2520.0;
      S4 = (252.0 + A * (672.0 + 1182.0 * A) +
         C * (294.0 + A * (889.0 + 1740.0 * A))) / 5040.0;
      S5 = (84.0 + 264.0 * A + C * (175.0 + 606.0 * A)) / 2520.0;
      S6 = (120.0 + C * (346.0 + 127.0 * C)) / 5040.0;
      CH = CH + T * (1.0 + 0.5 * T * S1 - B * C * (S1 - B * (S2 -
               B * (S3 - B * (S4 - B * (S5 - B * S6))))));
   }

   return CH;
}


/*=========================================================================*/

double finv_Student (long k, double u)
/*
 * Returns an approximation of the inverse of a the Student Distribution
 *     with k degrees of freedom.
 * As in Figure L.28 of P.Bratley, B.L.Fox, and L.E.Schrage.
 *       A Guide to Simulation Springer-Verlag,
 *	   New York, second edition, 1987.
 */
{
   const double dwarf = 1.e-307;
   const double piovr2 = 1.5707963268;
   double arg, T, vtemp, ux2, p2tail, X, C, Y2, Y;
   static double W[5] = { 0 };

   if (k < 1)
      util_Error ("Calling finv_Student with k < 1");
   if (u < 0. || u > 1.)
      util_Error ("Calling finv_Student with u < 0 or u > 1");

   /* For 1 degree of freedom we use a special method */
   /* of Standard Cauchy Inversion */
   if (k == 1) {
      arg = num_Pi * (1.0 - u);
      if (sin (arg) > dwarf)
         return (cos (arg) / sin (arg));
      else
         return (cos (arg) / dwarf);
   }

   /* There is also a special method for 2 degrees of freedom */
   if (k == 2) {
      if (u <= 0.5)
         T = 2.0 * u;
      else
         T = 2.0 * (1.0 - u);
      if (T <= dwarf)
         T = dwarf;
      vtemp = sqrt ((2.0 / (T * (2.0 - T))) - 2.0);
      if (u <= 0.5)
         return (-vtemp);
      else
         return (vtemp);
   }

   if (u <= dwarf || (1.0 - u) <= dwarf) {
      T = 10.0e30;
      if (u < 0.5)
         return (-T);
      else
         return (T);
   }

   ux2 = u * 2.0;

   if (ux2 < 2.0 - ux2)
      p2tail = ux2;
   else
      p2tail = 2.0 - ux2;

   W[2] = 1.0 / (k - 0.5);
   W[1] = 48.0 / (W[2] * W[2]);
   W[3] = ((20700.0 * W[2] / W[1] - 98.0) * W[2] - 16.0) * W[2] + 96.36;
   W[4] = ((94.5 / (W[1] + W[3]) - 3.0) / W[1] + 1.0) *
      sqrt (W[2] * piovr2) * k;

   X = W[4] * p2tail;
   C = W[3];
   Y = pow (X, 2.0 / k);

   if (Y <= (0.05 + W[2])) {
      Y2 = (k + 6.0) / (k + Y) - 0.089 * W[4] - 0.822;
      Y2 = Y2 * (k + 2.0) * 3.0;
      Y2 = (1.0 / Y2 + 0.5 / (k + 4.0)) * Y - 1.0;
      Y2 = Y2 * (k + 1.0) / (k + 2.0);
      Y = Y2 + 1.0 / Y;
   } else {
      X = finv_Normal1 (p2tail * 0.5);
      Y = X * X;
      if (k < 5)
         C = C + 0.3 * (k - 4.5) * (X + 0.6);
      C = (((0.05 * W[4] * X - 5.0) * X - 7.0) * X - 2.0) * X + W[1] + C;
      Y = (((((0.4 * Y + 6.3) * Y + 36.0) * Y + 94.5) / C - Y - 3.0) / W[1] +
         1.0) * X;
      Y = W[2] * Y * Y;
      if (Y <= 0.002)
         Y = 0.5 * Y * Y + Y;
      else
         Y = exp (Y) - 1.0;
   }

   T = sqrt (k * Y);
   if (u < 0.5)
      return (-T);
   else
      return (T);
}


/*=========================================================================*/
/* Constants for finv_BetaSymmetric */

#define EPSSINGLE  1.0e-5         /* Limited Tolerance */
#define EPSILON  1.0e-15          /* Tolerance double precision */
#define EPSBETA  1.0e-10          /* < 0.75 sqrt(DBL_EPSILON) */
#define MAXI  11                  /* Max number of Newton's iterations */
#define MAXIB 50                  /* Max number of bisection iterations */
#define MAXJ  2000                /* Max number of terms in series */
#define LOG2  0.6931471805599453  /* Ln(2) */
#define LOG4  1.3862943611198906  /* Ln(4) */
#define PI_2  1.5707963267948966  /* Pi / 2 */
#define SQPI_2  0.88622692545275801   /* Sqrt(Pi) / 2 */
#define ALIM1    100000.0         /* Limiting alpha for normal approx. */



/*------------------------------------------------------------------------*/

static double inverse1 (
   double alpha,                  /* Shape parameter */
   double bu                      /* u * Beta(alpha, alpha) */
   )
/* 
 * This method is used for alpha < 1 and x close to 0.
 */
{
   int i, j;
   double x, xnew;
   double poc, sum, term;

   x = pow (bu * alpha, 1.0 / alpha); /* First term of series */

   /* If T1/T0 is very small, neglect all terms of series except T0 */
   term = alpha * (1.0 - alpha) * x / (1.0 + alpha);
   if (term < EPSILON)
      return x;

   x = bu * alpha / (1.0 + term);
   xnew = pow (x, 1.0 / alpha);   /* Starting point of Newton's iterates */
   i = 0;
   do {
      ++i;
      x = xnew;

      /* Compute the series for F(x) */
      poc = 1.0;
      sum = 1.0 / alpha;
      j = 1;
      do {
         poc *= x * (j - alpha) / j;
         term = poc / (j + alpha);
         sum += term;
         ++j;
      } while ((term > sum * EPSILON) && (j < MAXJ));
      sum *= pow (x, alpha);

      /* Newton's method */
      term = (sum - bu) * pow (x * (1.0 - x), 1.0 - alpha);
      xnew = x - term;

   } while ((fabs (term) > EPSILON) && (i <= MAXI));

   return xnew;
}


/*------------------------------------------------------------------------*/

static double inverse2 (
   double alpha,                  /* Shape parameter */
   double w                       /* (0.5 - u)B/pow(4, 1 - alpha) */
   )
/* 
 * This method is used for alpha < 1 and x close to 1/2.
 */
{
   int i, j;
   double term, sum, y, ynew, z;
   double poc;

   term = (1.0 - alpha) * w * w * 4.0 / 3.0;
   /* If T1/T0 is very small, neglect all terms of series except T0 */
   if (term < EPSILON)
      return 0.5 - w;

   ynew = w / (1 + term);         /* Starting point of Newton's iterates */
   i = 0;
   do {
      ++i;
      y = ynew;
      z = 4.0 * y * y;

      /* Compute the series for G(y) */
      poc = sum = 1.0;
      j = 1;
      do {
         poc *= z * (j - alpha) / j;
         term = poc / (2 * j + 1);
         sum += term;
         ++j;
      } while ((term > sum * EPSILON) && (j < MAXJ));
      sum *= y;

      /* Newton's method */
      term = (sum - w) * pow (1.0 - z, 1.0 - alpha);
      ynew = y - term;

   } while ((fabs (term) > EPSILON) && (i <= MAXI));

   return 0.5 - ynew;
}


/*------------------------------------------------------------------------*/

static double bisect (
   double alpha,                  /* Shape parameter */
   double logBua,                 /* Ln(alpha * u * Beta(alpha, alpha)) */
   double a,                      /* x is presumed in [a, b] */
   double b)
/*
 * Bisection method to find a solution x.  This method is used for alpha > 1 and 
 * u very close to 0. It will rarely be called.
 */
{
   int i, j;
   double z, sum, term;
   double x, xprev;

   if (a >= 0.5 || a > b) {
      a = 0.0;
      b = 0.5;
   }

   x = 0.5 * (a + b);
   i = 0;
   do {
      ++i;
      z = -x / (1 - x);

      /* Compute the series for F(x) */
      sum = term = 1.0;
      j = 1;
      do {
         term *= z * (j - alpha) / (j + alpha);
         sum += term;
         ++j;
      } while ((fabs (term/sum) > EPSILON) && (j < MAXJ));
      sum *= x;

      /* Bisection method */
      z = logBua - (alpha - 1.0) * log (x * (1.0 - x));
      if (z > log(sum))
         a = x;
      else
         b = x;
      xprev = x;
      x = 0.5 * (a + b);

   } while ((fabs(xprev - x) > EPSILON) && (i < MAXIB));

   return x;
}


/*------------------------------------------------------------------------*/

static double inverse3 (
   double alpha,                  /* Shape parameter */
   double logBua                  /* Ln(alpha * u * Beta(alpha, alpha)) */
   )
/* 
 * This method is used for alpha > 1 and x close to 0.
 */
{
   int i, j;
   double z, x, w, xnew, sum, term;
   double eps = EPSSINGLE;
   /* For alpha <= 100000 and u < 1.0/(2.5 + 2.25*sqrt(alpha)), X0 is always
      to the right of the solution, so Newton is certain to converge. */
   const double X0 = 0.497;

   /* Compute starting point of Newton's iterates */
   w = logBua / alpha;
   x = exp (w);
   term = (num2_log1p(-x) + logBua) / alpha;
   z = exp (term);
   if (z >= 0.25)
       xnew = X0;
   else if (z > 1.0e-6)
       xnew = (1.0 - sqrt(1.0 - 4.0*z)) / 2.0;
   else
       xnew = z;

   i = 0;
   do {
      ++i;
      if (xnew >= 0.5)
         xnew = X0;
      x = xnew;

      w = logBua - (alpha - 1.0) * log (x * (1.0 - x));
      if (fabs (w) >= DBL_MAX_EXP * LOG2) {
         xnew = X0;
         continue;
      }
      w = exp (w);
      z = -x / (1 - x);

      /* Compute the series for F(x) */
      sum = term = 1.0;
      j = 1;
      do {
         term *= z * (j - alpha) / (j + alpha);
         sum += term;
         ++j;
      } while ((fabs (term/sum) > eps) && (j < MAXJ));
      sum *= x;

      /* Newton's method */
      term = (sum - w) / alpha;
      xnew = x - term;
      if (fabs(term) < 32.0*EPSSINGLE)
         eps = EPSILON;

   } while ((fabs (xnew - x) > EPSILON) && (fabs (xnew - x) > sum * EPSILON) &&
            (i <= MAXI));

   /* If Newton has not converged with enough precision, call bisection
      method. It is very slow, but will be called very rarely. */
   if (i >= MAXI && fabs (xnew - x) > 10.0 * EPSILON)
      return bisect (alpha, logBua, 0.0, 0.5);
   return xnew;
}


/*------------------------------------------------------------------------*/

static double inverse4 (
   double alpha,                  /* Shape parameter */
   double logBva                  /* Ln(B) + Ln(1/2 - u) + (alpha - 1)*Ln(4) */
   )
/* 
 * This method is used for alpha > 1 and x close to 1/2.
 */
{
   int i, j;
   double term, sum, y, ynew, z;
   double eps = EPSSINGLE;

   ynew = exp (logBva);           /* Starting point of Newton's iterates */

   i = 0;
   do {
      ++i;
      y = ynew;

      /* Compute the series for G(y) */
      z = 4.0 * y * y;
      term = sum = 1.0;
      j = 1;
      do {
         term *= z * (j + alpha - 0.5) / (0.5 + j);
         sum += term;
         ++j;
      } while ((term > sum * eps) && (j < MAXJ));
      sum *= y * (1.0 - z);

      /* Newton's method */
      term = sum - exp (logBva - (alpha - 1.0) * num2_log1p (-z));
      ynew = y - term;
      if (fabs(term) < 32.0*EPSSINGLE)
         eps = EPSILON;

   } while ((fabs (ynew - y) > EPSILON) && (fabs (ynew - y) > sum*EPSILON)
            && (i <= MAXI));

   return 0.5 - ynew;
}


/*-------------------------------------------------------------------------*/

static double PeizerInverse (double alpha, double u)
{
   /* Inverse of the normal approximation of Peizer and Pratt */
   double t1, t3, xprev;
   const double C2 = alpha - 1.0 / 3.0 + 0.025 / alpha;
   const double z = finv_Normal1 (u);
   double x = 0.5;
   double y = 1.0 - x;
   int i = 0;

   do {
      i++;
      t1 = (2.0 * alpha - 5.0 / 6.0) * x * y;
      t3 = 1.0 - y * fdist_belog (2.0 * x) - x * fdist_belog (2.0 * y);
      xprev = x;
      x = 0.5 + 0.5 * z * sqrt(t1 / t3) / C2;
      y = 1.0 - x;
   } while (i <= MAXI && fabs (x - xprev) > EPSBETA);

   return x;
}


/*------------------------------------------------------------------------*/

double finv_BetaSymmetric (double alpha, double u)
/* 
 * Compute the inverse of the symmetrical beta distribution.
 * Returns a negative value on error, otherwise returns x in [0, 1].
 */
{
   double temp, x, y0;
   int isUpper;                   /* True if u > 0.5 */
   double B;                      /* Beta(alpha, alpha) */
   double C;                      /* 4^(alpha-1) * Beta(alpha, alpha) */
   double logB;                   /* Ln(Beta(alpha, alpha)) */
   double logC;                   /* Ln(4^(alpha-1)*Beta(alpha, alpha)) */
   double logBua;                 /* Ln(alpha * u * Beta(alpha, alpha)) */
   double logBva;        /* Ln(4^(alpha-1)*Beta(alpha,alpha)*(0.5 - u)*/

   util_Assert (alpha > 0.0, "finv_BetaSymmetric:   p <= 0\n");
   util_Assert (u >= 0.0 && u <= 1.0,
                  "finv_BetaSymmetric:   u is not in [0, 1]\n");

   if (u == 0.0)
      return 0.0;
   if (u == 1.0)
      return 1.0;
   if (u == 0.5)
      return 0.5;
   if (alpha == 1.0)
      return u;                   /* alpha = 1 is the uniform law */
   if (alpha == 0.5) {            /* alpha = 1/2 is the arcsin law */
      temp = sin (u * PI_2);
      return temp * temp;
   }

   if (alpha > ALIM1)
      return PeizerInverse (alpha, u);

   if (u > 0.5) {
      u = 1.0 - u;
      isUpper = 1;
   } else
      isUpper = 0;

   fdist_CalcB4 (alpha, &B, &logB, &C, &logC);

   if (alpha <= 1.0) {
      /* First term of integrated series around 1/2 */
      y0 = C * (0.5 - u);
      if (y0 > 0.25)
         x = inverse1 (alpha, B * u);
      else
         x = inverse2 (alpha, y0);

   } else {
      if (u < 1.0 / (2.5 + 2.25*sqrt(alpha))) {
         logBua = logB + log (u * alpha);
         x = inverse3 (alpha, logBua);
      } else {
         logBva = logC - LOG2 + num2_log1p (-2.0*u);
         x = inverse4 (alpha, logBva);
      }
   }

   if (isUpper)
      return 1.0 - x - DBL_EPSILON;
   else
      return x;
}


/*=========================================================================*/

long finv_Geometric (double p, double u)
{
   static double pold = -1.0;
   static double v = 1.0;
   util_Assert (p >= 0.0 && p <= 1.0, "finv_Geometric:   p not in [0, 1]");
   util_Assert (u >= 0.0 && u <= 1.0, "finv_Geometric:   u not in [0, 1]");
   if (p >= 1.0)
      return 0;
   if (u <= 0.0)
      return 0;
   if (u >= 1.0 || p <= 0.0)
      return INT_MAX;
   if (pold != p) {
      pold = p;
      v = num2_log1p (-p);
   }
   return (long) (num2_log1p (-u) / v);
}


/*=========================================================================*/

double finv_GenericC (wdist_CFUNC F, double par[], double U, int prec,
                      int Detail)
{
   const int MaxIter = 100;		   /* Maximum number of iterations */
   const double Epsilon = EpsArray[prec];  /* Absolute precision */
   double x, y = 1.0;
   double yb, ya, xb, xa;
   int i;
   int fini;

   util_Assert (U >= 0.0 && U <= 1.0, "finv_GenericC:   U not in [0, 1]");
   util_Assert (prec <= DBL_DIG, "finv_GenericC:   d too large");
   util_Assert (prec > 0, "finv_GenericC:   d <= 0");
   if (Detail > 0) {
      printf ("---------------------------------------------------------");
      printf ("\n U =  %10.8f\n", U);
   }
   if (U <= 0.0) {
      x = -DBL_MAX;
      if (Detail > 0) {
         printf ("\n            x                   y\n");
         printf ("%17.2g   %17.*f\n", x, prec, U);
      }
      return x;
   }
   if (U >= 1.0) {
      x = DBL_MAX;
      if (Detail > 0) {
         printf ("\n            x                   y\n");
         printf ("%17.2g   %17.*f\n", x, prec, U);
      }
      return x;
   }

   xb = 8.0;                      /* tentative upper limit */
   xa = -8.0;                     /* tentative lower limit */
   yb = F (par, xb);
   ya = F (par, xa);

   util_Assert (yb >= ya, "finv_GenericC:   F is decreasing");

   /* make sure that unknown x is inside [xa, xb] */
   while (yb < U) {
      xa = xb;
      ya = yb;
      xb *= 2.0;
      yb = F (par, xb);
   }
   while (ya > U) {
      xb = xa;
      yb = ya;
      xa *= 2.0;
      ya = F (par, xa);
   }

   yb -= U;
   ya -= U;

   if (Detail > 0) {
      printf ("\niter              xa                   xb           F - u\n");
   }
   fini = 0;
   i = 0;
   while (!fini) {
      if (Detail > 0) {
         printf ("%3d  %18.*g  %18.*g  %14.4g\n", i, prec, xa, prec, xb, y);
      }
      x = (xa + xb) / 2.0;
      y = F (par, x) - U;
      if ((fabs (y) <= Epsilon) ||
          (fabs ((xb - xa)/(x + DBL_EPSILON)) <= Epsilon)) {
         fini = 1;
         if (Detail > 0) {
            printf ("\n                x                     U\n");
            printf ("%20.*g  %18.*g\n", prec, x, prec, y + U);
         }
      } else if (y * ya < 0.0) {
         xb = x;
      } else
         xa = x;
      ++i;
      if (i > MaxIter) {
         printf
            ("\n*******  finv_GenericC:   SEARCH DOES NOT SEEM TO CONVERGE\n");
         fini = 1;
      }
   }
   return x;
}


/*=========================================================================*/

long finv_GenericD1 (fmass_INFO W, double u)
{
   long i, j, k;
   double *F;

   util_Error ("finv_GenericD1 is not finished.");
   util_Assert (W != NULL, "finv_GenericD1:   fmass_INFO is NULL pointer");
   util_Assert (u >= 0.0 && u <= 1.0, "finv_GenericD1:   u not in [0, 1]");

   F = W->cdf;

   /* Remember: the upper part of cdf contains the complementary distribu-
      tion for W->smed < s <= W->smax, and the lower part of cdf the
      distribution for W->smin <= s <= W->smed */

   if (u <= F[W->smed - W->smin]) {
      /* In the lower part of cfd */
      if (u <= F[0])
         return W->smin;
      i = 0;
      j = W->smed - W->smin;
      while (i < j) {
         k = (i + j) / 2;
         if (u > F[k])
            i = k + 1;
         else
            j = k;
      }

   } else {
      /* In the upper part of cdf */
      u = 1 - u;
      if (u < W->cdf[W->smax - W->smin])
         return W->smax;

      i = W->smed - W->smin + 1;
      j = W->smax - W->smin;
      while (i < j) {
         k = (i + j) / 2;
         if (u < F[k])
            i = k + 1;
         else
            j = k;
      }
   }

   return i + W->smin;
}


/*=========================================================================*/
