// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_UI_ASH_ASSISTANT_TEST_FAKE_S3_SERVER_H_
#define CHROME_BROWSER_UI_ASH_ASSISTANT_TEST_FAKE_S3_SERVER_H_

#include <string>

#include "base/macros.h"
#include "base/process/process.h"

namespace chromeos {
namespace assistant {

class PortSelector;

enum class FakeS3Mode {
  // In this mode all S3 requests are forwarded to the S3 server.
  kProxy,
  // In this mode all S3 requests are forwarded to the S3 server, and the
  // responses are recorded.
  kRecord,
  // In this mode all S3 requests are handled by replaying the responses stored
  // while running in |kRecord| mode.
  kReplay,
};

// Class that starts/stops a fake S3 server.
// Note that this will also ensure the Assistant service knows to use the fake
// s3 server.
//
// A valid access token is required if mode is |kProxy| or |kReplay|. See
// |kGenerateTokenInstructions| for information on how to get one.
class FakeS3Server {
 public:
  FakeS3Server();
  ~FakeS3Server();

  // Starts the fake S3 server, and tells the Assistant service to use its URI
  // for all S3 requests.
  void Setup(FakeS3Mode mode);
  void Teardown();

  // Returns the access token used by the S3 Server. This is only populated
  // after |Setup| is called.
  std::string GetAccessToken() const;

 private:
  void SetAccessTokenForMode(FakeS3Mode mode);
  void SetFakeS3ServerURI();
  void UnsetFakeS3ServerURI();
  void StartS3ServerProcess(FakeS3Mode mode);
  void StopS3ServerProcess();

  int port() const;

  std::string access_token_{"FAKE_ACCESS_TOKEN"};
  std::string fake_s3_server_uri_;

  std::unique_ptr<PortSelector> port_selector_;

  base::Process fake_s3_server_;

  DISALLOW_COPY_AND_ASSIGN(FakeS3Server);
};

}  // namespace assistant
}  // namespace chromeos

#endif  // CHROME_BROWSER_UI_ASH_ASSISTANT_TEST_FAKE_S3_SERVER_H_
