/*  Copyright (C) 2016 CZ.NIC, z.s.p.o. <knot-dns@labs.nic.cz>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "utils/common/cert.h"
#include "libknot/error.h"

#include <tap/basic.h>
#include <string.h>

static const uint8_t CERT_DER[] = {
	0x30, 0x82, 0x06, 0xad, 0x30, 0x82, 0x05, 0x95, 0xa0, 0x03, 0x02, 0x01,
	0x02, 0x02, 0x07, 0x06, 0xdb, 0x97, 0x5f, 0xf1, 0xa8, 0x9b, 0x30, 0x0d,
	0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05,
	0x00, 0x30, 0x81, 0x8c, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04,
	0x06, 0x13, 0x02, 0x49, 0x4c, 0x31, 0x16, 0x30, 0x14, 0x06, 0x03, 0x55,
	0x04, 0x0a, 0x13, 0x0d, 0x53, 0x74, 0x61, 0x72, 0x74, 0x43, 0x6f, 0x6d,
	0x20, 0x4c, 0x74, 0x64, 0x2e, 0x31, 0x2b, 0x30, 0x29, 0x06, 0x03, 0x55,
	0x04, 0x0b, 0x13, 0x22, 0x53, 0x65, 0x63, 0x75, 0x72, 0x65, 0x20, 0x44,
	0x69, 0x67, 0x69, 0x74, 0x61, 0x6c, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69,
	0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x53, 0x69, 0x67, 0x6e, 0x69,
	0x6e, 0x67, 0x31, 0x38, 0x30, 0x36, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13,
	0x2f, 0x53, 0x74, 0x61, 0x72, 0x74, 0x43, 0x6f, 0x6d, 0x20, 0x43, 0x6c,
	0x61, 0x73, 0x73, 0x20, 0x32, 0x20, 0x50, 0x72, 0x69, 0x6d, 0x61, 0x72,
	0x79, 0x20, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6d, 0x65, 0x64, 0x69, 0x61,
	0x74, 0x65, 0x20, 0x53, 0x65, 0x72, 0x76, 0x65, 0x72, 0x20, 0x43, 0x41,
	0x30, 0x1e, 0x17, 0x0d, 0x31, 0x35, 0x30, 0x33, 0x30, 0x36, 0x30, 0x38,
	0x31, 0x30, 0x33, 0x36, 0x5a, 0x17, 0x0d, 0x31, 0x37, 0x30, 0x33, 0x30,
	0x36, 0x30, 0x38, 0x31, 0x33, 0x31, 0x33, 0x5a, 0x30, 0x81, 0xa2, 0x31,
	0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x43, 0x5a,
	0x31, 0x1b, 0x30, 0x19, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x12, 0x48,
	0x6c, 0x61, 0x76, 0x6e, 0x69, 0x20, 0x6d, 0x65, 0x73, 0x74, 0x6f, 0x20,
	0x50, 0x72, 0x61, 0x68, 0x61, 0x31, 0x1a, 0x30, 0x18, 0x06, 0x03, 0x55,
	0x04, 0x07, 0x13, 0x11, 0x50, 0x72, 0x61, 0x68, 0x61, 0x20, 0x2d, 0x20,
	0x56, 0x69, 0x6e, 0x6f, 0x68, 0x72, 0x61, 0x64, 0x79, 0x31, 0x19, 0x30,
	0x17, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x10, 0x43, 0x5a, 0x2e, 0x4e,
	0x49, 0x43, 0x2c, 0x20, 0x7a, 0x2e, 0x73, 0x2e, 0x70, 0x2e, 0x6f, 0x2e,
	0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x0f, 0x77,
	0x77, 0x77, 0x2e, 0x6b, 0x6e, 0x6f, 0x74, 0x2d, 0x64, 0x6e, 0x73, 0x2e,
	0x63, 0x7a, 0x31, 0x25, 0x30, 0x23, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
	0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16, 0x16, 0x68, 0x6f, 0x73, 0x74, 0x6d,
	0x61, 0x73, 0x74, 0x65, 0x72, 0x40, 0x6b, 0x6e, 0x6f, 0x74, 0x2d, 0x64,
	0x6e, 0x73, 0x2e, 0x63, 0x7a, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06,
	0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00,
	0x03, 0x82, 0x01, 0x0f, 0x00, 0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01,
	0x01, 0x00, 0x9d, 0xa9, 0x87, 0x3b, 0x2e, 0xfa, 0xfd, 0xf6, 0x0b, 0x63,
	0xa8, 0x23, 0xc6, 0x66, 0x3d, 0x02, 0x9e, 0xf0, 0xa0, 0x83, 0x44, 0xbd,
	0x1a, 0xea, 0xee, 0xdd, 0xb3, 0x8d, 0xe6, 0xbd, 0xe1, 0xdc, 0xff, 0xf6,
	0xa9, 0x10, 0xdd, 0x0e, 0x3e, 0x6b, 0xb2, 0x8d, 0xa7, 0x52, 0x2b, 0xd4,
	0xff, 0xc6, 0x7a, 0x65, 0x23, 0x34, 0x02, 0x09, 0xc0, 0x17, 0xcc, 0x5d,
	0x47, 0x29, 0x9a, 0xac, 0x40, 0xdc, 0x8a, 0x3a, 0x65, 0xa3, 0xf5, 0x9f,
	0x1b, 0xaa, 0xaf, 0xdf, 0xab, 0xa7, 0xd3, 0x14, 0x86, 0xcc, 0xb8, 0x28,
	0x9a, 0x65, 0x33, 0xda, 0x22, 0xae, 0x62, 0x1a, 0x6b, 0xb7, 0x67, 0xdc,
	0xf0, 0x8c, 0xa3, 0xc1, 0x84, 0x1e, 0xf2, 0xcc, 0xf3, 0xe5, 0xfe, 0xf4,
	0xd8, 0x90, 0x50, 0xbc, 0x9d, 0x77, 0xc9, 0x4d, 0xb9, 0x8c, 0xfe, 0xff,
	0x33, 0x02, 0x7c, 0x4f, 0xb1, 0x3d, 0x66, 0x30, 0x97, 0xa3, 0xe0, 0x54,
	0xc1, 0x3f, 0x4a, 0xd8, 0x3a, 0xa7, 0xcb, 0xe8, 0x16, 0x37, 0x72, 0xb3,
	0x36, 0x90, 0x75, 0x1a, 0x2f, 0x95, 0x55, 0xb5, 0x10, 0x18, 0x29, 0xb0,
	0xee, 0x32, 0x8b, 0x3e, 0x02, 0x38, 0x5f, 0x53, 0xd6, 0x73, 0x41, 0x4c,
	0x1e, 0xae, 0xcf, 0x4f, 0x50, 0xa9, 0x71, 0xbc, 0x79, 0xa5, 0x9e, 0xd6,
	0x13, 0x07, 0xa3, 0xaa, 0x89, 0x0d, 0x31, 0x8c, 0x3a, 0x80, 0xe1, 0x53,
	0x22, 0x29, 0x43, 0xb9, 0xdf, 0xf6, 0xfb, 0x6d, 0xad, 0xbd, 0xf8, 0xc3,
	0xee, 0xbe, 0x59, 0xd0, 0x06, 0x45, 0x35, 0x95, 0xce, 0xcb, 0x61, 0xd9,
	0x74, 0x9e, 0x90, 0x96, 0xb8, 0xd6, 0xb2, 0xc9, 0x29, 0xfd, 0x45, 0xaf,
	0xea, 0xbe, 0x7b, 0x96, 0x76, 0x59, 0xe8, 0x04, 0x05, 0x5c, 0x8e, 0xc1,
	0xb3, 0x7c, 0xeb, 0xd8, 0xc8, 0x3d, 0x84, 0x13, 0x50, 0x07, 0x6f, 0xff,
	0x27, 0x69, 0xcb, 0x33, 0x62, 0x87, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3,
	0x82, 0x02, 0xfa, 0x30, 0x82, 0x02, 0xf6, 0x30, 0x09, 0x06, 0x03, 0x55,
	0x1d, 0x13, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0b, 0x06, 0x03, 0x55, 0x1d,
	0x0f, 0x04, 0x04, 0x03, 0x02, 0x03, 0xa8, 0x30, 0x1d, 0x06, 0x03, 0x55,
	0x1d, 0x25, 0x04, 0x16, 0x30, 0x14, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05,
	0x05, 0x07, 0x03, 0x02, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07,
	0x03, 0x01, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04,
	0x14, 0x86, 0x4e, 0x14, 0x78, 0x0e, 0x5d, 0x44, 0x21, 0xe0, 0x80, 0x23,
	0x6b, 0x1f, 0x9b, 0xf2, 0xd6, 0x09, 0xc5, 0x50, 0xa6, 0x30, 0x1f, 0x06,
	0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x11, 0xdb,
	0x23, 0x45, 0xfd, 0x54, 0xcc, 0x6a, 0x71, 0x6f, 0x84, 0x8a, 0x03, 0xd7,
	0xbe, 0xf7, 0x01, 0x2f, 0x26, 0x86, 0x30, 0x36, 0x06, 0x03, 0x55, 0x1d,
	0x11, 0x04, 0x2f, 0x30, 0x2d, 0x82, 0x0f, 0x77, 0x77, 0x77, 0x2e, 0x6b,
	0x6e, 0x6f, 0x74, 0x2d, 0x64, 0x6e, 0x73, 0x2e, 0x63, 0x7a, 0x82, 0x0b,
	0x6b, 0x6e, 0x6f, 0x74, 0x2d, 0x64, 0x6e, 0x73, 0x2e, 0x63, 0x7a, 0x82,
	0x0d, 0x2a, 0x2e, 0x6b, 0x6e, 0x6f, 0x74, 0x2d, 0x64, 0x6e, 0x73, 0x2e,
	0x63, 0x7a, 0x30, 0x82, 0x01, 0x56, 0x06, 0x03, 0x55, 0x1d, 0x20, 0x04,
	0x82, 0x01, 0x4d, 0x30, 0x82, 0x01, 0x49, 0x30, 0x08, 0x06, 0x06, 0x67,
	0x81, 0x0c, 0x01, 0x02, 0x02, 0x30, 0x82, 0x01, 0x3b, 0x06, 0x0b, 0x2b,
	0x06, 0x01, 0x04, 0x01, 0x81, 0xb5, 0x37, 0x01, 0x02, 0x03, 0x30, 0x82,
	0x01, 0x2a, 0x30, 0x2e, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07,
	0x02, 0x01, 0x16, 0x22, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77,
	0x77, 0x77, 0x2e, 0x73, 0x74, 0x61, 0x72, 0x74, 0x73, 0x73, 0x6c, 0x2e,
	0x63, 0x6f, 0x6d, 0x2f, 0x70, 0x6f, 0x6c, 0x69, 0x63, 0x79, 0x2e, 0x70,
	0x64, 0x66, 0x30, 0x81, 0xf7, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05,
	0x07, 0x02, 0x02, 0x30, 0x81, 0xea, 0x30, 0x27, 0x16, 0x20, 0x53, 0x74,
	0x61, 0x72, 0x74, 0x43, 0x6f, 0x6d, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69,
	0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x41, 0x75, 0x74,
	0x68, 0x6f, 0x72, 0x69, 0x74, 0x79, 0x30, 0x03, 0x02, 0x01, 0x01, 0x1a,
	0x81, 0xbe, 0x54, 0x68, 0x69, 0x73, 0x20, 0x63, 0x65, 0x72, 0x74, 0x69,
	0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x77, 0x61, 0x73, 0x20, 0x69,
	0x73, 0x73, 0x75, 0x65, 0x64, 0x20, 0x61, 0x63, 0x63, 0x6f, 0x72, 0x64,
	0x69, 0x6e, 0x67, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x43,
	0x6c, 0x61, 0x73, 0x73, 0x20, 0x32, 0x20, 0x56, 0x61, 0x6c, 0x69, 0x64,
	0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x72, 0x65, 0x71, 0x75, 0x69, 0x72,
	0x65, 0x6d, 0x65, 0x6e, 0x74, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68,
	0x65, 0x20, 0x53, 0x74, 0x61, 0x72, 0x74, 0x43, 0x6f, 0x6d, 0x20, 0x43,
	0x41, 0x20, 0x70, 0x6f, 0x6c, 0x69, 0x63, 0x79, 0x2c, 0x20, 0x72, 0x65,
	0x6c, 0x69, 0x61, 0x6e, 0x63, 0x65, 0x20, 0x6f, 0x6e, 0x6c, 0x79, 0x20,
	0x66, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x69, 0x6e, 0x74, 0x65,
	0x6e, 0x64, 0x65, 0x64, 0x20, 0x70, 0x75, 0x72, 0x70, 0x6f, 0x73, 0x65,
	0x20, 0x69, 0x6e, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6c, 0x69, 0x61, 0x6e,
	0x63, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x65,
	0x6c, 0x79, 0x69, 0x6e, 0x67, 0x20, 0x70, 0x61, 0x72, 0x74, 0x79, 0x20,
	0x6f, 0x62, 0x6c, 0x69, 0x67, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2e,
	0x30, 0x35, 0x06, 0x03, 0x55, 0x1d, 0x1f, 0x04, 0x2e, 0x30, 0x2c, 0x30,
	0x2a, 0xa0, 0x28, 0xa0, 0x26, 0x86, 0x24, 0x68, 0x74, 0x74, 0x70, 0x3a,
	0x2f, 0x2f, 0x63, 0x72, 0x6c, 0x2e, 0x73, 0x74, 0x61, 0x72, 0x74, 0x73,
	0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x63, 0x72, 0x74, 0x32, 0x2d,
	0x63, 0x72, 0x6c, 0x2e, 0x63, 0x72, 0x6c, 0x30, 0x81, 0x8e, 0x06, 0x08,
	0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x01, 0x01, 0x04, 0x81, 0x81, 0x30,
	0x7f, 0x30, 0x39, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30,
	0x01, 0x86, 0x2d, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x6f, 0x63,
	0x73, 0x70, 0x2e, 0x73, 0x74, 0x61, 0x72, 0x74, 0x73, 0x73, 0x6c, 0x2e,
	0x63, 0x6f, 0x6d, 0x2f, 0x73, 0x75, 0x62, 0x2f, 0x63, 0x6c, 0x61, 0x73,
	0x73, 0x32, 0x2f, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72, 0x2f, 0x63, 0x61,
	0x30, 0x42, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x02,
	0x86, 0x36, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x61, 0x69, 0x61,
	0x2e, 0x73, 0x74, 0x61, 0x72, 0x74, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f,
	0x6d, 0x2f, 0x63, 0x65, 0x72, 0x74, 0x73, 0x2f, 0x73, 0x75, 0x62, 0x2e,
	0x63, 0x6c, 0x61, 0x73, 0x73, 0x32, 0x2e, 0x73, 0x65, 0x72, 0x76, 0x65,
	0x72, 0x2e, 0x63, 0x61, 0x2e, 0x63, 0x72, 0x74, 0x30, 0x23, 0x06, 0x03,
	0x55, 0x1d, 0x12, 0x04, 0x1c, 0x30, 0x1a, 0x86, 0x18, 0x68, 0x74, 0x74,
	0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x73, 0x74, 0x61, 0x72,
	0x74, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x30, 0x0d, 0x06,
	0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00,
	0x03, 0x82, 0x01, 0x01, 0x00, 0xae, 0xea, 0xec, 0xe0, 0x6e, 0xe1, 0x5e,
	0xe3, 0x06, 0xe6, 0x09, 0xff, 0xf2, 0xea, 0xeb, 0xbd, 0xc2, 0xf9, 0xa2,
	0x79, 0xbb, 0xd1, 0xc3, 0x9c, 0x9f, 0xbd, 0x74, 0x0c, 0x9c, 0xeb, 0x73,
	0xf1, 0x5c, 0x57, 0x98, 0x8c, 0xaf, 0xaa, 0xfb, 0xcf, 0xfb, 0x55, 0x31,
	0x54, 0x71, 0x07, 0xdd, 0x7c, 0x83, 0x70, 0xcb, 0x12, 0xbf, 0x05, 0xd8,
	0x62, 0xf1, 0xe0, 0x9d, 0x1c, 0x35, 0xb2, 0x42, 0xb1, 0x37, 0xe8, 0x73,
	0x4c, 0xe5, 0xda, 0xd9, 0xcb, 0xe6, 0x5a, 0x50, 0x31, 0x14, 0xce, 0x50,
	0xc0, 0xfb, 0x68, 0xb9, 0xe6, 0x48, 0x24, 0xdd, 0x4f, 0xbe, 0x34, 0x28,
	0xba, 0x21, 0x53, 0x86, 0x86, 0x91, 0x6f, 0xb0, 0x8e, 0x34, 0x20, 0x4d,
	0xdf, 0xef, 0xf3, 0x6f, 0xb0, 0x78, 0x89, 0x4b, 0x80, 0x36, 0xe9, 0x75,
	0x3a, 0xd6, 0x18, 0xc6, 0x84, 0xe3, 0x0c, 0xa9, 0x24, 0xcb, 0x49, 0xaf,
	0x72, 0x09, 0x3a, 0xb5, 0xdd, 0x59, 0xb9, 0xe0, 0xb6, 0x7e, 0xc2, 0x3c,
	0xd0, 0xea, 0xeb, 0x39, 0x06, 0x3f, 0xc6, 0xe9, 0x1f, 0x37, 0x25, 0x3c,
	0xdf, 0x0d, 0x95, 0xcc, 0x9f, 0xa3, 0x68, 0x15, 0x3b, 0x80, 0x9b, 0x17,
	0x1a, 0x54, 0x65, 0x61, 0xb0, 0xcf, 0xb5, 0x76, 0x7c, 0xc2, 0x7e, 0x54,
	0x4d, 0x03, 0xe6, 0x90, 0xa0, 0x59, 0xa9, 0x1c, 0x6d, 0x4b, 0x34, 0x03,
	0xc3, 0xbb, 0xcd, 0x77, 0x60, 0x0e, 0xb1, 0x4e, 0x22, 0x81, 0xe4, 0x17,
	0xf4, 0xd2, 0x58, 0x2c, 0x72, 0x4e, 0xde, 0xd0, 0x24, 0x25, 0xfb, 0xd8,
	0x3f, 0xc8, 0x0f, 0x3b, 0x0d, 0xec, 0x75, 0x81, 0x37, 0x08, 0xd0, 0x0a,
	0x29, 0x28, 0x9f, 0x7f, 0x35, 0x83, 0x70, 0x18, 0x6c, 0x4b, 0x24, 0x8e,
	0xc0, 0xe5, 0xc1, 0xbb, 0x5b, 0x24, 0xb4, 0x5c, 0x8e, 0xbc, 0x79, 0xc0,
	0xad, 0x47, 0x17, 0xdd, 0x7a, 0xf2, 0x26, 0x6e, 0xe4
};

static gnutls_x509_crt_t get_cert(void)
{
	gnutls_x509_crt_t cert = NULL;
	if (gnutls_x509_crt_init(&cert) != 0) {
		return NULL;
	}

	const gnutls_datum_t der = {
		.data = (uint8_t *)CERT_DER,
		.size = sizeof(CERT_DER)
	};
	if (gnutls_x509_crt_import(cert, &der, GNUTLS_X509_FMT_DER) != 0) {
		gnutls_x509_crt_deinit(cert);
		return NULL;
	}

	return cert;
}

void test_get_pin(void)
{
	gnutls_x509_crt_t cert = get_cert();
	ok(cert != NULL, "create testing certificate");

	static const uint8_t expected[] = {
		0x35, 0xa2, 0x1a, 0x6b, 0x95, 0x34, 0x53, 0xed, 0xf7, 0xf7,
		0x08, 0x76, 0x08, 0x17, 0x9c, 0x4a, 0x16, 0x6e, 0xcf, 0xd5,
		0xff, 0x46, 0x71, 0x1d, 0xa8, 0x08, 0xb7, 0xef, 0x75, 0xaf,
		0xfd, 0xa0
	};

	uint8_t pin[CERT_PIN_LEN] = { 0 };

	int r = cert_get_pin(cert, NULL, 0);
	ok(r == KNOT_EINVAL, "cert_get_pin(), no target buffer");

	r = cert_get_pin(cert, pin, sizeof(pin) - 1);
	ok(r == KNOT_EINVAL, "cert_get_pin(), invalid buffer size");

	r = cert_get_pin(cert, pin, sizeof(pin));
	ok(r == KNOT_EOK && sizeof(pin) == sizeof(expected) &&
	   memcmp(pin, expected, sizeof(pin)) == 0,
	   "cert_get_pin(), valid input");

	gnutls_x509_crt_deinit(cert);
}

int main(int argc, char *argv[])
{
	plan_lazy();

	test_get_pin();

	return 0;
}
