/******************************* */
/* Rule Set Based Access Control */
/* Author and (c) 1999-2001:     */
/*   Amon Ott <ao@rsbac.org>     */
/* API: Data structures          */
/* and functions for Access      */
/* Control Information           */
/* Last modified: 24/Aug/2001    */
/******************************* */

#ifndef __RSBAC_ACI_H
#define __RSBAC_ACI_H

#include <rsbac/types.h>
#include <linux/init.h>

/***************************************************/
/*                   Prototypes                    */
/***************************************************/

/* All functions return 0, if no error occurred, and a negative error code  */
/* otherwise. The error codes are defined in rsbac_error.h.                 */

/****************************************************************************/
/* Initialization, including ACI restoration for all mounted devices from   */
/* disk. After this call, all ACI is kept in memory for performance reasons,*/
/* but user and file/dir object ACI are written to disk on every change.    */

#ifdef CONFIG_RSBAC_INIT_DELAY
extern int rsbac_init(kdev_t root_dev);
#else
extern int rsbac_init(kdev_t root_dev) __init;
#endif

/* To turn RSBAC off on umount of root device */
extern void rsbac_off(void);

/* For other kernel parts to check, whether RSBAC was initialized correctly */
extern inline boolean rsbac_is_initialized(void);

/* When mounting a device, its ACI must be read and added to the ACI lists. */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
extern int rsbac_mount(struct super_block * sb_p);
#else
extern int rsbac_mount(struct super_block * sb_p, struct dentry * d_covers);
#endif
/* When umounting a device, its ACI must be removed from the ACI lists. */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
extern int rsbac_umount(struct super_block * sb_p);
#else
extern int rsbac_umount(struct super_block * sb_p, struct dentry * d_covers);
#endif

/* On pivot_root, we must unblock the dentry tree of the old root */
/* by putting all cached rsbac.dat dentries */
int rsbac_free_dat_dentries(void);

/* Some information about the current status is also available */
extern int rsbac_stats(void);

/* Trigger internal consistency check (int: if != 0: correct errors) */
extern int rsbac_check(int correct, int check_inode);

/* RSBAC attribute saving to disk can be triggered from outside
 * param: call lock_kernel() before disk access?
 */
#if defined(CONFIG_RSBAC_MAINT) || defined(CONFIG_RSBAC_AUTO_WRITE)
extern int rsbac_write(boolean);
#endif

/* get the parent of a target
 * returns -RSBAC_EINVALIDTARGET for non-fs targets
 * and -RSBAC_ENOTFOUND, if no parent available
 * In kernels >= 2.4.0, device_p->d_covers is used and the item is properly
 * locked for reading, so never call with a write lock held on device_p!
 */
int rsbac_get_parent(enum rsbac_target_t target,
                     union rsbac_target_id_t tid,
                     enum rsbac_target_t * parent_target_p,
                     union rsbac_target_id_t * parent_tid_p);

/****************************************************************************/
/* For objects, users and processes all manipulation is encapsulated by the */
/* function calls rsbac_set_attr, rsbac_get_attr and rsbac_remove_target.   */
                          
extern int rsbac_get_attr(enum rsbac_switch_target_t module,
                          enum rsbac_target_t target,
                          union rsbac_target_id_t,
                          enum rsbac_attribute_t,
                          union rsbac_attribute_value_t *,
                          boolean);  /* inherit? */

/* setting attribute security_level to SL_rsbac_internal returns an error */

extern int rsbac_set_attr(enum rsbac_switch_target_t module,
                          enum rsbac_target_t,
                          union rsbac_target_id_t,
                          enum rsbac_attribute_t,
                          union  rsbac_attribute_value_t);

/* All RSBAC targets should be removed, if no longer needed, to prevent     */
/* memory wasting.                                                          */

extern int rsbac_remove_target(enum rsbac_target_t,
                               union rsbac_target_id_t);


#endif
