/*
 * ec.c -- low level functions to access Embedded Conrtroller,
 *         Keyboard Controller and system I/O ports or memory
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * Written by Sos Pter <sp@osb.hu>, 2002-2003
 */


#include <linux/config.h>
#include <linux/types.h>
#include <linux/delay.h>
#include <linux/sched.h>

#if CONFIG_ACPI_EC
#include <linux/acpi.h>
#ifdef ACPI_CA_VERSION
#if ACPI_CA_VERSION > 0x20021121
#define USE_ACPI_EC
#endif
#endif
#endif

#include <asm/io.h>

#include "ec.h"

/*
 * Wait for embedded controller buffer
 */

static int omnibook_ec_wait(u8 event)
{
	int timeout = OMNIBOOK_TIMEOUT;

	switch (event) {
	case OMNIBOOK_EC_STAT_OBF:
		while (!(inb(OMNIBOOK_EC_SC) & event) && timeout--)
			mdelay(1);
		break;
	case OMNIBOOK_EC_STAT_IBF:
		while ((inb(OMNIBOOK_EC_SC) & event) && timeout--)
			mdelay(1);
		break;
	default:
		return -EINVAL;
	}
	if (timeout)
		return 0;
	return -ETIME;
}

/*
 * Read from the embedded controller
 */

int omnibook_ec_read(u8 addr, u8 *data)
{
	unsigned long flags;
	int retval;
	
#ifdef USE_ACPI_EC
	retval = ec_read(addr, data);
	if (!retval)
		return retval;
#endif
	omnibook_save_flags(flags);
	retval = omnibook_ec_wait(OMNIBOOK_EC_STAT_IBF);
	if (retval)
		goto end;
	outb(OMNIBOOK_EC_CMD_READ, OMNIBOOK_EC_SC);
	retval = omnibook_ec_wait(OMNIBOOK_EC_STAT_IBF);
	if (retval)
		goto end;
	outb(addr, OMNIBOOK_EC_DATA);
	retval = omnibook_ec_wait(OMNIBOOK_EC_STAT_OBF);
	if (retval)
		goto end;
	*data = inb(OMNIBOOK_EC_DATA);
end:
	omnibook_restore_flags(flags);
	return retval;
}

/*
 * Write to the embedded controller
 */

int omnibook_ec_write(u8 addr, u8 data)
{

	unsigned long flags;
	int retval;
	
#ifdef USE_ACPI_EC
	retval = ec_write(addr, data);
	if (!retval)
		return retval;
#endif
	omnibook_save_flags(flags);
	retval = omnibook_ec_wait(OMNIBOOK_EC_STAT_IBF);
	if (retval)
		goto end;
	outb(OMNIBOOK_EC_CMD_WRITE, OMNIBOOK_EC_SC);
	retval = omnibook_ec_wait(OMNIBOOK_EC_STAT_IBF);
	if (retval)
		goto end;
	outb(addr, OMNIBOOK_EC_DATA);
	retval = omnibook_ec_wait(OMNIBOOK_EC_STAT_IBF);
	if (retval)
		goto end;
	outb(data, OMNIBOOK_EC_DATA);
end:
	omnibook_restore_flags(flags);
	return retval;
}

int omnibook_ec_read16(u8 addr, u16 *data)
{
	int retval;
	u8 high;
	u8 low;
	u16 result;

	retval = omnibook_ec_read(addr, &low);
	if (retval)
		return retval;
	retval = omnibook_ec_read(addr + 0x01, &high);
	result = ((high << 8) + low);
	*data = result;
	return retval;
}

/*
 * Wait for keyboard buffer
 */

static int omnibook_kbc_wait(u8 event)
{
	int timeout = OMNIBOOK_TIMEOUT;

	switch (event) {
	case OMNIBOOK_KBC_STAT_OBF:
		while (!(inb(OMNIBOOK_KBC_SC) & event) && timeout--)
			mdelay(1);
		break;
	case OMNIBOOK_KBC_STAT_IBF:
		while ((inb(OMNIBOOK_KBC_SC) & event) && timeout--)
			mdelay(1);
		break;
	default:
		return -EINVAL;
	}
	if (timeout)
		return 0;
	return -ETIME;
}

/*
 * Write to the keyboard command register
 */

static int omnibook_kbc_write_command(u8 cmd)
{
	unsigned long flags;
	int retval;
	
	omnibook_save_flags(flags);
	retval = omnibook_kbc_wait(OMNIBOOK_KBC_STAT_IBF);
	if (retval)
		goto end;
	outb(cmd, OMNIBOOK_KBC_SC);
	retval = omnibook_kbc_wait(OMNIBOOK_KBC_STAT_IBF);
end:
	omnibook_restore_flags(flags);
	return retval;
}

/*
 * Write to the keyboard data register
 */

static int omnibook_kbc_write_data(u8 data)
{
	unsigned long flags;
	int retval;
	
	omnibook_save_flags(flags);
	retval = omnibook_kbc_wait(OMNIBOOK_KBC_STAT_IBF);
	if (retval)
		goto end;;
	outb(data, OMNIBOOK_KBC_DATA);
	retval = omnibook_kbc_wait(OMNIBOOK_KBC_STAT_IBF);
end:
	omnibook_restore_flags(flags);
	return retval;
}

/*
 * Send a command to keyboard controller
 */

int omnibook_kbc_command(u8 cmd, u8 data)
{
	int retval;
	
	retval = omnibook_kbc_write_command(cmd);
	if (retval)
		return retval;
	retval = omnibook_kbc_write_data(data);
	return retval;
}

/*
 * Read a value from a system I/O address
 */

int omnibook_io_read(u32 addr, u8 *data)
{
	unsigned long flags;

	omnibook_save_flags(flags);
	*data = inb(addr);
	omnibook_restore_flags(flags);
	return 0;
}

/*
 * Write a value to a system I/O address
 */

int omnibook_io_write(u32 addr, u8 data)
{
	unsigned long flags;

	omnibook_save_flags(flags);
	outb(data, addr);
	omnibook_restore_flags(flags);
	return 0;
}

/*
 * Read a value from a system memory address
 */

int omnibook_mem_read(u32 addr, u8 *data)
{
	char *base = ioremap(addr, 1);

	*data = readb(base);
	iounmap(base);
	return 0;
}

/*
 * Write a value to a system memory address
 */

int omnibook_mem_write(u32 addr, u8 data)
{
	char *base = ioremap(addr, 1);

	writeb(data, base);
	iounmap(base);
	return 0;
}

/* End of file */
