/*
 * hangcheck-comp.c
 *
 * Test driver for an I/O fencer that combines a timer and a thread.
 *
 * Copyright (C) 2002 Oracle Corporation.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have recieved a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 021110-1307, USA.
 */


#include <linux/module.h>
#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/reboot.h>
#include <linux/smp_lock.h>
#include <linux/init.h>
#include <asm/uaccess.h>

#include "hangcheck-comp.h"

/* These are in seconds */
#define IOFENCE_TIMER_MARGIN	30	/* Default timer fudge factor */
#define IOFENCE_TIMER_TICK	30	/* Default timer delay */
#define IOFENCE_SCHED_MARGIN	60	/* Default scheduler fudge factor */

#define SHUTDOWN_SIGS (sigmask(SIGKILL) | sigmask(SIGHUP) | \
                       sigmask(SIGINT) | sigmask(SIGQUIT))


static int hangcheck_timer_tick = IOFENCE_TIMER_TICK;
static int hangcheck_timer_margin = IOFENCE_TIMER_MARGIN;
static int hangcheck_sched_margin = IOFENCE_SCHED_MARGIN;

MODULE_PARM(hangcheck_timer_tick,"i");
MODULE_PARM_DESC(hangcheck_timer_tick, "Timer delay.");
MODULE_PARM(hangcheck_timer_margin,"i");
MODULE_PARM_DESC(hangcheck_timer_margin, "If the hangcheck timer has been delayed more than hangcheck_timer_margin seconds, the machine will reboot.");
MODULE_PARM(hangcheck_sched_margin,"i");
MODULE_PARM_DESC(hangcheck_sched_margin, "If the hangcheck thread has been delayed more than hangcheck_sched_margin seconds, the machine will reboot.");
MODULE_LICENSE("GPL");


static void hangcheck_fire(unsigned long);

/* Last time fired */
static unsigned long long hangcheck_tsc;

static struct timer_list hangcheck_timer_ticktock = {
	function:	hangcheck_fire,
};

/* Task of hangcheck thread */
static struct task_struct *hangcheck_task = NULL;
/* Wait queue for the hangcheck thread */
static wait_queue_head_t hangcheck_task_wait;
/* Last time scheduled */
static unsigned long hangcheck_last_jiffies;
/* Time to die */
static int hangcheck_die = 0;
/*
 * Lock around hangcheck_last_jiffies and hangcheck_die.  Yes, lock_kernel()
 * is probably as good, but there is no reason to hold up other CPUs.
 * This process is going to be scheduled enough with a non-trigger case
 * to make it possibly worthwhile.
 */
static spinlock_t hangcheck_lock = SPIN_LOCK_UNLOCKED;


int hangcheck_thread(void *thrway)
{
	int done = 0;
	unsigned long last_jiffies, cur_jiffies;
        sigset_t tmpsig;

        lock_kernel();
	sprintf(current->comm, "hangcheck");
        exit_files(current);
        current->files = init_task.files;
        atomic_inc(&current->files->count);
	daemonize();
        unlock_kernel();

	/* Block all signals except KILL, STOP, HUP, and INT */
	spin_lock_irq(&current->sigmask_lock);
	tmpsig = current->blocked;
	siginitsetinv(&current->blocked, SHUTDOWN_SIGS);
	recalc_sigpending(current);
	spin_unlock_irq(&current->sigmask_lock);

	/* Note that we are running */
	spin_lock(&hangcheck_lock);
	hangcheck_task = current;
	spin_unlock(&hangcheck_lock);
	wake_up(&hangcheck_task_wait);

	printk(KERN_CRIT "hangcheck thread started successfully.\n");

	while (1) {
		spin_lock(&hangcheck_lock);
		done = hangcheck_die;
		last_jiffies = hangcheck_last_jiffies;
		cur_jiffies = jiffies;
		hangcheck_last_jiffies = cur_jiffies;
		spin_unlock(&hangcheck_lock);

		if (done)
			break;

                /* Hmm, does this handle wraparound? */
		if ((last_jiffies+(hangcheck_sched_margin*HZ)) < cur_jiffies) {
			printk(KERN_CRIT "hangcheck is restarting the machine.\n");
			machine_restart(NULL);
		}

		schedule();
	}

	printk(KERN_CRIT "hangcheck thread exited.\n");

	spin_lock(&hangcheck_lock);
	hangcheck_task = NULL;
	spin_unlock(&hangcheck_lock);
	wake_up(&hangcheck_task_wait);

	return 0;
}  /* hangcheck_thread() */


static void hangcheck_fire(unsigned long data)
{
	unsigned long long cur_tsc, tsc_diff, tsc_margin;

	rdtscll(cur_tsc);

	if (cur_tsc > hangcheck_tsc)
		tsc_diff = cur_tsc - hangcheck_tsc;
	else
		tsc_diff = (cur_tsc + (~0ULL - hangcheck_tsc)); /* or something */

#if 0
	printk(KERN_CRIT "tsc_diff = %lu.%lu, predicted diff is %lu.%lu.\n",
	       (unsigned long) ((tsc_diff >> 32) & 0xFFFFFFFFULL),
	       (unsigned long) (tsc_diff & 0xFFFFFFFFULL),
	       (unsigned long) ((tsc_margin >> 32) & 0xFFFFFFFFULL),
	       (unsigned long) (tsc_margin & 0xFFFFFFFFULL));
	printk(KERN_CRIT "hangcheck_timer_margin = %lu, HZ = %lu, current_cpu_data.loops_per_jiffy = %lu.\n", hangcheck_timer_margin, HZ, current_cpu_data.loops_per_jiffy);
#endif
	tsc_margin = (unsigned long long)(hangcheck_timer_margin + hangcheck_timer_tick) * (unsigned long long)HZ * (unsigned long long)current_cpu_data.loops_per_jiffy;

	
	if (tsc_diff > tsc_margin) {
		printk(KERN_CRIT "hangcheck is restarting the machine.\n");
		machine_restart(NULL);
	}
	mod_timer(&hangcheck_timer_ticktock, jiffies + (hangcheck_timer_tick*HZ));
	rdtscll(hangcheck_tsc);
}  /* hangcheck_fire() */


static int __init hangcheck_init(void)
{
	int rc;

        version_hash_print();
	printk("Starting hangcheck timer (tick is %d seconds, margin is %d seconds).\n",
	       hangcheck_timer_tick, hangcheck_timer_margin);

	rdtscll(hangcheck_tsc);
	mod_timer(&hangcheck_timer_ticktock, jiffies + (hangcheck_timer_tick*HZ));

	hangcheck_last_jiffies = jiffies;
	init_waitqueue_head(&hangcheck_task_wait);

	printk("Starting hangcheck thread (margin is %d seconds).\n",
	       hangcheck_sched_margin);
	rc = kernel_thread(hangcheck_thread, NULL, 0);
	if (rc < 0) {
		printk("Unable to start hangcheck thread.\n");
		return rc;
	} 

	spin_lock(&hangcheck_lock);
	while (!hangcheck_task) {
		spin_unlock(&hangcheck_lock);
		sleep_on(&hangcheck_task_wait);
		spin_lock(&hangcheck_lock);
	}
	spin_unlock(&hangcheck_lock);

	return 0;
}  /* hangcheck_init() */


static void __exit hangcheck_exit(void)
{
	printk("Stopping hangcheck timer.\n");

	lock_kernel();
	del_timer(&hangcheck_timer_ticktock);
	unlock_kernel();

	spin_lock(&hangcheck_lock);
	hangcheck_die = 1;
	spin_unlock(&hangcheck_lock);

	spin_lock(&hangcheck_lock);
	while (hangcheck_task) {
		spin_unlock(&hangcheck_lock);
		sleep_on(&hangcheck_task_wait);
		spin_lock(&hangcheck_lock);
	}
	spin_unlock(&hangcheck_lock);
}  /* hangcheck_exit() */

module_init(hangcheck_init);
module_exit(hangcheck_exit);

