/*
 * Copyright 2008 Sony Corporation
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the names of the copyright holders nor the names of their
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <openssl/evp.h>
#include <openssl/engine.h>

#include <stdio.h>
#include <string.h>
#include <pthread.h>
#include <sys/stat.h>

#include "e_cell.h"
#include "bench_utils.c"

#define DEFAULT_SIZE (1024 * 1024 * 256)
#define BUFFER_SIZE (1024 * 8)

typedef struct params_st
	{
	int index;
	pthread_t thread;
	ENGINE *engine;
	const EVP_CIPHER *cipher;
	const char *in_filename;
	const char *out_filename;
	unsigned long long max_size;
	const unsigned char *key;
	const unsigned char *iv;
	} params_t;

static void *thread_proc(void *ptr)
	{
	EVP_CIPHER_CTX ctx;
	int ret;
	params_t *params = (params_t *)ptr;
	void *in_buffer;
	void *out_buffer;
	FILE *in_fp;
	FILE *out_fp;
	char *out_filename;
	int out_size;
	long long total_size = 0;

	/* allocate I/O buffers */
	in_buffer = aligned_malloc(SHARED_DATA_ALIGN, BUFFER_SIZE);
	if (!in_buffer)
		{
		perror("align_malloc");
		exit(1);
		}

	out_buffer = aligned_malloc(SHARED_DATA_ALIGN, BUFFER_SIZE);
	if (!out_buffer)
		{
		perror("align_malloc");
		exit(1);
		}

	/* open I/O streams */
	out_filename = OPENSSL_malloc(strlen(params->out_filename) + 20);
	if (!out_filename)
		{
		perror("OPENSSL_malloc");
		exit(1);
		}
	sprintf(out_filename, params->out_filename, params->index);

	in_fp = fopen(params->in_filename, "r");
	if (!in_fp)
		{
		perror(params->in_filename);
		exit(1);
		}

	out_fp = fopen(out_filename, "w");
	if (!out_fp)
		{
		perror(out_filename);
		exit(1);
		}

	/* initialize cipher context */
	EVP_CIPHER_CTX_init(&ctx);

	ret = EVP_EncryptInit_ex(&ctx, params->cipher, params->engine,
		params->key, params->iv);
	if (!ret)
		{
		exit(1);
		}

	/* read input, do encryption/decryption and write output */
	for ( ; ; )
		{
		size_t size;
		if (params->max_size < 0)
			{
			size = BUFFER_SIZE;
			}
		else
			{
			size = BUFFER_SIZE < (params->max_size - total_size) ?
				BUFFER_SIZE : (params->max_size - total_size);
			if (!size)
				break;
			}
		size = fread(in_buffer, 1, size, in_fp);
		if (!size)
			{
			break;
			}
		ret = EVP_EncryptUpdate(&ctx, out_buffer, &out_size, in_buffer, size);
		if (!ret)
			{
			exit(1);
			}

		ret = fwrite(out_buffer, out_size, 1, out_fp);
		if (ret != 1)
			{
			perror(out_filename);
			exit(1);
			}
		total_size += size;
		}

	/* finalize cipher context */
	ret = EVP_EncryptFinal_ex(&ctx, out_buffer, &out_size);
	if (!ret)
		{
		exit(1);
		}
	if (out_size)
		{
		ret = fwrite(out_buffer, out_size, 1, out_fp);
		if (ret != 1)
			{
			perror(out_filename);
			exit(1);
			}
		}

	EVP_CIPHER_CTX_cleanup(&ctx);

	/* close I/O streams */
	fclose(out_fp);
	fclose(in_fp);

	OPENSSL_free(out_filename);

	/* free I/O buffers */
	aligned_free(out_buffer);
	aligned_free(in_buffer);

	return NULL;
	}

int main(int argc, char **argv)
	{
	ENGINE *engine = NULL;
	const EVP_CIPHER *cipher = NULL;
	int num_threads = 1;
	long long max_size = -1;
	char *in_filename = "/dev/zero";
	char *out_filename = "/dev/null";
	unsigned char key[EVP_MAX_KEY_LENGTH];
	unsigned char iv[EVP_MAX_IV_LENGTH];
	params_t *params;
	int ret;
	int i;

	/* initialize OpenSSL */
	locking_setup();
	OpenSSL_add_all_algorithms();
	ENGINE_load_builtin_engines();

	/* initialize variables */
	RAND_bytes(key, sizeof(key));
	RAND_bytes(iv, sizeof(iv));

	/* parse arguments */
	for (i = 1; i < argc; i++)
		{
		if (*argv[i] != '-')
			break;
		if (strcmp("-engine", argv[i]) == 0)
			{
			if (i + 1 >= argc)
				{
				few_args:
				fprintf(stderr, "%s: Too few arguments.\n", argv[i]);
				return 1;
				}
			i++;
			engine = ENGINE_by_id(argv[i]);
			if (!engine)
				{
				fprintf(stderr, "%s: Unknown engine.\n", argv[i]);
				return 1;
				}
			}
		else if (strcmp("-cipher", argv[i]) == 0)
			{
			if (i + 1 >= argc)
				goto few_args;
			i++;
			cipher = EVP_get_cipherbyname(argv[i]);
			if (!cipher)
				{
					fprintf(stderr, "%s: Unknown cipher.\n", argv[i]);
					return 1;
				}
			}
		else if (strcmp("-threads", argv[i]) == 0)
			{
			if (i + 1 >= argc)
				goto few_args;
			i++;
			num_threads = strtol(argv[i], NULL, 0);
			if (num_threads <= 0)
				{
					fprintf(stderr, "%s: Invalid # of threads.\n", argv[i]);
					return 1;
				}
			}
		else if (strcmp("-in", argv[i]) == 0)
			{
			if (i + 1 >= argc)
				goto few_args;
			i++;
			in_filename = argv[i];
			}
		else if (strcmp("-out", argv[i]) == 0)
			{
			if (i + 1 >= argc)
				goto few_args;
			i++;
			out_filename = argv[i];
			}
		else if (strcmp("-size", argv[i]) == 0)
			{
			if (i + 1 >= argc)
				goto few_args;
			i++;
			if (!parse_size(&max_size, argv[i]))
				{
				fprintf(stderr, "%s: Invalid size.\n", argv[i]);
				return 1;
				}
			}
		else if (strcmp("-key", argv[i]) == 0)
			{
			if (i + 1 >= argc)
				goto few_args;
			i++;
			if (!parse_hex(key, sizeof(key), argv[i]))
				{
					fprintf(stderr, "%s: Invalid key.\n", argv[i]);
					return 1;
				}
			}
		else if (strcmp("-iv", argv[i]) == 0)
			{
			if (i + 1 >= argc)
				goto few_args;
			i++;
			if (!parse_hex(iv, sizeof(iv), argv[i]))
				{
					fprintf(stderr, "%s: Invalid IV.\n", argv[i]);
					return 1;
				}
			}
		else
			{
			fprintf(stderr, "%s: Invalid option.\n", argv[i]);
			return 1;
			}
		}

	if (i < argc)
		{
		fprintf(stderr,
			"Usage: %s [-engine <engine>] [-cipher <algorithm>]\n"
			"            [-key <key>] [-iv <iv>] [-threads <num_threads>] [-size <size>]\n"
			"            [-in <input>] [-out <output>]\n",
			argv[0]);
		return 1;
		}

	if (!cipher)
		{
		fprintf(stderr, "No algorithm is specified.\n");
		return 1;
		}

	if (max_size < 0)
		{
		struct stat s;
		if (stat(in_filename, &s) == -1)
			{
			perror(in_filename);
			return 1;
			}
		if (S_ISCHR(s.st_mode) || S_ISFIFO(s.st_mode))
			{
			max_size = DEFAULT_SIZE;
			}
		}

	/* initialize engine */
	if (engine)
		{
		if (!ENGINE_init(engine))
			{
			fprintf(stderr, "ENGINE_init: could not initialize engine.\n");
			return 1;
			}
		}

	/* start threads */
	params = OPENSSL_malloc(sizeof(params[0]) * num_threads);
	if (!params)
		{
		perror("OPENSSL_malloc");
		return 1;
		}
	memset(params, 0, sizeof(params[0]) * num_threads);

	for (i = 0; i < num_threads; i++)
		{
		params[i].index = i;
		params[i].cipher = cipher;
		params[i].engine = engine;
		params[i].max_size = max_size;
		params[i].in_filename = in_filename;
		params[i].out_filename = out_filename;
		params[i].key = key;
		params[i].iv = iv;
		ret = pthread_create(&params[i].thread, NULL, thread_proc, &params[i]);
		if (ret)
			{
			fprintf(stderr, "pthread_create: %s\n", strerror(ret));
			return 1;
			}
		}

	/* wait for threads */
	for (i = 0; i < num_threads; i++)
		{
		pthread_join(params[i].thread, NULL);
		}

	OPENSSL_free(params);

	/* finalize engine */
	if (engine)
		{
		if (!ENGINE_finish(engine))
			{
			fprintf(stderr, "ENGINE_init: could not finalize engine.\n");
			return 1;
			}
		if (!ENGINE_free(engine))
			{
			fprintf(stderr, "ENGINE_init: could not free engine.\n");
			return 1;
			}
		}

	/* cleanup OpenSSL */
	ENGINE_cleanup();
	locking_cleanup();

	return 0;
	}
