/*
 * tools/pvscan.c
 *
 * Copyright (C) 1997 - 2000  Heinz Mauelshagen, Germany
 *
 * May, November 1997
 * May,June 1998
 * February,April,May,October 1999
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with LVM; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    08/10/1997 - changed some messages
 *    30/11/1997 -           "
 *    05/05/1998 - added multiple device support
 *    16/05/1998 - added lvmtab checking
 *    17/05/1998 - obsoleted physical volume name change (it's now in pv_read())
 *    18/05/1998 - fixed display bug for unused physical volumes
 *    22/05/1998 - changed display of physical volumes in case
 *                 of multiple devives
 *    06/08/1998 - LVMTAB_CHECK after command line options
 *    27/06/1998 - changed lvm_tab_* calling convention
 *    06/02/1999 - added options n and s
 *    21/02/1999 - removed LVM_LOCK and LVM_UNLOCK
 *    22/04/1999 - avoided LVM_CHECK_IOP
 *    09/05/1999 - added capacity totals
 *    06/10/1999 - implemented support for long options
 *
 */

#include <lvm_user.h>

char *cmd = NULL;

#ifdef DEBUG
int opt_d = 0;
#endif

int main ( int argc, char **argv)
{
   int c = 0;
   int len = 0;
   int opt_e = 0;
   int opt_n = 0;
   int opt_s = 0;
   int opt_v = 0;
   int p = 0;
   int new_pv = 0;
   int pv_max_name_len = 0;
   int ret = 0;
   int vg_name_len = 0;
   int vg_max_name_len = 0;
   ulong total_cap = 0;
   ulong new_cap = 0;
   char *dummy1 = NULL;
   char *dummy2 = NULL;
   char *dummy3 = NULL;
#ifdef DEBUG
   char *options = "deh?nsv";
#else
   char *options = "eh?nsv";
#endif
   struct option long_options[] = {
#ifdef DEBUG
      { "debug",         no_argument, NULL, 'd'},
#endif
      { "exported",      no_argument, NULL, 'e'},
      { "help",          no_argument, NULL, 'h'},
      { "novolumegroup", no_argument, NULL, 'n'},
      { "short",         no_argument, NULL, 's'},
      { "verbose",       no_argument, NULL, 'v'},
      { NULL, 0, NULL, 0}
   };
   char pv_tmp_name[NAME_LEN] = { 0, };
   char vg_tmp_name[NAME_LEN] = { 0, };
   char vg_name_this[NAME_LEN] = { 0, };
   pv_t **pv = NULL;
   pv_t *pv_tmp;

   cmd = basename ( argv[0]);

   SUSER_CHECK;

   while ( ( c = getopt_long ( argc, argv, options,
                               long_options, NULL)) != EOF) {
      switch ( c) {
#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            opt_d++;
            break;
#endif

         case 'e':
            if ( opt_n > 0) {
               fprintf ( stderr, "%s -- n option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            if ( opt_e > 0) {
               fprintf ( stderr, "%s -- e option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            opt_e++;
            break;


         case 'h':
         case '?':
            printf ( "\n%s  (IOP %d)\n\n%s -- Physical Volume Scan\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
#ifdef DEBUG
                     "\t[-d/--debug]\n"
#endif
                     "\t[-e/--exported]\n"
                     "\t[-h/-?/--help]\n"
                     "\t[-n/--novolumegroup]\n"
                     "\t[-s/--short]\n"
                     "\t[-v[v]/--verbose [--verbose]]\n\n",
                     lvm_version, LVM_LIB_IOP_VERSION,  cmd, cmd);
            return 0;
            break;

         case 'n':
            if ( opt_e > 0) {
               fprintf ( stderr, "%s -- e option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            if ( opt_n > 0) {
               fprintf ( stderr, "%s -- n option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            opt_n++;
            break;

         case 'v':
            if ( opt_v > 1) {
               fprintf ( stderr, "%s -- v option already given two times\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            opt_v++;
            break;

         case 's':
            if ( opt_s > 0) {
               fprintf ( stderr, "%s -- s option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            opt_s++;
            break;

         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n",
                      cmd, c);
            return LVM_EINVALID_CMD_LINE;
      }
   }

   CMD_MINUS_CHK;
   LVM_CHECK_IOP;

   printf ( "%s -- reading all physical volumes (this may take a while...)\n",
            cmd);
   if ( ( ret = pv_read_all_pv ( &pv, TRUE)) < 0) {
      fprintf ( stderr, "%s -- ERROR %d: reading physical volumes\n\n",
                cmd, ret);
      return LVM_EPVSCAN_PV_READ_ALL_PV;
   }


   if ( opt_e + opt_n > 0) printf ( "%s -- WARNING: only displaying physical "
                                    "volumes ", cmd);

   if ( opt_e > 0) printf ( "of exported volume group(s)\n");
   if ( opt_n > 0) printf ( "in no volume group\n");

   if ( opt_v > 0) printf ( "%s -- walking through all physical "
                            "volumes found\n", cmd);
   /* check maximum vg and pv name length */
   if ( pv != NULL) for ( p = 0; pv[p] != NULL; p++) {
      len = strlen ( pv[p]->pv_name);
      if ( pv_max_name_len < len) pv_max_name_len = len;
      len = strlen ( pv[p]->vg_name);
      if ( vg_max_name_len < len) vg_max_name_len = len;
   }
   pv_max_name_len += 2;
   vg_max_name_len += 2;

   if ( pv != NULL) for ( p = 0; pv[p] != NULL; p++) {
      if ( ( ret = pv_read ( pv[p]->pv_name, &pv_tmp, NULL)) < 0 &&
           ret == -LVM_EPV_READ_MD_DEVICE) {
         if ( MAJOR ( pv_create_kdev_t ( pv[p]->pv_name)) != MD_MAJOR) {
            printf ( "%s -- WARNING: physical volume \"%s\" belongs "
                     "to a multiple device\n",
                     cmd, pv[p]->pv_name);
         }
         if ( MAJOR ( pv[p]->pv_dev) != MD_MAJOR) continue;
      }

      total_cap += pv[p]->pv_size;

      /* short listing? */
      if ( opt_s > 0) {
         int out = 0;
         if ( opt_e > 0 && ret == -LVM_EPV_READ_PV_EXPORTED) out = 1;
         if ( opt_n > 0 && pv_check_new ( pv_tmp) == TRUE)   out = 1;
         if ( out == 1) printf ( "%s\n", pv[p]->pv_name);
         continue;
      } else {
         /* only exported PVs? */
         if ( opt_e > 0 && ret != -LVM_EPV_READ_PV_EXPORTED) continue;
         /* only free/new PVs? */
         if ( opt_n > 0 && pv_check_new ( pv_tmp) == FALSE) continue;
      }

      if ( pv_check_new ( pv[p]) == TRUE) {
         new_pv++;
         new_cap += pv[p]->pv_size;
      }

      if ( opt_v > 1) {
         printf ( "%s -- getting data for physical volume \"%s\" from kernel\n",
                  cmd, pv[p]->pv_name);
         if ( pv_status ( pv[p]->vg_name, pv[p]->pv_name, &pv_tmp) < 0)
            printf ( "%s -- physical volume \"%s\" is not active\n",
                     cmd, pv[p]->pv_name);
         pv_show ( pv[p]);
         printf ( "System Id             %s\n\n", pv[p]->system_id);
         printf ( "\n");
      } else {
         printf ( "%s -- ", cmd);
         if ( pv_check_active ( pv[p]->vg_name, pv[p]->pv_name) != TRUE)
            printf ( "inactive ");
         else
            printf ( "ACTIVE   ");
         vg_name_len = strlen ( pv[p]->vg_name) - sizeof ( EXPORTED) + 1;
         if ( pv_check_new ( pv[p]) == TRUE) {
            printf ( "PV \"%s\"  is in no VG  [%s]\n",
                     pv[p]->pv_name,
                     ( dummy1 = lvm_show_size ( pv[p]->pv_size / 2, SHORT)));
            free ( dummy1); dummy1 = NULL;
         } else if ( strcmp ( &pv[p]->vg_name[vg_name_len],
                              EXPORTED) == 0) {
            strncpy ( vg_name_this, pv[p]->vg_name, vg_name_len);
            printf ( "PV \"%s\"  is in EXPORTED VG \"%s\" [%s / %s free]\n",
                     pv[p]->pv_name, vg_name_this,
                     ( dummy1 = lvm_show_size ( pv[p]->pe_total *
                                            pv[p]->pe_size / 2, SHORT)),
                     ( dummy2 = lvm_show_size ( ( pv[p]->pe_total - pv[p]->pe_allocated) * pv[p]->pe_size / 2, SHORT)));
            free ( dummy1); dummy1 = NULL;
            free ( dummy2); dummy2 = NULL;
         } else if ( lvm_tab_vg_check_exist ( pv[p]->vg_name, NULL) != TRUE) {
            printf ( "PV \"%s\"  is associated to an unknown VG "
                     "(run vgscan)\n", pv[p]->pv_name);
         } else {
            sprintf ( pv_tmp_name, "\"%s\"", pv[p]->pv_name);
            sprintf ( vg_tmp_name, "\"%s\"", pv[p]->vg_name);
            printf ( "PV %-*s of VG %-*s [%s / %s free]\n",
                     pv_max_name_len, pv_tmp_name, vg_max_name_len,
                     vg_tmp_name,
                     ( dummy1 = lvm_show_size ( pv[p]->pe_total *
                                            pv[p]->pe_size / 2, SHORT)),
                     ( dummy2 = lvm_show_size ( ( pv[p]->pe_total -
                                                  pv[p]->pe_allocated) *
                                                pv[p]->pe_size / 2, SHORT)));
            free ( dummy1); dummy1 = NULL;
            free ( dummy2); dummy2 = NULL;
         }
      }
   }

   if ( p == 0) printf ( "%s -- no valid physical volumes found\n\n", cmd);
   else {
      printf ( "%s -- total: %d [%s] / in use: %d [%s] / in no "
               "VG: %d [%s]\n\n",
               cmd, p,
               ( dummy1 = lvm_show_size ( total_cap / 2, SHORT)),
               p - new_pv,
               ( dummy2 = lvm_show_size ( ( total_cap - new_cap) / 2, SHORT)),
               new_pv,
               ( dummy3 = lvm_show_size ( new_cap / 2, SHORT)));
      free ( dummy1); dummy1 = NULL;
      free ( dummy2); dummy2 = NULL;
      free ( dummy3); dummy3 = NULL;
   }


   if ( p == 0) return LVM_EPVSCAN_NO_PV_FOUND;
   else         return 0;
}
