/* ====================================================================
 * Copyright (c) 2003-2007, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "config.h"
#include "ImportDialog.h"
#include "ExternProvider.h"
#include "Settings.h"
#include "commands/ImportParam.h"
#include "sublib/Gui.h"
#include "sublib/ExternButton.h"
#include "sublib/settings/LayoutSettings.h"
#include "util/String.h"

// qt
#include <QtGui/QLayout>
#include <QtGui/QPushButton>
#include <QtGui/QLabel>
#include <QtGui/QComboBox>
#include <QtGui/QToolTip>
#include <QtGui/QCheckBox>
#include <Qt3Support/Q3VButtonGroup>
#include <Qt3Support/Q3GroupBox>



ImportDialog::ImportDialog( ExternProvider* p, bool recurse, QWidget *parent )
: super( parent, 0, true, Qt::WStyle_Customize | Qt::WStyle_Dialog |
  Qt::WStyle_NormalBorder | Qt::WStyle_Title | Qt::WStyle_SysMenu ), _p(p)
{
  setName( "ImportDialog" );
  setCaption( _q("subcommander:import") );

  QVBoxLayout *vbl = new QVBoxLayout(this,5,8);
  vbl->setSpacing(10);
  {
    Q3GroupBox* gb = new Q3GroupBox(1,Qt::Vertical,this);
    gb->setTitle( _q("import options: ") );
    //gb->setInsideSpacing(10);
    gb->setInsideMargin(0);
    gb->setFlat(true);
    vbl->addWidget(gb);

    QGridLayout* gl = new QGridLayout(vbl,3,3);
    gl->setMargin(0);
    gl->setRowStretch( 1, 1 );
    {
      {
        QLabel* l = new QLabel(this);
        _src      = new QComboBox(this);
        _srcEx    = new ExternButton(this);

        l->setBuddy(_srcEx);
        l->setText( _q("&source file/folder:") );
        l->setFixedWidth( l->sizeHint().width() );
        _src->setEditable(true);
        _src->setAutoCompletion(true);

        gl->addWidget(l,0,0);
        gl->addWidget(_src,0,1); 
        gl->addWidget(_srcEx,0,2); 

        connect( _src, SIGNAL(activated(const QString&)), SLOT(checkOk(const QString&)) );
        connect( _src, SIGNAL(highlighted(const QString&)), SLOT(checkOk(const QString&)) );
        connect( _src, SIGNAL(textChanged(const QString&)), SLOT(checkOk(const QString&)) );
        connect( _srcEx, SIGNAL(clicked()), SLOT(srcExtern()) );

        QToolTip::add( _src, _q("the source file/folder you like to import") );
      }
      {
        QLabel* l = new QLabel(this);
        _dst      = new QComboBox(this);
        _dstEx    = new ExternButton(this);

        l->setBuddy(_dstEx);
        l->setText( _q("&destination url:") );
        _dst->setEditable(true);
        _dst->setAutoCompletion(true);

        gl->addWidget(l,2,0);
        gl->addWidget(_dst,2,1); 
        gl->addWidget(_dstEx,2,2); 

        connect( _dst, SIGNAL(activated(const QString&)), SLOT(checkOk(const QString&)) );
        connect( _dst, SIGNAL(highlighted(const QString&)), SLOT(checkOk(const QString&)) );
        connect( _dst, SIGNAL(textChanged(const QString&)), SLOT(checkOk(const QString&)) );
        connect( _dstEx, SIGNAL(clicked()), SLOT(dstExtern()) );

        QToolTip::add( _dst, _q("the destination repository folder for the import") );
      }
      {
        _recursive = new QCheckBox(this);
        _recursive->setText( _q("&recursive") );
        _recursive->setChecked(recurse);

        gl->addMultiCellWidget(_recursive,3,3,0,2);
      }
    }

    QHBoxLayout* hu = new QHBoxLayout;
    vbl->addLayout(hu);
    {
      // eats extra space, so the buttons keep their size
      hu->addStretch(1); 

      _ok = new QPushButton(this);
      _ok->setEnabled(false);
      _ok->setText( _q("&Ok") );
      _ok->setDefault(true);
      hu->addWidget(_ok);
      QPushButton* ca = new QPushButton(this);
      ca->setText( _q("&Cancel") );
      hu->addWidget(ca);
      
      hu->addSpacing(getSizeGripSpacing());

      connect( _ok, SIGNAL(clicked()), SLOT(accept()) );
      connect( ca, SIGNAL(clicked()), SLOT(reject()) );
    }
  }

  Settings s;
  resize( s.layout().getSize( name(), QSize() ) );

  // don't resize vertically.
  setMaximumHeight( sizeHint().height() );
}

ImportDialog::~ImportDialog()
{
  Settings s;
  s.layout().setSize( name(), geometry().size() );
}

void ImportDialog::setDstUrl( const QString& url )
{
  _dst->insertItem( url, 0 );
}

void ImportDialog::checkOk( const QString& text )
{
  QString src = _src->currentText();
  QString dst = _dst->currentText();

  if( ! src.isEmpty() && ! dst.isEmpty() )
  {
    _ok->setEnabled(true);
  }
  else
  {
    _ok->setEnabled(false);
  }
}

void ImportDialog::srcExtern()
{
  sc::String res;

  if( _p->selectPath( this, sc::String(_src->currentText().utf8()), res, ExternProvider::Dir ) )
  {
    _src->insertItem( QString::fromUtf8(res), 0 );
  }
}

void ImportDialog::dstExtern()
{
  sc::String res;

  if( _p->selectUrl( this, sc::String(_dst->currentText().utf8()), res, ExternProvider::Dir ) )
  {
    _dst->setCurrentText(QString::fromUtf8(res));
  }
}

ImportParam* ImportDialog::getParameters() const
{
  return new ImportParam( sc::String(_src->currentText().utf8()),
    sc::String(_dst->currentText().utf8()), _recursive->isChecked() );
}
