package Parse::EventLog;

require Exporter;
@ISA     = qw(Exporter);
@EXPORT  = qw(EVENT_INFORMATION EVENT_WARNING EVENT_ERROR EVENT_AUDIT_SUCCESS EVENT_AUDIT_FAILURE GetEventType);
$VERSION = '0.7';

use warnings;
use strict;
use Encode;
use constant { EVENT_ERROR => 1, EVENT_WARNING => 2, EVENT_INFORMATION => 4,
               EVENT_AUDIT_SUCCESS => 8, EVENT_AUDIT_FAILURE => 16 };

# Internal function for retrieving UTF-16LE strings from the string table.
sub ShiftUTF16LEAsString($) {
 my $str_ref = shift;
 my $shifted = '';
 my $two;
 my $x = 0;
 my $l = length($$str_ref);
 while ($x < $l) {
  $two = substr($$str_ref, $x, 2);
  $x += 2;
  if ($two eq "\0\0") {
   $$str_ref = substr($$str_ref, $x);
   return decode('UTF-16LE', $shifted);
  } else {
   $shifted .= $two;
  }
 }
}

# Internal function. See 'struct _SID' in WTypes.h for the SID binary format.
sub ShiftSIDAsString($) {
 my $str_ref = shift;
 my $x = 0;                # Shift this much off of the input string.
 my $revision;             # SID version.
 my $sub_authorities;      # Sub authorities partly defines the size of the struct.
 my @sid_id_auth;          # SID Identifier Authority as an array of bytes
 my @sub_auths;            # An array of the actual sub authorities.
 my $sid;                  # The final SID;

 ($revision, $sub_authorities, @sid_id_auth) = unpack('CCCCCCCC', $$str_ref);

 # Unpack all of the sub authorities into @sub_auths. 8 is the offset to the array.
 @sub_auths = unpack("V[$sub_authorities]", substr($$str_ref, 8));

 # Length of the above header, plus the length of one ULONG per sub authority.
 $x = 8 + ($sub_authorities * 4);

 $$str_ref = substr($$str_ref, $x);
 $sid = "S-$revision-";

 # The following code is taken from the Windows Platform SDK regarding, in
 # "Converting a Binary SID to String Format".
 if (($sid_id_auth[0] != 0) || ($sid_id_auth[1] != 0)) {
        $sid .= sprintf('0x%02hx%02hx%02hx%02hx%02hx%02hx',
                    $sid_id_auth[0],
                    $sid_id_auth[1],
                    $sid_id_auth[2],
                    $sid_id_auth[3],
                    $sid_id_auth[4],
                    $sid_id_auth[5]);
 } else {
        $sid .= sprintf('%lu',
                    $sid_id_auth[5]           +
                    ($sid_id_auth[4] <<  8)   +
                    ($sid_id_auth[3] << 16)   +
                    ($sid_id_auth[2] << 24)   );
 }

 if (@sub_auths) { $sid .= '-'; }
 local $" = '-';
 $sid .= "@sub_auths";

 return $sid;
}

=head1 NAME

  Parse::EventLog - Parse information from Windows Event Logs

=head1 SYNOPSIS

 use Parse::EventLog;
 my $elog =
  Parse::EventLog->new("c:\\winnt\\system32\\config\\SysEvent.Evt");

 my %c = $elog->getNewestEvent();
 # Do something interesting here.

 while (%c = $elog->getPreviousEvent()) {
  print localtime($c{TimeGenerated}) .
   ": $c{EventID} generated by $c{Source} on $c{Computer}.\n";
  # Do a lot of interesting somethings here.
 }

=head1 DESCRIPTION

Parse::EventLog is a module designed to allow parsing of binary Windows 2000+ 
Event Logs on any platform. For better Event Log parsing on Windows platforms,
see Win32::EventLog.

When instansiated with the location of an Event Log file, you will be able to
walk through the existing events in chronological order from oldest to newest or
vice versa, as well as being able to extract the whole log to a hash table and
seek through however you wish.

=head1 METHODS

=head2 new

 my $eventlog =
  Parse::EventLog->new("c:\\winnt\\system32\\config\\SysEvent.Evt");

Parse::EventLog constructor, called with the location of the event log file to
open. Returns a Parse::EventLog object or dies on error. Stores the internal
record pointer at the newest event, so the next call to getPreviousEvent will
succeed but getNextEvent will fail (there are no newer events).

=cut

sub new {
  my $class = shift;
  my $filename = shift;
  my %events;
  my @events_by_time;
  my @data;
  my $RecordNumber;
  my $TimeGenerated;
  my $TimeWritten;
  my $EventID;
  my $EventType;
  my $NumStrings;
  my $EventCategory;
  my $unused;

 {
  local $/ = undef; #$INPUT_RECORD_SEPARATOR;
  my $filedata;
  my $file;
  open($file, $filename) or die "Unable to open file $filename";
  binmode($file);
  $filedata = <$file>;
  close $file;
  # Split each record out by its header.
  @data = split(/LfLe/, $filedata); #chr(0x4c).chr(0x66).chr(0x4c).chr(0x65)
 }

 foreach my $rec (@data) {
  if (length($rec) < 92) {  # Magic number! Entry not long enough.
   next;
  }

  ($RecordNumber, $TimeGenerated, $TimeWritten,
   $EventID, $unused, $unused, $EventType, $unused,
   $NumStrings) = unpack('VVVvccccv', $rec); # More magic.

   if ($events{$RecordNumber}) {
    die "Error parsing event log, duplicate record number encountered ($RecordNumber)";
   }

   my $sid_length = unpack('V', substr($rec, 32, 4));

   $events{$RecordNumber}{'TimeGenerated'} = $TimeGenerated;
   $events{$RecordNumber}{'TimeWritten'} = $TimeWritten;
   $events{$RecordNumber}{'RecordNumber'} = $RecordNumber;
   $events{$RecordNumber}{'EventID'} = $EventID;
   $events{$RecordNumber}{'EventType'} = $EventType;
   $events{$RecordNumber}{'NumStrings'} = $NumStrings;

   # Strings are unicode and in a null terminated array. Split on double-null
   # (a unicode null), and then remove nulls. Note: Replaced with wcstombs.

   # Here, 48 is the offset from the start to the string table.
   
   my $stringtable = substr($rec, 48);

   # Strings are encoded as UTF-16LE, so decode them here. Remember, when we
   # split above, we truncated the last null character, so re-add it here.
   $events{$RecordNumber}{'Source'} = ShiftUTF16LEAsString(\$stringtable);
   $events{$RecordNumber}{'Computer'} = ShiftUTF16LEAsString(\$stringtable);
   if ($sid_length) {
    $events{$RecordNumber}{'SID'} = ShiftSIDAsString(\$stringtable);
   } else {
    $events{$RecordNumber}{'SID'} = undef;
   }


   for (my $msgct = 0; $msgct < $NumStrings; $msgct++) {
    push @{$events{$RecordNumber}{'Strings'}}, ShiftUTF16LEAsString(\$stringtable);
   }
 }

 # Sort the events and store it in an array with the oldest being at
 # $events_by_time[0].
 @events_by_time = sort {
  $events{$b}{'TimeGenerated'} <=> $events{$a}{'TimeGenerated'}
 } keys %events;

 my $self = {
    'filename' => $filename,
    'events' => \%events,
    'events_by_time' => \@events_by_time,
    'lastindex' => 0,
    'totalevents' => scalar @events_by_time,
 };
 bless $self, $class;

 return( $self );
};

=head2 getOldestEvent

 my %event = $eventlog->getOldestEvent();

Returns a hash table containing the oldest entry in the event logand moves the
current event pointer to the oldest event.

=cut

#Remember, the list (@events_by_time) goes from newest -> oldest, so we want
# the last entry in that array. Extract the array into $ebt
sub getOldestEvent {
 my $self = shift;
 my @ebt = @{$self->{'events_by_time'}};
 $self->{'lastindex'} = (scalar @ebt) - 1;
 return %{$self->{'events'}->{$ebt[(scalar @ebt) - 1]}};
}

=head2 getNewestEvent

 my %event = $eventlog->getNewestEvent();

Returns a hash table containing the newest entry in the event log and moves the
current event pointer to the newest event.

=cut

# Get the newest entry in the list. Remember, the list goes from
# newest -> oldest, so we want the first entry. Code looks a lot like
# getOldestEvent.
sub getNewestEvent {
 my $self = shift;
 my @ebt = @{$self->{'events_by_time'}};
 $self->{'lastindex'} = 0;
 return %{$self->{'events'}->{$ebt[0]}};
}

=head2 getNextEvent

 my %event = $eventlog->getNextEvent();

Moves us forward in time, returning a hash table containing the next event in
the event log. Returns an empty array/hashtable if there are no more events to
get.

=cut

# Move forward in time.
sub getNextEvent {
 my $self = shift;
 my @ebt = @{$self->{'events_by_time'}};
 my $newindex = $self->{'lastindex'} - 1;
 if ($newindex < 0) { return (); }
 $self->{'lastindex'} = $newindex;
 return %{$self->{'events'}->{$ebt[$newindex]}};
}

=head2 getPreviousEvent

 my %event = $eventlog->getPreviousEvent();

Moves us backward in time, returning a hash table containing the previous event
in the event log. Returns an empty array/hashtable if there are no more events
to get.

=cut

# Move backward in time.
sub getPreviousEvent {
 my $self = shift;
 my @ebt = @{$self->{'events_by_time'}};
 my $newindex = $self->{'lastindex'} + 1;
 if ($newindex > (scalar @ebt) - 1) { return (); }
 $self->{'lastindex'} = $newindex;
 return %{$self->{'events'}->{$ebt[$newindex]}};
}

=head2 getAll

 my %event = $eventlog->getAll();

Returns a copy of the internal Event Log hash table for your own nefarious
pleasure. The main hash table is keyed by the internal RecordNumber (which is
unique to each entry in the file). The value of each of those keys is another
hash table containing the actual event (in the same format as is returned by
the other get*Event functions).

=head3 Example

 my %c = $eventlog->getAll();
 foreach my $a (keys %c) {
  print 'Time: ' . localtime($c{$a}{'TimeGenerated'} . "\n";
  print "Computer: $c{$a}{'Computer'}\n";
  print "Source: $c{$a}{'Source'}\n";
  print 'Type: ' . GetEventType($c{$a}{'EventType'}) . "\n";
  print '-' x 10;
 }

=cut

# Return a copy everything, might be slow.
sub getAll {
 my $self = shift;
 return %{$self->{'events'}};
}

=head2 count

 my $count = $eventlog->count();

Returns the number of entries loaded from the event log.

=cut

sub count {
 my $self = shift;
 return $self->{'totalevents'};
}

=head1 FUNCTIONS

=head2 GetEventType

 print 'Event type:' . GetEventType($c{'EventType'}) . "\n";

Returns a string containing the type of event (Error, Warning, Information,
Audit Success, Audit Failure, or Unknown).

GetEventType is exported by default.

=cut

sub GetEventType {
 my $EventType = shift;
 
 if ($EventType == EVENT_ERROR) {
  return 'Error';
 } elsif ($EventType == EVENT_WARNING) {
  return 'Warning';
 }if ($EventType == EVENT_INFORMATION) {
  return 'Information';
 }if ($EventType == EVENT_AUDIT_SUCCESS) {
  return 'Audit Success';
 }if ($EventType == EVENT_AUDIT_FAILURE) {
  return 'Audit Failure';
 } else {
  return 'Unknown';
 }
}

=head1 CONSTANTS

Several constants are exported by default. These constants are Event Type
definitions. They are as follows:

=over 4

=item * EVENT_ERROR (1)

=item * EVENT_WARNING (2)

=item * EVENT_INFORMATION (4)

=item * EVENT_AUDIT_SUCCESS (8)

=item * EVENT_AUDIT_FAILURE (16)

=back

=head1 RETURN VALUES

On success, all of the get*Event functions return a hash of the current Event.
The hash itself has several keys made up of each of the values in the Event
itself. The following keys are defined (note the capitals):

=head2 TimeGenerated

The epoch time the event was generated, suitable for feeding to gmtime and localtime.

=head2 TimeWritten

The epoch time the event was written to the event log,
usually the same as TimeGenerated (give or take a second). Suitable for feeding
to gmtime and localtime.

=head2 RecordNumber

The record number of the event. This is assigned and due
to the binary nature of the file, and can not be expected to be consistant or
unique on subsequent parsings of the same file.

=head2 EventID

The event identification number, vendor and application
specific. Details can often be found in the Microsoft Knowledge Base.

=head2 EventType

The event type, vendor and application specific.

=head2 Source

Source of the event, usually the application or service name.

=head2 Computer

The name of the computer that generated the event.

=head2 SID

The SID associated with the entry. It is unknown whether this is the SID of the
source PC, the PC the log was stored on, the user, or something else.

=head2 NumStrings

Number of strings in the event.

=head2 Strings

An array reference to all (if any) of the strings included
with the event. Events work a lot like printf format strings, that is, there is
a fixed message with a variable part, such as:

 The %s service failed to start due to the following error:
 The system cannot find the file specified.

In this case, the Strings parameter would contain the name of the service
that would be placed in %s above to complete the event message.

=head1 TO DO

=over 4

=item * Support for random access reads.

=item * Parsing without loading the whole file into memory. By default, the files are only 64k, but it's possible to specify a much larger size.

=item * Descriptions for general Microsoft Windows events.

=item * Rename/alias getNextEvent and getPreviousEvent to getNewerEvent and getOlderEvent?

=back

=head1 KNOWN BUGS

=over 4

=item * Occasional spurious Strings are read from the Event Log. It seems the log
itself specifies they should be there, but there is only garbage data to be
read.

=item * Incomplete parsing. Due to the mysterious nature of the Event Logs and
the lack of documentation on their format, there are several unknown fields in
the log. These fields are currently ignored. If anyone knows what these fields
are for, please let me know.

=back

=head1 CAVEATS

While this module can extract general information on the events, most of the
error description is stored in system libraries specific to the Windows version
and patch level that created the log. At this time, Parse::EventLog does not
have a facility to extract that data.

Assume the Event Viewer shows the following event description:

 The Print Spooler service was successfully sent a stop control.

The data to build this description is stored in two locations. The first, a DLL
file associated with the Service Control Manager, contains the string:

 The %s service was successfully sent a %s control.

The Event Log itself contains the strings:

 Print Spooler
 
and

 stop

Unfortunately, at this time Parse::EventLog is not able to extract the string
data from files outside of the Event Log itself, only the strings contained
within it. At this time, I recommend either parsing the events on the Windows
server itself (via Win32::EventLog or a syslog plugin) or manually looking up
the data elsewhere.

=head1 SEE ALSO

L<Win32::EventLog>

=head1 THANKS

Thanks to Malik of www.whitehats.ca for the initial data structure format that this
module was based on. See: L<http://www.whitehats.ca/main/members/Malik/malik_eventlogs/malik_eventlogs.html>
for more details and his origional work.

=head1 COPYRIGHT

Copyright 2005-2006, John Eaglesham

This module is free software. It may be used, redistributed and/or modified
under the same terms as Perl itself.

=cut

1;
