package CXC::DB::DDL::Util;

# ABSTRACT: CXC::DB::DDL utilities

use v5.26;
use strict;
use warnings;
use experimental 'signatures', 'postderef';

our $VERSION = '0.14';

use List::Util      qw( pairs first );
use Sub::Util       qw( set_subname );
use Module::Runtime ();
use Import::Into;
use Digest::MD5;

use Package::Stash;
use Ref::Util  ();
use Hash::Util ();

use DBI ();

use CXC::DB::DDL::Constants -all;

use namespace::clean;

use base 'Exporter::Tiny';

use constant BASE_TYPE_PACKAGE   => __PACKAGE__ . '::Type';
use constant DEFAULT_FIELD_CLASS => 'CXC::DB::DDL::Field';

our %EXPORT_TAGS = (
    schema_funcs => [qw( xFIELDS xCHECK xTYPE  )],
    misc         => [ 'SQL_TYPE_NAMES', 'SQL_TYPE_VALUES', 'sqlt_producer_map' ],
);

our @EXPORT_OK = ( map { Ref::Util::is_arrayref( $_ ) ? $_->@* : () } values %EXPORT_TAGS );

my sub gen_package_name;
my sub types;

my sub croak {
    require Carp;
    goto \&Carp::croak;
}


my %CACHE = (
    'DBI' => {
        tag   => ':sql_types',
        types => {
            package => gen_package_name( 'DBI' ),
        },
        subs   => {},
        fields => {
            default => +( DEFAULT_FIELD_CLASS ),
        },
    },
);

# this is the base setup for DBI's sql_types, and initializes
# $CACHE{DBI}{types}{symbols};
# types( 'DBI' );

my sub is_supported_dbd ( $dbd ) {
    my %supported;
    @supported{ 'DBI', SUPPORTED_DBDS, keys %CACHE } = ();
    return exists $supported{$dbd};
}

sub gen_package_name ( $dbd, @xtra ) {
    # create a unique class for this blend
    return join q{::}, BASE_TYPE_PACKAGE, Digest::MD5::md5_hex( $dbd // (), @xtra );
}

my sub init ( $globals ) {

    # we can reach this sub through multiple paths; only init once.
    return if exists $globals->{ __PACKAGE__() };

    # request to add support for specified DBD?
    if ( my $request = $globals->{add_dbd} ) {
        Ref::Util::is_hashref( $request )
          or croak( "add_dbd: expected the DBD entry to be a hashref, got @{[ ref $request ]} " );
        my ( $dbd, $tag, $field_class ) = $request->@{ 'dbd', 'tag', 'field_class' };
        defined( $dbd ) && defined( $tag ) && defined( $field_class )
          or croak(
            sprintf( 'add_dbd: missing dbd (%s), tag(%s), or field_class(%s)',
                map { $_ // 'undef' } ( $dbd, $tag, $field_class ), ) );

        # silently ignores attempts to redefine.  should it warn?
        if ( !exists $CACHE{$dbd} ) {
            $CACHE{$dbd} = {
                tag    => $tag,
                fields => {
                    default => $field_class,
                },
                types => {
                    package => gen_package_name( $dbd ),
                },
                subs => {},
            };
            # # initialize symbols
            # types( $dbd );
        }

        # load the dbd types by default.
        $globals->{dbd} //= $dbd;
    }

    my %stash;

    # request particular dbd or fallback to generic DBI support
    my $dbd = $globals->{dbd} // 'DBI';
    Ref::Util::is_ref( $dbd )
      and croak( 'dbd: value must be a scalar' );

    defined( my $cache = $CACHE{$dbd} )
      or croak( "dbd: unsupported DBD: $dbd" );

    $stash{dbd}   = $dbd;
    $stash{cache} = $cache;

    # Field wrappers generated by mk_field
    # override field_class?
    $stash{field_class} = $globals->{field_class} // $cache->{fields}{default};
    $stash{fields}      = $cache->{fields}{ $stash{field_class} } //= {};

    $globals->{ __PACKAGE__() } = \%stash;

    return;
}

# load the types for DBI and requested DBD's into individual
# packages and create a merged hash of names and subs
# cached by a hash of the DBD names.

sub types ( $dbd ) {

    defined( my $cache = $CACHE{$dbd} )
      or croak( "types: unsupported dbd: $dbd" );

    return $cache->{types}{symbols} if defined $cache->{types}{symbols};

    my %symbol;

    # if we're not setting up DBI, add DBI's symbols. Gather DBI's
    # symbols if necessary.
    %symbol = ( $CACHE{DBI}{types}{symbols} //= types( 'DBI' ) )->%*
      if $dbd ne 'DBI';

    my $stash  = Package::Stash->new( $cache->{types}{package} );
    my $module = Module::Runtime::use_module( $dbd eq 'DBI' ? $dbd : "DBD::$dbd" );
    $module->import::into( $stash->name, $cache->{tag} );
    my $lsymbol   = $stash->get_all_symbols( 'CODE' );
    my @from_keys = keys $lsymbol->%*;

    # strip off SQL_ from DBI types
    my @to_keys
      = $dbd eq 'DBI'
      ? map { s/^SQL_//r } @from_keys
      : @from_keys;

    @symbol{@to_keys} = $lsymbol->@{@from_keys};

    Hash::Util::lock_hash( %symbol );
    return $cache->{types}{symbols} = \%symbol;
}


# ensure that $field_class has been required prior to calling this.
my sub _mk_field ( $name, $type, $attr, $field_class ) {

    # do this here and use require to prevent import loop from
    # anything which uses CXC::DB::DDL::Field, which uses this module

    set_subname "${name}::_mk_field", sub ( $field ) {
        $field_class->new( {
            name        => $field,
            data_type   => $type,
            is_nullable => 0,
            $attr->%*,
        } );
    };

}

my sub expand_field_sub ( $cache, $field_class, $name, $type ) {

    my $symbols = ( ( $cache->{fields} //= {} )->{$field_class} //= {} )->{symbols} //= {};

    return "&$name", $symbols->{$name}
      if exists $symbols->{$name};

    Module::Runtime::use_module( $field_class );

    my $sub = $symbols->{$name} = set_subname $name, sub ( %attr ) {
        _mk_field( $name, $type, \%attr, $field_class );
    };

    return ( "&$name", $sub );
}












sub _exporter_validate_opts ( $class, $globals ) {
    init( $globals );
}

sub _exporter_expand_tag ( $class, $name, $value, $globals ) {

    # _exporter_expand_tag is called before _exporter_validate_opts,
    # so init just in case
    init( $globals );
    my $stash = $globals->{ __PACKAGE__() };

    # mindless copy from Exporter::Tiny::_exporter_expand_tag
    return (
        $class->_exporter_merge_opts( $value, $globals, @EXPORT_OK, keys types( $stash->{dbd} )->%*, ) )
      if $name eq 'all';

    return ( $class->_exporter_merge_opts( $value, $globals, keys types( $stash->{dbd} )->%*, ) )
      if $name eq 'type_funcs';

    $class->SUPER::_exporter_expand_tag( $name, $value, $globals );
}

sub _exporter_expand_sub ( $class, $name, $value, $globals, $permitted ) {

    my $stash = $globals->{ __PACKAGE__() };
    my $cache = $stash->{cache};
    my $dbd   = $stash->{dbd};

    if ( $name eq 'SQL_TYPE_NAMES' ) {
        return "&$name", $cache->{subs}{$name} //= set_subname $name, do {
            my @names = sort keys types( $dbd )->%*;
            sub { @names };
        };
    }

    if ( $name eq 'SQL_TYPE_VALUES' ) {
        return "&$name", $cache->{subs}{$name} //= set_subname $name, do {
            my $types  = types( $dbd );
            my @values = map { $types->{$_}->() } sort keys $types->%*;
            sub { @values };
        };
    }

    if ( $name eq 'xTYPE' ) {
        # field class may be specific to this use of Util, rather than dbd specific,
        my $field_class = Module::Runtime::use_module( $stash->{field_class} );
        return "&$name", $cache->{subs}{$name}{$field_class} //= set_subname $name, sub ( $type, %attr ) {
            _mk_field( $name, $type, \%attr, $field_class );
        };
    }

    # $symbols is a locked hash, so can't just grab a value
    my $symbols = types( $dbd );
    if ( exists $symbols->{$name} && defined( my $sub = $symbols->{$name} ) ) {
        return expand_field_sub( $cache, $stash->{field_class}, $name, $sub->() );
    }

    $class->SUPER::_exporter_expand_sub( $name, $value, $globals, $permitted );
}








































































































































sub xFIELDS ( @fields ) {
    return fields => [ map { $_->value->( $_->key ) } pairs( @fields ) ];
}
























sub xCHECK ( $field, @values ) {
    ( check => sprintf( "$field in ( %s )", join( ', ', map { qq("$_") } @values ) ), )
}










sub sqlt_producer_map ( $dbd ) {
    state %map = ( +( DBD_POSTGRESQL ) => 'PostgreSQL' );
    return $map{$dbd} // $dbd;
}


1;

#
# This file is part of CXC-DB-DDL
#
# This software is Copyright (c) 2022 by Smithsonian Astrophysical Observatory.
#
# This is free software, licensed under:
#
#   The GNU General Public License, Version 3, June 2007
#

__END__

=pod

=for :stopwords Diab Jerius Smithsonian Astrophysical Observatory TYPENAME VARCHAR xCHECK
xFIELDS xTYPE DBD dbd

=head1 NAME

CXC::DB::DDL::Util - CXC::DB::DDL utilities

=head1 VERSION

version 0.14

=head1 SYNOPSIS

  use CXC::DB::DDL::Util -all;

  # import xFIELDS, xCHECK, xTYPE
  use CXC::DB::DDL::Util -schema_funcs;

  # import type function generators (e.g. INTEGER, DOUBLE )
  use CXC::DB::DDL::Util -type_funcs;

  # import type function generators with a prefix
  # (e.g. MK_INTEGER, MK_DOUBLE )
  use CXC::DB::DDL::Util { prefix => 'MK_' }, -type_funcs;


  use DBI;
  $ddl = CXC::DB::DDL->new( [ {
              name => 'observation',
              xFIELDS(
                  segment     => MK_INTEGER,
                  obsid       => INTEGER( is_primary_key => 1 ),
                  cohort      => TEXT,
                  date        => xTYPE( [DBI::SQL_TIMESTAMP] ),
                  dec         => TEXT,
                  object      => TEXT,
                  event_count => INTEGER,
                  exposure    => REAL,
                  instrument  => TEXT,
                  obs_cycle   => INTEGER,
                  prop_cycle  => INTEGER,
                  ra          => TEXT,
              ),
          },
      ] );

=head1 DESCRIPTION

C<CXC::DB::DDL::Util> provides generators to ease creation of,
amongst, others, L<CXC::DB::DDL::Field> objects.  It uses
L<Exporter::Tiny> as its base exporter, allowing renaming of exported
symbols and other things.

=head2 DBD Specific Types

Some database drivers (e.g. L<DBD::Pg>) provide additional types.  For the
generic mechanism to add these see L</ADVANCED USES>.

To access the PostgreSQL types, first load the L<DBD::Pg> specific
subclass of L<CXC::DB::DDL::Field>, then pass the global B<< dbd =>
'Pg' >> to B<CXC::DB::DDL::Util>:

  use CXC::DB::DDL::Field::Pg;
  use CXC::DB::DDL::Util { dbd => 'Pg' }, -type_funcs;

The PostgreSQL specific type function generators are now available as e.g., B<PG_JSONB>
(the B<PG_> prefix is I<not> removed):

  @fields = xFIELDS(
      segment        => INTEGER,
      pars           => PG_JSONB,
  );

The generated field objects will be in the L<CXC::DB::DDL::Field::Pg>
class.

=head1 SUBROUTINES

=head2 SQL_TYPE_NAMES

  @typenames = SQL_TYPE_NAMES;

returns all of the DBI supported types (without the C<SQL_> prefix)

=head2 I<TYPENAME>

I<TYPENAME> is one of the SQL types recognized by L<DBI> or by
a particular L<DBD> driver (see L</DBD Specific Types>).

See L<CXC::DB::DDL::Constants/sql_type_constants> for more information.

Called as, e.g.

   INTEGER( %attr )

these are generators which return subroutines with the following signature:

   sub ( $field_name )

which return a L<CXC::DB::DDL::Field> object with the specified SQL
datatype (in this example C<INTEGER>), field name (C<$field_name>)
and attributes (C<%attr>).

These are available for individual export or in entirety via the
C<type_funcs> tag.

They are typically used in conjunction with the L</xFIELDS>
subroutine, e.g.

  xFIELDS(
      segment        => INTEGER,
      obsid          => INTEGER( is_primary_key => 1 ),
      target_type    => VARCHAR( is_nullable => 1 ),
  )

L</xFIELDS> essentially turns this into:

  fields => [
      INTEGER()->('segment'),
      INTEGER(is_primary_key => 1 )->('obsid'),
      VARCHAR(is_nullable => 1 )->( 'target_type' ),
  ]

which is more painful to write and look at. So don't.

=head2 xTYPE

  xTYPE ( $type, %attr )

A generic form of e.g., L</INTEGER>. Type is a numeric C<DATA_TYPE> made
available from L<DBI> or a L<DBD> driver (e.g. L<DBD::Pg>).

Be careful when importing the type functions from
B<CXC::DB::DDL::Util> for driver specific types.

For example, the type constants provided by L<DBI> all begin with
B<SQL_>; the L<DBI> type functions provided by B<CXC::DB::DDL::Util>
match those, but with the B<SQL_> prefix removed, so there is no
collision between L<DBI> type constant names and B<CXC::DB::DDL::Util>
type function names, e.g.:

  use DBI 'SQL_INTEGER';
  use CXC::DB::DDL:Util -type_funcs, 'xTYPE';

  xTYPE( SQL_INTEGER, ... );
  INTEGER( ... )

However, because it is possible that a driver specific type might have
a similar name to a L<DBI> provided type (e.g. for the fictitious
B<DBD::MYDB> driver, B<MYDB_INTEGER> vs B<SQL_INTEGER>), when
constructing the type functions for driver specific types, the leading
prefix is I<not> removed.

So,

  # DONT DO THIS
  use DBD::MYDB 'MYDB_INTEGER';
  use CXC::DB::DDL:Util { dbd => 'MYDB' }, -type_funcs, 'xTYPE';

  xTYPE( MYDB_INTEGER, ... );
  MYDB_INTEGER( ... )

will cause a collision between B<MYDB_INTEGER>, the type constant, and
B<MYDB_INTEGER>, the type function.  To avoid this, either rename the type
function generators upon import (see
L<Exporter::Tiny::Manual::Importing>):

  # DO THIS
  use DBD::MYDB 'MYDB_INTEGER';
  use CXC::DB::DDL:Util
    { as => sub { $_[0] =~ s/MYDB_/MK_/r },
      dbd => 'MYDB',
    },
    -type_funcs, 'xTYPE';

  xTYPE( MYDB_INTEGER, ... );
  MK_INTEGER( ... )

or use the fully qualified type constant name:

  # DO THIS
  use DBD::MYDB ();
  use CXC::DB::DDL:Util { dbd => 'MYDB'}, -type_funcs, 'xTYPE';

  xTYPE( DBD::MYDB::MYDB_INTEGER, ... );
  INTEGER( ... )

=head2 xFIELDS

   @field_spec = xFIELDS( array of  Tuple[ NonEmptyStr, CodeRef ] );

returns a list of

   fields => \@spec,

where C<@spec> generated by running

     CodeRef->(NonEmptyStr)

for each tuple.

=head2 xCHECK

DEPRECATED; use a table constraint B<type> set to the constant
L<CHECK_C|CXC::DB::DDL::Constants/CHECK_C>, as follows:

  my $table = CXC::DB::DDL::Table_>new( ...,
      constraints => [
          {
              expression => '"type" in ( "a", "b", "c" )',
              type       => CHECK_C,
          },
     ] );

DEPRECATED USE BELOW:

    $string = xCHECK( $field, @values )

generates a check constraint as a string which looks like

  $field in ( $value[0], $value[1], ...  )

=head2 sqlt_producer_map

  $sqlt_producer = sqlt_producer_map( $dbd );

Map C<$dbd> (typically from C<$dbh->{Driver}{NAME}>) to what L<SQL::Translator> wants.
Actually, this just checks for known (to me) deviations; everything else is returned as.

=head1 SUPPORT

=head2 Bugs

Please report any bugs or feature requests to bug-cxc-db-ddl@rt.cpan.org  or through the web interface at: L<https://rt.cpan.org/Public/Dist/Display.html?Name=CXC-DB-DDL>

=head2 Source

Source is available at

  https://gitlab.com/djerius/cxc-db-ddl

and may be cloned from

  https://gitlab.com/djerius/cxc-db-ddl.git

=head1 SEE ALSO

Please see those modules/websites for more information related to this module.

=over 4

=item *

L<CXC::DB::DDL|CXC::DB::DDL>

=back

=head1 AUTHOR

Diab Jerius <djerius@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2022 by Smithsonian Astrophysical Observatory.

This is free software, licensed under:

  The GNU General Public License, Version 3, June 2007

=cut
