################################################################################
# SLOCCounter performs base SLOC counts, and compares itself against other
# SLOCCounter objects.
# Change Log:
#-------------------------------------------------------------------------------
# JRF JAN 27 2005 EST Created
# JRF JAN 28 2005 EST Updated for comparing against another SLOCCounter
#-------------------------------------------------------------------------------
# See POD for more details
################################################################################
package Utility::SLOCCounter;

use strict;
use warnings;

use Utility::Rule;

sub new{
	my ($class, $file_name) = @_;
	
	open (IN, "<$file_name") or die "SLOCCounter can't open filename:\n$!";
	my @lines = <IN>;
	
	my $struct = 
	{
		verbose => 1,
		lines   => \@lines,
		cleaned_lines => undef,
		counted_lines => undef,
		rules => [ new Utility::Rule::WhitespaceRule(),
				   new Utility::Rule::CommentRule(),
				   new Utility::Rule::BracketRule(),
				   new Utility::Rule::EndAndDataRule() ]
	};
	my $self = bless $struct, $class;
	$self->clean_lines();
	$self->count();
	return $self;
}

sub count{
	my ($self) = @_;
	my $count = 0;
	my @counted;
	
	LINE_LOOP:
	foreach my $line (@{$self->{cleaned_lines}}){
		foreach my $rule (@{$self->{rules}}){
			if ($rule->apply($line) == $rule->{FAIL}){
				next LINE_LOOP;
			}
			elsif($rule->apply($line) == $rule->{STOP}){
				last LINE_LOOP;
			}
		}
		$count++;
		push @counted, $line;
	}
	$self->{counted_lines} = \@counted;
	return $count;
}

sub clean_lines{
	my($self) = @_;
	my @cleaned;
	foreach my $line (@{$self->{lines}}){
		chomp($line);
		$line =~ s/^\s+//;
		$line =~ s/\s+$//;
		push (@cleaned, $line) if $line;
	}
	$self->{cleaned_lines} = \@cleaned;
}

sub compare{
	my($self, $old_sloc) = @_;
	
	my @new_lines = @{$self->{counted_lines}};
	my @old_lines = @{$old_sloc->{counted_lines}};
	
	my $deleted = $self->_generic_compare(\@old_lines, \@new_lines);
	my $added   = $self->_generic_compare(\@new_lines, \@old_lines);
	return ($deleted, $added, $old_sloc->count() - $deleted)
}

sub _generic_compare{
	my ($self, $comparethis, $tothis) = @_;
	
	my @compare = @{$comparethis};
	my @to      = @{$tothis};
	my $found 	= -1;
	my $base_count = @compare;
	my $count   = 0;
	
	for(my $i=0; $i< @compare; $i++){
		for(my $j=0; $j< @to; $j++){
			if (($to[$j]) && ($compare[$i] eq $to[$j])){
				$found = $j;
				print "MATCH: " . $compare[$i] . " === " . $to[$j] . "\n" if $self->{verbose};
				last;
			}
		}
		
		if($found > -1){
			delete $to[$found];
			$count++;
			$found = -1;
		}
	}
	return $base_count - $count;
}

1;

__END__

=pod

=head1 NAME

SLOCCounter.pm

=head1 SYNOPSIS

my $sloc = new Utility::SLOCCounter("some_file.pl");
$sloc->count(); # returns the Source Lines of Code

=head1 SUMMARY

SLOCCounter is a source lines of code counter primarily designed for perl, although it is not limited to Perl.
It uses a simple Rule based mechanism to determine if a line in a file counts.
See Utility::Rule for more information on rules.

=head2 Utility::SLOCCounter(*)

Takes a file handle for source code.
Opens file handle and converts contents to lines of code.
Creates Perl rules for counting use.

=head3 count

Applys all rules to each line of code.
If a rule passes, then the count gets incremented.
If a rule declares a stop, then the couting stops.

=head3 clean_lines

Trips leading and trailling space off of the lines.

=head3 compare

Takes another SLCO which it treats as old.
Returns a list in the form of (deleted, added, reused)

=head1 AUTHOR

johnfraser <johnfraser@somewhere.com<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (c) 2005, John Fraser C<< <raptnor@yahoo.com> >>. All rights reserved.

This module is free software; you can redistribute it and/or
modify it under the same terms as Perl itself. See L<perlartistic>.


=head1 DISCLAIMER OF WARRANTY

BECAUSE THIS SOFTWARE IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY
FOR THE SOFTWARE, TO THE EXTENT PERMITTED BY APPLICABLE LAW. EXCEPT WHEN
OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES
PROVIDE THE SOFTWARE "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOFTWARE IS WITH
YOU. SHOULD THE SOFTWARE PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL
NECESSARY SERVICING, REPAIR, OR CORRECTION.

IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING
WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR
REDISTRIBUTE THE SOFTWARE AS PERMITTED BY THE ABOVE LICENCE, BE
LIABLE TO YOU FOR DAMAGES, INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL,
OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OR INABILITY TO USE
THE SOFTWARE (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING
RENDERED INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A
FAILURE OF THE SOFTWARE TO OPERATE WITH ANY OTHER SOFTWARE), EVEN IF
SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGES.

=cut