#!/usr/local/bin/perl
#
#Base class for sacctmgr entities which can do all of
#"list", "modify", "add" and "delete".
#I.e. have full read/write capabilities

package Slurm::Sacctmgr::EntityBaseRW;
use strict;
use warnings;
use base qw(
	Slurm::Sacctmgr::EntityBaseListable
	Slurm::Sacctmgr::EntityBaseModifiable
	Slurm::Sacctmgr::EntityBaseAddDel
);
use Carp qw(carp croak);

sub compare($$)
#Compare two instances, field by field (using
#Returns a list ref of triplets [ fieldname, value1, value2 ] for every
#field that differs.  If no differences, returns undef.
#value1 is the value for the invocant, value2 is the value of the field
#of the explicit argument.
#
#Compares fieldsd from _sacctmgr_fields_in_order;
{	my $obj1 = shift;
	my $obj2 = shift;

	my $me = 'compare';
	croak "$me must be called as an instance method at "
		unless $obj1 && ref($obj1);
	croak "Bad invalid argument to $me: $obj2"
		unless $obj2 && ref($obj2) eq ref($obj1);

	my $fields = $obj1->_sacctmgr_fields_in_order;

	my @diffs = ();
	foreach my $fld (@$fields)
	{	my $meth = $fld;
		my $val1 = $obj1->$meth;
		my $val2 = $obj2->$meth;

		if ( defined $val1 )
		{	if ( defined $val2 )
			{	push @diffs, [ $fld, $val1, $val2 ]
					unless $val1 eq $val2;
			} else
			{	push @diffs, [ $fld, $val1, undef ];
			}
		} elsif ( defined $val2 )
		{	push @diffs, [ $fld, undef, $val2 ];
		}
	}

	return unless @diffs;
	return [@diffs];
}

sub sacctmgr_save_me($$@)
{	my $obj = shift;
	my $sacctmgr = shift;
	my %extra = @_;

	my $me = 'sacctmgr_save_me';
	croak "$me must be called as an instance method at "
		unless $obj && ref($obj);
	croak "No/invalid Slurm::Sacctmgr object passed to $me at "
		unless $sacctmgr && ref($sacctmgr);

	my $current = $obj->sacctmgr_list_me($sacctmgr);
	unless ( defined $current )
	{	#No current entity matching me, so just do sacctmgr_add_me
		return $obj->sacctmgr_add_me($sacctmgr, %extra);
	}
	croak "Error looking up entity in $me : $current at"
		unless ref($current);

	my $diffs = $obj->compare($current);
	return unless ( $diffs ); #Nothing to do

	my %updates = ();
	foreach my $rec (@$diffs)
	{	my ($fld, $val1, $val2) = @$rec;
		$val1 = '' unless defined $val1;
		$updates{$fld} = $val1;
	}
	%updates = ( %updates, %extra );

	$obj->sacctmgr_modify_me($sacctmgr, %updates);
}


1;
__END__

=head1 NAME

Slurm::Sacctmgr::EntityBaseAddDel

=head1 SYNOPSIS

  package Slurm::Sacctmgr::Account;
  use base qw(Slurm::Sacctmgr::EntityBaseRW);


=head1 DESCRIPTION

This is the base class for entities managed by sacctmgr, for entities
which support the full set of read/write commands, i.e. can do all
of

=over 4

=item add (i.e. inherits from B<Slurm::Sacctmgr::EntityBaseAddDel> )

=item delete (i.e. inherits from B<Slurm::Sacctmgr::EntityBaseAddDel> )

=item list (i.e. inherits from B<Slurm::Sacctmgr::EntityBaseListable> )

=item modify (i.e. inherits from B<Slurm::Sacctmgr::EntityBaseModifiable> )

=back

And for most part, this class just inherits from the classes above.
But it also defines

=over 4

=item B<compare>($sacctmgr, $instance2)

This compares the invocant to $instance2, field by field.
It returns undef if no differences, or a list of triplets
[ fieldname, value1, value2 ] for each field fieldname that
differs, with value1 being the value in the invocant and
value2 the value in instance2.
Compares fields from B<_sacctmgr_fields_in_order>

=item B<sacctmgr_save_me>($sacctmgr, [ extra1 => val1, [ extra2=>val2 ... ]])

This is an instance method, and calls B<sacctmgr_modify> to update the
entity with the same name to the values of the Perl object.  Obviously
cannot be used to change the name of an object.  If no entity exists in
sacctmgr db, does same as B<sacctmgr_add_me>.
B<NOTE:> The extra arguments will override data members if there is a conflict.

=back

=head2 EXPORT

Nothing.  Pure OO interface.

=head1 SEE ALSO

B<EntityBase>

=head1 AUTHOR

Tom Payerle, payerle@umd.edu

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2014-2016 by the University of Maryland.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

