package SMS::Send::Plivo;

use strict;
use warnings;

use 5.008_005;

use Carp;
use JSON::PP;
use WWW::Plivo::API;

use parent qw(SMS::Send::Driver);

our $VERSION = '0.01_02';
$VERSION = eval $VERSION;

=encoding utf-8

=head1 NAME

SMS::Send::Plivo - SMS::Send backend for Plivo API

=head1 SYNOPSIS

(This is a Plivo port of the existing SMS::Send::Twilio module.  See
that module for more detailed documentation.)

  use SMS::Send;
  # Create an object. There are three required values:
  my $sender = SMS::Send->new('Plivo',
    _accounauthid => 'ACb657bdcb16f06893fd127e099c070eca',
    _authtoken  => 'b857f7afe254fa86c689648447e04cff',
    _from       => '+15005550006',
  );

  # Send a message to me
  my $sent = $sender->send_sms(
    text => 'Messages can be up to 1600 characters',
    to   => '+31645742418',
  );

  # Did it send?
  if ( $sent ) {
    print "Sent test message\n";
  } else {
    print "Test message failed\n";
  }

=head1 DESCRIPTION

SMS::Send::Plivo is an SMS::Send driver for the Plivo web service.

=pod

=head2 new

  # Create a new sender using this driver
  my $sender = SMS::Send->new('Plivo',
    _accountauthid => 'ACb657bdcb16f06893fd127e099c070eca',
    _authtoken  => 'b857f7afe254fa86c689648447e04cff',
    _from       => '+15005550006',
  );

The C<new> constructor takes three parameters, which should be passed
through from the L<SMS::Send> constructor.

=head2 send_sms

It's really easy; if it returns a true value, sending the message was OK.
If not we'd see an error message on STDERR.

  # Send a message to me
  my $sent = $sender->send_sms(
    text => 'Messages can be up to 1600 characters',
    to   => '+31645742418',
  );

=cut

sub new {
    my $class  = shift;
    my %params = @_;

    # check required parameters
    for my $param (qw ( _accountauthid _from _authtoken )) {
        exists $params{$param}
          or croak $class . "->new requires $param parameter";
    }

    my $self = \%params;
    bless $self, $class;

    # Create plivo object
    $self->{plivo} = WWW::Plivo::API->new(
        AccountAuthid => $self->{_accountauthid},
        AuthToken  => $self->{_authtoken},
    ) or croak $class . "->new can't set up connection: $!";

    return $self;
}

sub send_sms {
    my $self   = shift;
    my %params = @_;

    # Get the message and destination
    my $message   = delete $params{text};
    my $recipient = delete $params{to};

    my $response = $self->{plivo}->POST(
        'Message',
        From => $self->{_from},
        To   => $recipient,
        Body => $message,
    );

    if ( ($response->{code} >= 200) && ($response->{code} <= 204) ) {
        my $result = JSON::PP->new->utf8->decode( $response->{content} );

        if ( $result->{message_uuid} ) {
            return $result->{message_uuid};
        }
    }
    elsif ( $response->{code} == '400' ) {
        my $result = JSON::PP->new->utf8->decode( $response->{content} );
        if ( $result->{message} ) {
            croak "$result->{message}";
        }
    }

    croak "Can't send SMS: $response->{code} $response->{message}";
}

=head1 AUTHOR

Mike Lempriere, C<< <mikevntnr at cpan.org> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-sms-send-plivo at rt.cpan.org>, or through
the web interface at L<https://rt.cpan.org/NoAuth/ReportBug.html?Queue=SMS-Send-Plivo>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.




=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc SMS::Send::Plivo


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<https://rt.cpan.org/NoAuth/Bugs.html?Dist=SMS-Send-Plivo>

=item * CPAN Ratings

L<https://cpanratings.perl.org/d/SMS-Send-Plivo>

=item * Search CPAN

L<https://metacpan.org/release/SMS-Send-Plivo>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

This software is Copyright (c) 2023 by Mike Lempriere.

This is free software, licensed under:

  The Artistic License 2.0 (GPL Compatible)


=head1 SEE ALSO

L<SMS::Send>
L<WWW::Plivo::API>

=cut

1; # End of SMS::Send::Plivo
