/* LibTomCrypt, modular cryptographic library -- Tom St Denis */
/* SPDX-License-Identifier: Unlicense */

#include "tomcrypt_private.h"

/**
  @file ecc.c
  ECC Crypto, Tom St Denis
*/

#ifdef LTC_MECC

/* This array holds the curve parameters.
 * Curves (prime field only) are taken from:
 *   - http://www.secg.org/collateral/sec2_final.pdf (named: SECP*)
 *   - http://nvlpubs.nist.gov/nistpubs/FIPS/NIST.FIPS.186-4.pdf (named: NISTP*)
 *   - ANS X9.62 (named: PRIMEP*)
 *   - http://www.ecc-brainpool.org/download/Domain-parameters.pdf (named: BRAINPOOLP*)
 */
const ltc_ecc_curve ltc_ecc_curves[] = {
#ifdef LTC_ECC_SECP112R1
{
   /* prime    */ "DB7C2ABF62E35E668076BEAD208B",
   /* A        */ "DB7C2ABF62E35E668076BEAD2088",
   /* B        */ "659EF8BA043916EEDE8911702B22",
   /* order    */ "DB7C2ABF62E35E7628DFAC6561C5",
   /* Gx       */ "09487239995A5EE76B55F9C2F098",
   /* Gy       */ "A89CE5AF8724C0A23E0E0FF77500",
   /* cofactor */ 1,
   /* OID      */ "1.3.132.0.6"
},
#endif
#ifdef LTC_ECC_SECP112R2
{
   /* prime    */ "DB7C2ABF62E35E668076BEAD208B",
   /* A        */ "6127C24C05F38A0AAAF65C0EF02C",
   /* B        */ "51DEF1815DB5ED74FCC34C85D709",
   /* order    */ "36DF0AAFD8B8D7597CA10520D04B",
   /* Gx       */ "4BA30AB5E892B4E1649DD0928643",
   /* Gy       */ "ADCD46F5882E3747DEF36E956E97",
   /* cofactor */ 4,
   /* OID      */ "1.3.132.0.7"
},
#endif
#ifdef LTC_ECC_SECP128R1
{
   /* prime    */ "FFFFFFFDFFFFFFFFFFFFFFFFFFFFFFFF",
   /* A        */ "FFFFFFFDFFFFFFFFFFFFFFFFFFFFFFFC",
   /* B        */ "E87579C11079F43DD824993C2CEE5ED3",
   /* order    */ "FFFFFFFE0000000075A30D1B9038A115",
   /* Gx       */ "161FF7528B899B2D0C28607CA52C5B86",
   /* Gy       */ "CF5AC8395BAFEB13C02DA292DDED7A83",
   /* cofactor */ 1,
   /* OID      */ "1.3.132.0.28"
},
#endif
#ifdef LTC_ECC_SECP128R2
{
   /* prime    */ "FFFFFFFDFFFFFFFFFFFFFFFFFFFFFFFF",
   /* A        */ "D6031998D1B3BBFEBF59CC9BBFF9AEE1",
   /* B        */ "5EEEFCA380D02919DC2C6558BB6D8A5D",
   /* order    */ "3FFFFFFF7FFFFFFFBE0024720613B5A3",
   /* Gx       */ "7B6AA5D85E572983E6FB32A7CDEBC140",
   /* Gy       */ "27B6916A894D3AEE7106FE805FC34B44",
   /* cofactor */ 4,
   /* OID      */ "1.3.132.0.29"
},
#endif
#ifdef LTC_ECC_SECP160R1
{
   /* prime    */ "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF7FFFFFFF",
   /* A        */ "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF7FFFFFFC",
   /* B        */ "1C97BEFC54BD7A8B65ACF89F81D4D4ADC565FA45",
   /* order    */ "0100000000000000000001F4C8F927AED3CA752257",
   /* Gx       */ "4A96B5688EF573284664698968C38BB913CBFC82",
   /* Gy       */ "23A628553168947D59DCC912042351377AC5FB32",
   /* cofactor */ 1,
   /* OID      */ "1.3.132.0.8"
},
#endif
#ifdef LTC_ECC_SECP160R2
{
   /* prime    */ "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFAC73",
   /* A        */ "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFAC70",
   /* B        */ "B4E134D3FB59EB8BAB57274904664D5AF50388BA",
   /* order    */ "0100000000000000000000351EE786A818F3A1A16B",
   /* Gx       */ "52DCB034293A117E1F4FF11B30F7199D3144CE6D",
   /* Gy       */ "FEAFFEF2E331F296E071FA0DF9982CFEA7D43F2E",
   /* cofactor */ 1,
   /* OID      */ "1.3.132.0.30"
},
#endif
#ifdef LTC_ECC_SECP160K1
{
   /* prime    */ "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFAC73",
   /* A        */ "0000000000000000000000000000000000000000",
   /* B        */ "0000000000000000000000000000000000000007",
   /* order    */ "0100000000000000000001B8FA16DFAB9ACA16B6B3",
   /* Gx       */ "3B4C382CE37AA192A4019E763036F4F5DD4D7EBB",
   /* Gy       */ "938CF935318FDCED6BC28286531733C3F03C4FEE",
   /* cofactor */ 1,
   /* OID      */ "1.3.132.0.9"
},
#endif
#ifdef LTC_ECC_SECP192R1
{
   /* prime    */ "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFF",
   /* A        */ "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFC",
   /* B        */ "64210519E59C80E70FA7E9AB72243049FEB8DEECC146B9B1",
   /* order    */ "FFFFFFFFFFFFFFFFFFFFFFFF99DEF836146BC9B1B4D22831",
   /* Gx       */ "188DA80EB03090F67CBF20EB43A18800F4FF0AFD82FF1012",
   /* Gy       */ "07192B95FFC8DA78631011ED6B24CDD573F977A11E794811",
   /* cofactor */ 1,
   /* OID      */ "1.2.840.10045.3.1.1"
},
#endif
#ifdef LTC_ECC_PRIME192V2
{
   /* prime    */ "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFF",
   /* A        */ "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFC",
   /* B        */ "CC22D6DFB95C6B25E49C0D6364A4E5980C393AA21668D953",
   /* order    */ "FFFFFFFFFFFFFFFFFFFFFFFE5FB1A724DC80418648D8DD31",
   /* Gx       */ "EEA2BAE7E1497842F2DE7769CFE9C989C072AD696F48034A",
   /* Gy       */ "6574D11D69B6EC7A672BB82A083DF2F2B0847DE970B2DE15",
   /* cofactor */ 1,
   /* OID      */ "1.2.840.10045.3.1.2"
},
#endif
#ifdef LTC_ECC_PRIME192V3
{
   /* prime    */ "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFF",
   /* A        */ "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFC",
   /* B        */ "22123DC2395A05CAA7423DAECCC94760A7D462256BD56916",
   /* order    */ "FFFFFFFFFFFFFFFFFFFFFFFF7A62D031C83F4294F640EC13",
   /* Gx       */ "7D29778100C65A1DA1783716588DCE2B8B4AEE8E228F1896",
   /* Gy       */ "38A90F22637337334B49DCB66A6DC8F9978ACA7648A943B0",
   /* cofactor */ 1,
   /* OID      */ "1.2.840.10045.3.1.3"
},
#endif
#ifdef LTC_ECC_SECP192K1
{
   /* prime    */ "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFEE37",
   /* A        */ "000000000000000000000000000000000000000000000000",
   /* B        */ "000000000000000000000000000000000000000000000003",
   /* order    */ "FFFFFFFFFFFFFFFFFFFFFFFE26F2FC170F69466A74DEFD8D",
   /* Gx       */ "DB4FF10EC057E9AE26B07D0280B7F4341DA5D1B1EAE06C7D",
   /* Gy       */ "9B2F2F6D9C5628A7844163D015BE86344082AA88D95E2F9D",
   /* cofactor */ 1,
   /* OID      */ "1.3.132.0.31"
},
#endif
#ifdef LTC_ECC_SECP224R1
{
   /* prime    */ "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF000000000000000000000001",
   /* A        */ "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFFFFFFFFFE",
   /* B        */ "B4050A850C04B3ABF54132565044B0B7D7BFD8BA270B39432355FFB4",
   /* order    */ "FFFFFFFFFFFFFFFFFFFFFFFFFFFF16A2E0B8F03E13DD29455C5C2A3D",
   /* Gx       */ "B70E0CBD6BB4BF7F321390B94A03C1D356C21122343280D6115C1D21",
   /* Gy       */ "BD376388B5F723FB4C22DFE6CD4375A05A07476444D5819985007E34",
   /* cofactor */ 1,
   /* OID      */ "1.3.132.0.33"
},
#endif
#ifdef LTC_ECC_SECP224K1
{
   /* prime    */ "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFE56D",
   /* A        */ "00000000000000000000000000000000000000000000000000000000",
   /* B        */ "00000000000000000000000000000000000000000000000000000005",
   /* order    */ "010000000000000000000000000001DCE8D2EC6184CAF0A971769FB1F7",
   /* Gx       */ "A1455B334DF099DF30FC28A169A467E9E47075A90F7E650EB6B7A45C",
   /* Gy       */ "7E089FED7FBA344282CAFBD6F7E319F7C0B0BD59E2CA4BDB556D61A5",
   /* cofactor */ 1,
   /* OID      */ "1.3.132.0.32"
},
#endif
#ifdef LTC_ECC_SECP256R1
{
   /* prime    */ "FFFFFFFF00000001000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFF",
   /* A        */ "FFFFFFFF00000001000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFC",
   /* B        */ "5AC635D8AA3A93E7B3EBBD55769886BC651D06B0CC53B0F63BCE3C3E27D2604B",
   /* order    */ "FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC632551",
   /* Gx       */ "6B17D1F2E12C4247F8BCE6E563A440F277037D812DEB33A0F4A13945D898C296",
   /* Gy       */ "4FE342E2FE1A7F9B8EE7EB4A7C0F9E162BCE33576B315ECECBB6406837BF51F5",
   /* cofactor */ 1,
   /* OID      */ "1.2.840.10045.3.1.7"
},
#endif
#ifdef LTC_ECC_SECP256K1
{
   /* prime    */ "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFC2F",
   /* A        */ "0000000000000000000000000000000000000000000000000000000000000000",
   /* B        */ "0000000000000000000000000000000000000000000000000000000000000007",
   /* order    */ "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEBAAEDCE6AF48A03BBFD25E8CD0364141",
   /* Gx       */ "79BE667EF9DCBBAC55A06295CE870B07029BFCDB2DCE28D959F2815B16F81798",
   /* Gy       */ "483ADA7726A3C4655DA4FBFC0E1108A8FD17B448A68554199C47D08FFB10D4B8",
   /* cofactor */ 1,
   /* OID      */ "1.3.132.0.10"
},
#endif
#ifdef LTC_ECC_SECP384R1
{
   /* prime    */ "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFF0000000000000000FFFFFFFF",
   /* A        */ "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFF0000000000000000FFFFFFFC",
   /* B        */ "B3312FA7E23EE7E4988E056BE3F82D19181D9C6EFE8141120314088F5013875AC656398D8A2ED19D2A85C8EDD3EC2AEF",
   /* order    */ "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFC7634D81F4372DDF581A0DB248B0A77AECEC196ACCC52973",
   /* Gx       */ "AA87CA22BE8B05378EB1C71EF320AD746E1D3B628BA79B9859F741E082542A385502F25DBF55296C3A545E3872760AB7",
   /* Gy       */ "3617DE4A96262C6F5D9E98BF9292DC29F8F41DBD289A147CE9DA3113B5F0B8C00A60B1CE1D7E819D7A431D7C90EA0E5F",
   /* cofactor */ 1,
   /* OID      */ "1.3.132.0.34"
},
#endif
#ifdef LTC_ECC_SECP521R1
{
   /* prime    */ "01FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
   /* A        */ "01FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFC",
   /* B        */ "0051953EB9618E1C9A1F929A21A0B68540EEA2DA725B99B315F3B8B489918EF109E156193951EC7E937B1652C0BD3BB1BF073573DF883D2C34F1EF451FD46B503F00",
   /* order    */ "01FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFA51868783BF2F966B7FCC0148F709A5D03BB5C9B8899C47AEBB6FB71E91386409",
   /* Gx       */ "00C6858E06B70404E9CD9E3ECB662395B4429C648139053FB521F828AF606B4D3DBAA14B5E77EFE75928FE1DC127A2FFA8DE3348B3C1856A429BF97E7E31C2E5BD66",
   /* Gy       */ "011839296A789A3BC0045C8A5FB42C7D1BD998F54449579B446817AFBD17273E662C97EE72995EF42640C550B9013FAD0761353C7086A272C24088BE94769FD16650",
   /* cofactor */ 1,
   /* OID      */ "1.3.132.0.35"
},
#endif
#ifdef LTC_ECC_PRIME239V1
{
   /* prime    */ "7FFFFFFFFFFFFFFFFFFFFFFF7FFFFFFFFFFF8000000000007FFFFFFFFFFF",
   /* A        */ "7FFFFFFFFFFFFFFFFFFFFFFF7FFFFFFFFFFF8000000000007FFFFFFFFFFC",
   /* B        */ "6B016C3BDCF18941D0D654921475CA71A9DB2FB27D1D37796185C2942C0A",
   /* order    */ "7FFFFFFFFFFFFFFFFFFFFFFF7FFFFF9E5E9A9F5D9071FBD1522688909D0B",
   /* Gx       */ "0FFA963CDCA8816CCC33B8642BEDF905C3D358573D3F27FBBD3B3CB9AAAF",
   /* Gy       */ "7DEBE8E4E90A5DAE6E4054CA530BA04654B36818CE226B39FCCB7B02F1AE",
   /* cofactor */ 1,
   /* OID      */ "1.2.840.10045.3.1.4"
},
#endif
#ifdef LTC_ECC_PRIME239V2
{
   /* prime    */ "7FFFFFFFFFFFFFFFFFFFFFFF7FFFFFFFFFFF8000000000007FFFFFFFFFFF",
   /* A        */ "7FFFFFFFFFFFFFFFFFFFFFFF7FFFFFFFFFFF8000000000007FFFFFFFFFFC",
   /* B        */ "617FAB6832576CBBFED50D99F0249C3FEE58B94BA0038C7AE84C8C832F2C",
   /* order    */ "7FFFFFFFFFFFFFFFFFFFFFFF800000CFA7E8594377D414C03821BC582063",
   /* Gx       */ "38AF09D98727705120C921BB5E9E26296A3CDCF2F35757A0EAFD87B830E7",
   /* Gy       */ "5B0125E4DBEA0EC7206DA0FC01D9B081329FB555DE6EF460237DFF8BE4BA",
   /* cofactor */ 1,
   /* OID      */ "1.2.840.10045.3.1.5"
},
#endif
#ifdef LTC_ECC_PRIME239V3
{
   /* prime    */ "7FFFFFFFFFFFFFFFFFFFFFFF7FFFFFFFFFFF8000000000007FFFFFFFFFFF",
   /* A        */ "7FFFFFFFFFFFFFFFFFFFFFFF7FFFFFFFFFFF8000000000007FFFFFFFFFFC",
   /* B        */ "255705FA2A306654B1F4CB03D6A750A30C250102D4988717D9BA15AB6D3E",
   /* order    */ "7FFFFFFFFFFFFFFFFFFFFFFF7FFFFF975DEB41B3A6057C3C432146526551",
   /* Gx       */ "6768AE8E18BB92CFCF005C949AA2C6D94853D0E660BBF854B1C9505FE95A",
   /* Gy       */ "1607E6898F390C06BC1D552BAD226F3B6FCFE48B6E818499AF18E3ED6CF3",
   /* cofactor */ 1,
   /* OID      */ "1.2.840.10045.3.1.6"
},
#endif
#ifdef LTC_ECC_BRAINPOOLP160R1
{
   /* prime    */ "E95E4A5F737059DC60DFC7AD95B3D8139515620F",
   /* A        */ "340E7BE2A280EB74E2BE61BADA745D97E8F7C300",
   /* B        */ "1E589A8595423412134FAA2DBDEC95C8D8675E58",
   /* order    */ "E95E4A5F737059DC60DF5991D45029409E60FC09",
   /* Gx       */ "BED5AF16EA3F6A4F62938C4631EB5AF7BDBCDBC3",
   /* Gy       */ "1667CB477A1A8EC338F94741669C976316DA6321",
   /* cofactor */ 1,
   /* OID      */ "1.3.36.3.3.2.8.1.1.1"
},
#endif
#ifdef LTC_ECC_BRAINPOOLP192R1
{
   /* prime    */ "C302F41D932A36CDA7A3463093D18DB78FCE476DE1A86297",
   /* A        */ "6A91174076B1E0E19C39C031FE8685C1CAE040E5C69A28EF",
   /* B        */ "469A28EF7C28CCA3DC721D044F4496BCCA7EF4146FBF25C9",
   /* order    */ "C302F41D932A36CDA7A3462F9E9E916B5BE8F1029AC4ACC1",
   /* Gx       */ "C0A0647EAAB6A48753B033C56CB0F0900A2F5C4853375FD6",
   /* Gy       */ "14B690866ABD5BB88B5F4828C1490002E6773FA2FA299B8F",
   /* cofactor */ 1,
   /* OID      */ "1.3.36.3.3.2.8.1.1.3"
},
#endif
#ifdef LTC_ECC_BRAINPOOLP224R1
{
   /* prime    */ "D7C134AA264366862A18302575D1D787B09F075797DA89F57EC8C0FF",
   /* A        */ "68A5E62CA9CE6C1C299803A6C1530B514E182AD8B0042A59CAD29F43",
   /* B        */ "2580F63CCFE44138870713B1A92369E33E2135D266DBB372386C400B",
   /* order    */ "D7C134AA264366862A18302575D0FB98D116BC4B6DDEBCA3A5A7939F",
   /* Gx       */ "0D9029AD2C7E5CF4340823B2A87DC68C9E4CE3174C1E6EFDEE12C07D",
   /* Gy       */ "58AA56F772C0726F24C6B89E4ECDAC24354B9E99CAA3F6D3761402CD",
   /* cofactor */ 1,
   /* OID      */ "1.3.36.3.3.2.8.1.1.5"
},
#endif
#ifdef LTC_ECC_BRAINPOOLP256R1
{
   /* prime    */ "A9FB57DBA1EEA9BC3E660A909D838D726E3BF623D52620282013481D1F6E5377",
   /* A        */ "7D5A0975FC2C3057EEF67530417AFFE7FB8055C126DC5C6CE94A4B44F330B5D9",
   /* B        */ "26DC5C6CE94A4B44F330B5D9BBD77CBF958416295CF7E1CE6BCCDC18FF8C07B6",
   /* order    */ "A9FB57DBA1EEA9BC3E660A909D838D718C397AA3B561A6F7901E0E82974856A7",
   /* Gx       */ "8BD2AEB9CB7E57CB2C4B482FFC81B7AFB9DE27E1E3BD23C23A4453BD9ACE3262",
   /* Gy       */ "547EF835C3DAC4FD97F8461A14611DC9C27745132DED8E545C1D54C72F046997",
   /* cofactor */ 1,
   /* OID      */ "1.3.36.3.3.2.8.1.1.7"
},
#endif
#ifdef LTC_ECC_BRAINPOOLP320R1
{
   /* prime    */ "D35E472036BC4FB7E13C785ED201E065F98FCFA6F6F40DEF4F92B9EC7893EC28FCD412B1F1B32E27",
   /* A        */ "3EE30B568FBAB0F883CCEBD46D3F3BB8A2A73513F5EB79DA66190EB085FFA9F492F375A97D860EB4",
   /* B        */ "520883949DFDBC42D3AD198640688A6FE13F41349554B49ACC31DCCD884539816F5EB4AC8FB1F1A6",
   /* order    */ "D35E472036BC4FB7E13C785ED201E065F98FCFA5B68F12A32D482EC7EE8658E98691555B44C59311",
   /* Gx       */ "43BD7E9AFB53D8B85289BCC48EE5BFE6F20137D10A087EB6E7871E2A10A599C710AF8D0D39E20611",
   /* Gy       */ "14FDD05545EC1CC8AB4093247F77275E0743FFED117182EAA9C77877AAAC6AC7D35245D1692E8EE1",
   /* cofactor */ 1,
   /* OID      */ "1.3.36.3.3.2.8.1.1.9"
},
#endif
#ifdef LTC_ECC_BRAINPOOLP384R1
{
   /* prime    */ "8CB91E82A3386D280F5D6F7E50E641DF152F7109ED5456B412B1DA197FB71123ACD3A729901D1A71874700133107EC53",
   /* A        */ "7BC382C63D8C150C3C72080ACE05AFA0C2BEA28E4FB22787139165EFBA91F90F8AA5814A503AD4EB04A8C7DD22CE2826",
   /* B        */ "04A8C7DD22CE28268B39B55416F0447C2FB77DE107DCD2A62E880EA53EEB62D57CB4390295DBC9943AB78696FA504C11",
   /* order    */ "8CB91E82A3386D280F5D6F7E50E641DF152F7109ED5456B31F166E6CAC0425A7CF3AB6AF6B7FC3103B883202E9046565",
   /* Gx       */ "1D1C64F068CF45FFA2A63A81B7C13F6B8847A3E77EF14FE3DB7FCAFE0CBD10E8E826E03436D646AAEF87B2E247D4AF1E",
   /* Gy       */ "8ABE1D7520F9C2A45CB1EB8E95CFD55262B70B29FEEC5864E19C054FF99129280E4646217791811142820341263C5315",
   /* cofactor */ 1,
   /* OID      */ "1.3.36.3.3.2.8.1.1.11"
},
#endif
#ifdef LTC_ECC_BRAINPOOLP512R1
{
   /* prime    */ "AADD9DB8DBE9C48B3FD4E6AE33C9FC07CB308DB3B3C9D20ED6639CCA703308717D4D9B009BC66842AECDA12AE6A380E62881FF2F2D82C68528AA6056583A48F3",
   /* A        */ "7830A3318B603B89E2327145AC234CC594CBDD8D3DF91610A83441CAEA9863BC2DED5D5AA8253AA10A2EF1C98B9AC8B57F1117A72BF2C7B9E7C1AC4D77FC94CA",
   /* B        */ "3DF91610A83441CAEA9863BC2DED5D5AA8253AA10A2EF1C98B9AC8B57F1117A72BF2C7B9E7C1AC4D77FC94CADC083E67984050B75EBAE5DD2809BD638016F723",
   /* order    */ "AADD9DB8DBE9C48B3FD4E6AE33C9FC07CB308DB3B3C9D20ED6639CCA70330870553E5C414CA92619418661197FAC10471DB1D381085DDADDB58796829CA90069",
   /* Gx       */ "81AEE4BDD82ED9645A21322E9C4C6A9385ED9F70B5D916C1B43B62EEF4D0098EFF3B1F78E2D0D48D50D1687B93B97D5F7C6D5047406A5E688B352209BCB9F822",
   /* Gy       */ "7DDE385D566332ECC0EABFA9CF7822FDF209F70024A57B1AA000C55B881F8111B2DCDE494A5F485E5BCA4BD88A2763AED1CA2B2FA8F0540678CD1E0F3AD80892",
   /* cofactor */ 1,
   /* OID      */ "1.3.36.3.3.2.8.1.1.13"
},
#endif
#ifdef LTC_ECC_BRAINPOOLP160T1
{
   /* prime    */ "E95E4A5F737059DC60DFC7AD95B3D8139515620F",
   /* A        */ "E95E4A5F737059DC60DFC7AD95B3D8139515620C",
   /* B        */ "7A556B6DAE535B7B51ED2C4D7DAA7A0B5C55F380",
   /* order    */ "E95E4A5F737059DC60DF5991D45029409E60FC09",
   /* Gx       */ "B199B13B9B34EFC1397E64BAEB05ACC265FF2378",
   /* Gy       */ "ADD6718B7C7C1961F0991B842443772152C9E0AD",
   /* cofactor */ 1,
   /* OID      */ "1.3.36.3.3.2.8.1.1.2"
},
#endif
#ifdef LTC_ECC_BRAINPOOLP192T1
{
   /* prime    */ "C302F41D932A36CDA7A3463093D18DB78FCE476DE1A86297",
   /* A        */ "C302F41D932A36CDA7A3463093D18DB78FCE476DE1A86294",
   /* B        */ "13D56FFAEC78681E68F9DEB43B35BEC2FB68542E27897B79",
   /* order    */ "C302F41D932A36CDA7A3462F9E9E916B5BE8F1029AC4ACC1",
   /* Gx       */ "3AE9E58C82F63C30282E1FE7BBF43FA72C446AF6F4618129",
   /* Gy       */ "097E2C5667C2223A902AB5CA449D0084B7E5B3DE7CCC01C9",
   /* cofactor */ 1,
   /* OID      */ "1.3.36.3.3.2.8.1.1.4"
},
#endif
#ifdef LTC_ECC_BRAINPOOLP224T1
{
   /* prime    */ "D7C134AA264366862A18302575D1D787B09F075797DA89F57EC8C0FF",
   /* A        */ "D7C134AA264366862A18302575D1D787B09F075797DA89F57EC8C0FC",
   /* B        */ "4B337D934104CD7BEF271BF60CED1ED20DA14C08B3BB64F18A60888D",
   /* order    */ "D7C134AA264366862A18302575D0FB98D116BC4B6DDEBCA3A5A7939F",
   /* Gx       */ "6AB1E344CE25FF3896424E7FFE14762ECB49F8928AC0C76029B4D580",
   /* Gy       */ "0374E9F5143E568CD23F3F4D7C0D4B1E41C8CC0D1C6ABD5F1A46DB4C",
   /* cofactor */ 1,
   /* OID      */ "1.3.36.3.3.2.8.1.1.6"
},
#endif
#ifdef LTC_ECC_BRAINPOOLP256T1
{
   /* prime    */ "A9FB57DBA1EEA9BC3E660A909D838D726E3BF623D52620282013481D1F6E5377",
   /* A        */ "A9FB57DBA1EEA9BC3E660A909D838D726E3BF623D52620282013481D1F6E5374",
   /* B        */ "662C61C430D84EA4FE66A7733D0B76B7BF93EBC4AF2F49256AE58101FEE92B04",
   /* order    */ "A9FB57DBA1EEA9BC3E660A909D838D718C397AA3B561A6F7901E0E82974856A7",
   /* Gx       */ "A3E8EB3CC1CFE7B7732213B23A656149AFA142C47AAFBC2B79A191562E1305F4",
   /* Gy       */ "2D996C823439C56D7F7B22E14644417E69BCB6DE39D027001DABE8F35B25C9BE",
   /* cofactor */ 1,
   /* OID      */ "1.3.36.3.3.2.8.1.1.8"
},
#endif
#ifdef LTC_ECC_BRAINPOOLP320T1
{
   /* prime    */ "D35E472036BC4FB7E13C785ED201E065F98FCFA6F6F40DEF4F92B9EC7893EC28FCD412B1F1B32E27",
   /* A        */ "D35E472036BC4FB7E13C785ED201E065F98FCFA6F6F40DEF4F92B9EC7893EC28FCD412B1F1B32E24",
   /* B        */ "A7F561E038EB1ED560B3D147DB782013064C19F27ED27C6780AAF77FB8A547CEB5B4FEF422340353",
   /* order    */ "D35E472036BC4FB7E13C785ED201E065F98FCFA5B68F12A32D482EC7EE8658E98691555B44C59311",
   /* Gx       */ "925BE9FB01AFC6FB4D3E7D4990010F813408AB106C4F09CB7EE07868CC136FFF3357F624A21BED52",
   /* Gy       */ "63BA3A7A27483EBF6671DBEF7ABB30EBEE084E58A0B077AD42A5A0989D1EE71B1B9BC0455FB0D2C3",
   /* cofactor */ 1,
   /* OID      */ "1.3.36.3.3.2.8.1.1.10"
},
#endif
#ifdef LTC_ECC_BRAINPOOLP384T1
{
   /* prime    */ "8CB91E82A3386D280F5D6F7E50E641DF152F7109ED5456B412B1DA197FB71123ACD3A729901D1A71874700133107EC53",
   /* A        */ "8CB91E82A3386D280F5D6F7E50E641DF152F7109ED5456B412B1DA197FB71123ACD3A729901D1A71874700133107EC50",
   /* B        */ "7F519EADA7BDA81BD826DBA647910F8C4B9346ED8CCDC64E4B1ABD11756DCE1D2074AA263B88805CED70355A33B471EE",
   /* order    */ "8CB91E82A3386D280F5D6F7E50E641DF152F7109ED5456B31F166E6CAC0425A7CF3AB6AF6B7FC3103B883202E9046565",
   /* Gx       */ "18DE98B02DB9A306F2AFCD7235F72A819B80AB12EBD653172476FECD462AABFFC4FF191B946A5F54D8D0AA2F418808CC",
   /* Gy       */ "25AB056962D30651A114AFD2755AD336747F93475B7A1FCA3B88F2B6A208CCFE469408584DC2B2912675BF5B9E582928",
   /* cofactor */ 1,
   /* OID      */ "1.3.36.3.3.2.8.1.1.12"
},
#endif
#ifdef LTC_ECC_BRAINPOOLP512T1
{
   /* prime    */ "AADD9DB8DBE9C48B3FD4E6AE33C9FC07CB308DB3B3C9D20ED6639CCA703308717D4D9B009BC66842AECDA12AE6A380E62881FF2F2D82C68528AA6056583A48F3",
   /* A        */ "AADD9DB8DBE9C48B3FD4E6AE33C9FC07CB308DB3B3C9D20ED6639CCA703308717D4D9B009BC66842AECDA12AE6A380E62881FF2F2D82C68528AA6056583A48F0",
   /* B        */ "7CBBBCF9441CFAB76E1890E46884EAE321F70C0BCB4981527897504BEC3E36A62BCDFA2304976540F6450085F2DAE145C22553B465763689180EA2571867423E",
   /* order    */ "AADD9DB8DBE9C48B3FD4E6AE33C9FC07CB308DB3B3C9D20ED6639CCA70330870553E5C414CA92619418661197FAC10471DB1D381085DDADDB58796829CA90069",
   /* Gx       */ "640ECE5C12788717B9C1BA06CBC2A6FEBA85842458C56DDE9DB1758D39C0313D82BA51735CDB3EA499AA77A7D6943A64F7A3F25FE26F06B51BAA2696FA9035DA",
   /* Gy       */ "5B534BD595F5AF0FA2C892376C84ACE1BB4E3019B71634C01131159CAE03CEE9D9932184BEEF216BD71DF2DADF86A627306ECFF96DBB8BACE198B61E00F8B332",
   /* cofactor */ 1,
   /* OID      */ "1.3.36.3.3.2.8.1.1.14"
},
#endif
{
   NULL, NULL, NULL, NULL, NULL, NULL, 0, NULL
}
};

#endif

