package Dancer::Plugin::Email;
{
  $Dancer::Plugin::Email::VERSION = '0.1300_01';
}

use Dancer ':syntax';
use Dancer::Plugin;
use Email::Simple;
use Email::Sender::Simple 'sendmail';
use Module::Load 'load';
use Scalar::Util 'blessed';
use Try::Tiny;

register email => sub {
    my $params = shift || {};
    my $conf = plugin_setting;
    my $conf_headers = $conf->{headers} || {};
    my %headers = ( %$conf_headers, %$params );
    delete $headers{$_} for qw(body message async);

    my $email = Email::Simple->create(
        header => [ %headers ],
        body => $params->{body} || $params->{message},
    );
    my $transport;
    my $conf_transport = $conf->{transport} || {};
    if (my ($transport_name) = keys %$conf_transport) {
        my $transport_params = $conf_transport->{$transport_name} || {};
        my $transport_class = "Email::Sender::Transport::$transport_name";
        load $transport_class;
        $transport = $transport_class->new($transport_params);
    }
    return try {
        sendmail $email, { transport => $transport };
    } catch {
        my $err = $_;
        $err = $err->message if blessed($err) and $err->can('message');
        die $err;
    };
};


register_plugin;

# ABSTRACT: Simple email sending for Dancer applications


1;

__END__
=pod

=head1 NAME

Dancer::Plugin::Email - Simple email sending for Dancer applications

=head1 VERSION

version 0.1300_01

=head1 SYNOPSIS

    use Dancer;
    use Dancer::Plugin::Email;
    
    post '/contact' => sub {
        email {
            from    => 'bob@foo.com',
            to      => 'sue@foo.com',
            subject => 'allo',
            body    => 'Dear Sue, ...',
        };
    };

=head1 DESCRIPTION

This plugin tries to make sending emails from L<Dancer> applications as simple
as possible.
It uses L<Email::Sender> under the hood.
In a lot of cases, no configuration is required.
For example, if your app is hosted on a unix-like server with sendmail
installed, calling C<email()> will just do the right thing.

IMPORTANT: Version 1.x of this module is not backwards compatible with the
0.x versions.
This module was originally built on Email::Stuff which was built on
Email::Send which has been deprecated in favor of Email::Sender.
Versions 1.x and on have be refactored to use Email::Sender.
I have tried to keep the interface the same as much as possible.
The main difference is the configuration.
If there are features missing that you were using in older versions,
then please let me know by creating an issue on 
L<github|https://github.com/ironcamel/Dancer-Plugin-Email>.

=head1 FUNCTIONS

This module by default exports the single function C<email>.

=head2 email

This function sends an email.
It takes a single argument, a hashref of parameters.
Default values for the parameters may be provided in the headers section of
the L</CONFIGURATION>.
Paramaters provided to this function will override the corresponding
configuration values if there is any overlap.
An exception is thrown if sending the email fails,
so wrapping calls to C<email> with try/catch is recommended.

    use Dancer;
    use Dancer::Plugin::Email;
    use Try::Tiny;

    post '/contact' => sub {
        try {
            email {
                from    => 'bob@foo.com',
                to      => 'sue@foo.com',
                subject => 'allo',
                body    => 'Dear Sue, ...',
                # Optional extra headers
                headers => {
                    "X-Mailer"          => 'This fine Dancer application',
                    "X-Accept-Language" => 'en',
                }
            };
        } catch {
            error "Could not send email: $_";
        };
    };

=head1 CONFIGURATION

No configuration is necessarily required.
L<Email::Sender::Simple> tries to make a good guess about how to send the
message.
It will usually try to use the sendmail program on unix-like systems
and SMTP on Windows.
However, you may explicitly configure a transport in your configuration.
Only one transport may be configured.
For documentation for the parameters of the transport, see the corresponding
Email::Sender::Transport::* module.
For example, the parameters available for the SMTP transport are documented
here L<Email::Sender::Transport::SMTP/ATTRIBUTES>.

You may also provide default headers in the configuration:

    plugins:
      Email:
        # Set default headers (OPTIONAL)
        headers:
          from: 'bob@foo.com'
          subject: 'default subject'
          X-Mailer: 'MyDancer 1.0'
          X-Accept-Language: 'en'
        # Explicity set a transport (OPTIONAL)
        transport:
          Sendmail:
            sendmail: '/usr/sbin/sendmail'

Example configuration for sending mail via Gmail:

    plugins:
      Email:
        transport:
          SMTP:
            ssl: 1
            host: 'smtp.gmail.com'
            port: 465
            sasl_username: 'bob@gmail.com'
            sasl_password: 'secret'

Use the Sendmail transport using the sendmail program in the system path:

    plugins:
      Email:
        transport:
          Sendmail:

Use the Sendmail transport with an explicit path to the sendmail program:

    plugins:
      Email:
        transport:
          Sendmail:
            sendmail: '/usr/sbin/sendmail'

=head1 SEE ALSO

=over

=item L<Email::Sender>

=back

=head1 AUTHORS

=over 4

=item *

Al Newkirk <awncorp@cpan.org>

=item *

Naveed Massjouni <naveedm9@gmail.com>

=back

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2010 by awncorp.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut

