/* riemann/client.h -- Riemann C client library
 * SPDX-FileCopyrightText: 2013-2024 Gergely Nagy
 * SPDX-FileContributor: Gergely Nagy
 *
 * SPDX-License-Identifier: EUPL-1.2
 */

#pragma once

#include <riemann/message.h>
#include <sys/time.h>

typedef enum
  {
    RIEMANN_CLIENT_NONE,
    RIEMANN_CLIENT_TCP,
    RIEMANN_CLIENT_UDP,
    RIEMANN_CLIENT_TLS
  } riemann_client_type_t;

typedef enum
  {
    RIEMANN_CLIENT_OPTION_NONE,
    RIEMANN_CLIENT_OPTION_TLS_CA_FILE,
    RIEMANN_CLIENT_OPTION_TLS_CERT_FILE,
    RIEMANN_CLIENT_OPTION_TLS_KEY_FILE,
    RIEMANN_CLIENT_OPTION_TLS_HANDSHAKE_TIMEOUT,
    RIEMANN_CLIENT_OPTION_TLS_PRIORITIES,
  } riemann_client_option_t;

typedef enum
  {
   RIEMANN_CLIENT_TLS_LIBRARY_NONE,
   RIEMANN_CLIENT_TLS_LIBRARY_GNUTLS,
   RIEMANN_CLIENT_TLS_LIBRARY_WOLFSSL,
   RIEMANN_CLIENT_TLS_LIBRARY_OPENSSL
  } riemann_client_tls_library_t;

typedef struct _riemann_client_t riemann_client_t;

#ifdef __cplusplus
extern "C" {
#endif

const char *riemann_client_version (void);
const char *riemann_client_version_string (void);
int riemann_client_check_version (int major, int minor, int patch);
riemann_client_tls_library_t riemann_client_get_tls_library (void);

riemann_client_t *riemann_client_new (void);
riemann_client_t *riemann_client_create (riemann_client_type_t type,
                                         const char *hostname, int port,
                                         ...);
void riemann_client_free (riemann_client_t *client);

int riemann_client_get_fd (riemann_client_t *client);
int riemann_client_set_timeout (riemann_client_t *client,
                                struct timeval *timeout);

int riemann_client_connect (riemann_client_t *client, riemann_client_type_t type,
                            const char *hostname, int port, ...);
int riemann_client_disconnect (riemann_client_t *client);

int riemann_client_send_message (riemann_client_t *client,
                                 riemann_message_t *message);
int riemann_client_send_message_oneshot (riemann_client_t *client,
                                         riemann_message_t *message);
riemann_message_t *riemann_client_recv_message (riemann_client_t *client);

#ifdef __cplusplus
}
#endif
