# frozen_string_literal: true

# Licensed to Elasticsearch B.V. under one or more contributor
# license agreements. See the NOTICE file distributed with
# this work for additional information regarding copyright
# ownership. Elasticsearch B.V. licenses this file to you under
# the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.

module Elastic
  module EnterpriseSearch
    module AppSearch
      module Actions
        # Curations - Update a curation
        # Updates an existing curation
        #
        # @param [String] engine_name Name of the engine (*Required*)
        # @param [Hash] arguments endpoint arguments
        # @option arguments [String] :curation_id Curation ID (*Required*)
        # @option arguments [Hash] :body (Required: queries)
        # @option body [string] :id
        # @option body [Array<string>] :queries List of affected search queries
        # @option body [Array<string>] :promoted List of promoted document IDs
        # @option body [Array<string>] :hidden List of hidden document IDs
        # @option body :suggestion
        # @option arguments [Hash] :headers optional HTTP headers to send with the request
        #
        # @see https://www.elastic.co/guide/en/app-search/current/curations.html#curations-update
        #
        def put_curation(engine_name, arguments = {})
          raise ArgumentError, "Required parameter 'engine_name' missing" unless engine_name
          raise ArgumentError, "Required parameter 'curation_id' missing" unless arguments[:curation_id]
          raise ArgumentError, "Required parameter 'body (queries)' missing" unless arguments[:body]

          curation_id = arguments.delete(:curation_id)
          body = arguments.delete(:body) || {}
          headers = arguments.delete(:headers) || {}
          request(
            :put,
            "api/as/v1/engines/#{engine_name}/curations/#{curation_id}/",
            arguments,
            body,
            headers
          )
        end
      end
    end
  end
end
