"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MetricsService = void 0;

var _ = require(".");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class MetricsService {
  constructor(logger) {
    this.logger = logger;

    _defineProperty(this, "interval", _.METRIC_INTERVAL.ONE_MINUTE);

    _defineProperty(this, "windowSize", _.DEFAULT_WINDOW_SIZE);

    _defineProperty(this, "data", {});

    _defineProperty(this, "componentCounts", {});

    _defineProperty(this, "statusCodeCounts", {});

    _defineProperty(this, "overall", {});
  }

  setup(interval = _.METRIC_INTERVAL.ONE_MINUTE, windowSize = _.DEFAULT_WINDOW_SIZE) {
    this.interval = interval;
    this.windowSize = windowSize;

    const addMetric = (componentName, action, statusCode, value) => {
      const currInterval = Math.floor(Date.now() / this.interval);
      this.trim();

      if (!this.data[currInterval]) {
        this.data[currInterval] = {};
        this.overall[currInterval] = {
          response_time_total: 0,
          count: 0
        };
        this.componentCounts[currInterval] = {};
        this.statusCodeCounts[currInterval] = {};
      }

      if (!this.data[currInterval][componentName]) {
        this.data[currInterval][componentName] = {};
        this.componentCounts[currInterval][componentName] = 0;
      }

      if (!this.statusCodeCounts[currInterval][statusCode]) {
        this.statusCodeCounts[currInterval][statusCode] = 0;
      }

      if (!this.data[currInterval][componentName][action]) {
        this.data[currInterval][componentName][action] = {};
      }

      if (!this.data[currInterval][componentName][action][statusCode]) {
        this.data[currInterval][componentName][action][statusCode] = {
          response_time_total: 0,
          count: 0
        };
      }

      const {
        response_time_total,
        count
      } = this.data[currInterval][componentName][action][statusCode];
      this.data[currInterval][componentName][action][statusCode] = {
        response_time_total: response_time_total + value,
        count: count + 1
      };
      this.componentCounts[currInterval][componentName]++;
      this.statusCodeCounts[currInterval][statusCode]++;
      this.overall[currInterval].response_time_total += value;
      this.overall[currInterval].count++;
    };

    const getStats = () => {
      const prevInterval = Math.floor(Date.now() / this.interval) - 1;
      const data = { ...this.data[prevInterval]
      } || {};
      const overall = { ...this.overall[prevInterval]
      } || {};
      let requestsPerSecond = 0,
          responseTimeAvg = 0;

      if (Object.keys(overall).length !== 0 && overall.count !== 0) {
        responseTimeAvg = overall.response_time_total / overall.count;
        requestsPerSecond = overall.count / (this.interval / 1000);
      }

      return {
        data,
        overall: {
          response_time_avg: responseTimeAvg,
          requests_per_second: requestsPerSecond
        },
        counts_by_component: { ...this.componentCounts[prevInterval]
        } || {},
        counts_by_status_code: { ...this.statusCodeCounts[prevInterval]
        } || {}
      };
    };

    return {
      addMetric,
      getStats
    };
  }

  start() {}

  stop() {
    this.resetMetrics();
  }

  resetMetrics() {
    this.data = {};
    this.overall = {};
    this.componentCounts = {};
    this.statusCodeCounts = {};
  }

  trim() {
    const oldestTimestampToKeep = Math.floor((Date.now() - this.windowSize * this.interval) / this.interval);

    for (const timestampStr in this.data) {
      const timestamp = parseInt(timestampStr);

      if (timestamp < oldestTimestampToKeep) {
        delete this.data[timestamp];
      }
    }
  }

}

exports.MetricsService = MetricsService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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