<?php

declare(strict_types=1);

/*
 * Copyright (c) 2017-2023 François Kooman <fkooman@tuxed.net>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

require_once 'vendor/autoload.php';

use fkooman\OAuth\Server\Base64UrlSafe;

/**
 * Decode a token as used by this OAuth Server.
 *
 * @see TOKEN_FORMAT.md
 */
try {
    $jsonFormatPayload = false;
    $tokenToParse = null;
    foreach ($argv as $arg) {
        if ('--format' === $arg) {
            $jsonFormatPayload = true;
            continue;
        }

        $tokenToParse = $arg;
    }

    if (null === $tokenToParse) {
        throw new Exception('no token provided');
    }

    $explodedToken = explode('.', $tokenToParse, 4);
    if (4 !== count($explodedToken)) {
        throw new Exception('invalid token');
    }

    [$tokenVersion, $keyId, $tokenPayload,] = explode('.', $tokenToParse);

    echo 'Version'.PHP_EOL."\t".$tokenVersion.PHP_EOL;
    echo 'Key ID'.PHP_EOL."\t".$keyId.PHP_EOL;

    $tokenPayload = Base64UrlSafe::decode($tokenPayload);
    if ($jsonFormatPayload) {
        $tokenPayload = PHP_EOL.json_encode(
            json_decode($tokenPayload, true, JSON_THROW_ON_ERROR),
            JSON_PRETTY_PRINT|JSON_THROW_ON_ERROR
        );
    }

    echo 'Payload'.PHP_EOL."\t".$tokenPayload.PHP_EOL;
} catch(Exception $e) {
    echo 'ERROR: '.$e->getMessage().PHP_EOL;
    exit(1);
}
