"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _constants = require("../utils/constants");

var _helpers = require("../utils/helpers");

var _DataStreamService = require("./DataStreamService");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class IndexService {
  constructor(osDriver) {
    _defineProperty(this, "osDriver", void 0);

    _defineProperty(this, "getIndices", async (context, request, response) => {
      try {
        // @ts-ignore
        const {
          from,
          size,
          sortField,
          sortDirection,
          terms,
          indices,
          dataStreams,
          showDataStreams,
          expandWildcards
        } = request.query;
        const params = {
          index: (0, _helpers.getSearchString)(terms, indices, dataStreams),
          format: "json",
          s: `${sortField}:${sortDirection}`
        };

        if (expandWildcards) {
          params.expand_wildcards = expandWildcards;
        }

        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const [recoverys, tasks, indicesResponse, indexToDataStreamMapping] = await Promise.all([callWithRequest("cat.recovery", {
          format: "json",
          detailed: true
        }), callWithRequest("cat.tasks", {
          format: "json",
          detailed: true,
          actions: "indices:data/write/reindex"
        }), callWithRequest("cat.indices", params), (0, _DataStreamService.getIndexToDataStreamMapping)({
          callAsCurrentUser: callWithRequest
        })]);
        const formattedTasks = tasks.map(item => {
          const {
            description
          } = item;
          const regexp = /reindex from \[([^\]]+)\] to \[([^\]]+)\]/i;
          const matchResult = description.match(regexp);

          if (matchResult) {
            const [, fromIndex, toIndex] = matchResult;
            return { ...item,
              fromIndex,
              toIndex
            };
          } else {
            return { ...item,
              fromIndex: "",
              toIndex: ""
            };
          }
        });
        const onGoingRecovery = recoverys.filter(item => item.stage !== "done"); // Augment the indices with their parent data stream name.

        indicesResponse.forEach(index => {
          index.data_stream = indexToDataStreamMapping[index.index] || null;
          let extraStatus = index.status;

          if (index.health === "green") {
            if (formattedTasks.find(item => item.toIndex === index.index)) {
              extraStatus = "reindex";
            }
          } else {
            if (onGoingRecovery.find(item => item.index === index.index)) {
              extraStatus = "recovery";
            }
          }

          if (extraStatus) {
            index.extraStatus = extraStatus;
          }
        });

        if (sortField === "status") {
          // add new more status to status field so we need to sort
          indicesResponse.sort((a, b) => {
            let flag;
            const aStatus = a.extraStatus;
            const bStatus = b.extraStatus;

            if (sortDirection === "asc") {
              flag = aStatus < bStatus;
            } else {
              flag = aStatus > bStatus;
            }

            return flag ? -1 : 1;
          });
        } // Filtering out indices that belong to a data stream. This must be done before pagination.


        const filteredIndices = showDataStreams ? indicesResponse : indicesResponse.filter(index => index.data_stream === null); // _cat doesn't support pagination, do our own in server pagination to at least reduce network bandwidth

        const fromNumber = parseInt(from, 10);
        const sizeNumber = parseInt(size, 10);
        const paginatedIndices = filteredIndices.slice(fromNumber, fromNumber + sizeNumber);
        const indexNames = paginatedIndices.map(value => value.index);
        const managedStatus = await this._getManagedStatus(request, indexNames); // NOTE: Cannot use response.ok due to typescript type checking

        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              indices: paginatedIndices.map(catIndex => ({ ...catIndex,
                managed: managedStatus[catIndex.index] ? "Yes" : "No",
                managedPolicy: managedStatus[catIndex.index]
              })),
              totalIndices: filteredIndices.length
            }
          }
        });
      } catch (err) {
        // Throws an error if there is no index matching pattern
        if (err.statusCode === 404 && err.body.error.type === "index_not_found_exception") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: {
                indices: [],
                totalIndices: 0
              }
            }
          });
        }

        console.error("Index Management - IndexService - getIndices:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "_getManagedStatus", async (request, indexNames) => {
      try {
        const explainParamas = {
          index: indexNames.toString()
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const explainResponse = await callWithRequest("ism.explain", explainParamas);
        const managed = {};

        for (const indexName in explainResponse) {
          if (indexName === "total_managed_indices") continue;
          const explain = explainResponse[indexName];
          managed[indexName] = explain["index.plugins.index_state_management.policy_id"] === null ? "" : explain["index.plugins.index_state_management.policy_id"];
        }

        return managed;
      } catch (err) {
        // otherwise it could be an unauthorized access error to config index or some other error
        // in which case we will return managed status N/A
        console.error("Index Management - IndexService - _getManagedStatus:", err);
        return indexNames.reduce((accu, value) => ({ ...accu,
          [value]: "N/A"
        }), {});
      }
    });

    _defineProperty(this, "applyPolicy", async (context, request, response) => {
      try {
        const {
          indices,
          policyId
        } = request.body;
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const params = {
          index: indices.join(","),
          body: {
            policy_id: policyId
          }
        };
        const addResponse = await callWithRequest("ism.add", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              failures: addResponse.failures,
              updatedIndices: addResponse.updated_indices,
              failedIndices: addResponse.failed_indices.map(failedIndex => ({
                indexName: failedIndex.index_name,
                indexUuid: failedIndex.index_uuid,
                reason: failedIndex.reason
              }))
            }
          }
        });
      } catch (err) {
        console.error("Index Management - IndexService - applyPolicy:", err); // return { ok: false, error: err.message };

        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "editRolloverAlias", async (context, request, response) => {
      try {
        const {
          alias,
          index
        } = request.body;
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const params = {
          index,
          body: {
            [_constants.Setting.RolloverAlias]: alias
          }
        };
        const rollOverResponse = await callWithRequest("indices.putSettings", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: rollOverResponse
          }
        });
      } catch (err) {
        console.error("Index Management - IndexService - editRolloverAlias", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    this.osDriver = osDriver;
  }

}

exports.default = IndexService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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