package google

import (
	"fmt"
	"log"
	"time"

	"github.com/hashicorp/terraform-plugin-sdk/v2/helper/schema"
)

func resourceProjectServiceIdentity() *schema.Resource {
	return &schema.Resource{
		Create: resourceProjectServiceIdentityCreate,
		Read:   resourceProjectServiceIdentityRead,
		Delete: resourceProjectServiceIdentityDelete,

		Timeouts: &schema.ResourceTimeout{
			Create: schema.DefaultTimeout(20 * time.Minute),
			Read:   schema.DefaultTimeout(10 * time.Minute),
			Delete: schema.DefaultTimeout(20 * time.Minute),
		},

		Schema: map[string]*schema.Schema{
			"service": {
				Type:     schema.TypeString,
				Required: true,
				ForceNew: true,
			},
			"project": {
				Type:     schema.TypeString,
				Optional: true,
				Computed: true,
				ForceNew: true,
			},
			"email": {
				Type:     schema.TypeString,
				Optional: true,
				Computed: true,
			},
		},
		UseJSONNumber: true,
	}
}

func resourceProjectServiceIdentityCreate(d *schema.ResourceData, meta interface{}) error {
	config := meta.(*Config)
	userAgent, err := generateUserAgentString(d, config.userAgent)
	if err != nil {
		return err
	}

	url, err := replaceVars(d, config, "{{ServiceUsageBasePath}}projects/{{project}}/services/{{service}}:generateServiceIdentity")
	if err != nil {
		return err
	}

	project, err := getProject(d, config)
	if err != nil {
		return err
	}

	billingProject := project

	// err == nil indicates that the billing_project value was found
	if bp, err := getBillingProject(d, config); err == nil {
		billingProject = bp
	}

	res, err := sendRequestWithTimeout(config, "POST", billingProject, url, userAgent, nil, d.Timeout(schema.TimeoutCreate))
	if err != nil {
		return fmt.Errorf("Error creating Service Identity: %s", err)
	}

	var opRes map[string]interface{}
	err = serviceUsageOperationWaitTimeWithResponse(
		config, res, &opRes, billingProject, "Creating Service Identity", userAgent,
		d.Timeout(schema.TimeoutCreate))
	if err != nil {
		return err
	}

	log.Printf("[DEBUG] Finished creating Service Identity %q: %#v", d.Id(), res)

	id, err := replaceVars(d, config, "projects/{{project}}/services/{{service}}")
	if err != nil {
		return fmt.Errorf("Error constructing id: %s", err)
	}
	d.SetId(id)

	// This API may not return the service identity's details, even if the relevant
	// Google API is configured for service identities.
	if emailVal, ok := opRes["email"]; ok {
		email, ok := emailVal.(string)
		if !ok {
			return fmt.Errorf("unexpected type for email: got %T, want string", email)
		}
		if err := d.Set("email", email); err != nil {
			return fmt.Errorf("Error setting email: %s", err)
		}
	}
	return nil
}

// There is no read endpoint for this API.
func resourceProjectServiceIdentityRead(d *schema.ResourceData, meta interface{}) error {
	return nil
}

// There is no delete endpoint for this API.
func resourceProjectServiceIdentityDelete(d *schema.ResourceData, meta interface{}) error {
	return nil
}
