"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class SnapshotManagementService {
  constructor(osDriver) {
    _defineProperty(this, "osDriver", void 0);

    _defineProperty(this, "getAllSnapshotsWithPolicy", async (context, request, response) => {
      try {
        var _getRepositoryRes$pay;

        // if no repository input, we need to first get back all repositories
        const getRepositoryRes = await this.catRepositories(context, request, response);
        let repositories;

        if ((_getRepositoryRes$pay = getRepositoryRes.payload) !== null && _getRepositoryRes$pay !== void 0 && _getRepositoryRes$pay.ok) {
          var _getRepositoryRes$pay2;

          repositories = (_getRepositoryRes$pay2 = getRepositoryRes.payload) === null || _getRepositoryRes$pay2 === void 0 ? void 0 : _getRepositoryRes$pay2.response.map(repo => repo.id);
        } else {
          var _getRepositoryRes$pay3;

          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: (_getRepositoryRes$pay3 = getRepositoryRes.payload) === null || _getRepositoryRes$pay3 === void 0 ? void 0 : _getRepositoryRes$pay3.error
            }
          });
        }

        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        let snapshots = [];

        for (let i = 0; i < repositories.length; i++) {
          const res = await callWithRequest("snapshot.get", {
            repository: repositories[i],
            snapshot: "_all",
            ignore_unavailable: true
          });
          const snapshotWithPolicy = res.snapshots.map(s => {
            var _s$metadata;

            return {
              id: s.snapshot,
              status: s.state,
              start_epoch: s.start_time_in_millis,
              end_epoch: s.end_time_in_millis,
              duration: s.duration_in_millis,
              indices: s.indices.length,
              successful_shards: s.shards.successful,
              failed_shards: s.shards.failed,
              total_shards: s.shards.total,
              repository: repositories[i],
              policy: (_s$metadata = s.metadata) === null || _s$metadata === void 0 ? void 0 : _s$metadata.sm_policy
            };
          });
          snapshots = [...snapshots, ...snapshotWithPolicy];
        }

        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              snapshots: snapshots,
              totalSnapshots: snapshots.length
            }
          }
        });
      } catch (err) {
        // If getting a non-existing snapshot, need to handle the missing snapshot exception, and return empty
        return this.errorResponse(response, err, "getAllSnapshotsWithPolicy");
      }
    });

    _defineProperty(this, "getSnapshot", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const {
          repository
        } = request.query;
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const res = await callWithRequest("snapshot.get", {
          repository: repository,
          snapshot: `${id}`,
          ignore_unavailable: true
        });
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: res.snapshots[0]
          }
        });
      } catch (err) {
        return this.errorResponse(response, err, "getSnapshot");
      }
    });

    _defineProperty(this, "deleteSnapshot", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const {
          repository
        } = request.query;
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const resp = await callWithRequest("snapshot.delete", {
          repository: repository,
          snapshot: `${id}`
        });
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: resp
          }
        });
      } catch (err) {
        return this.errorResponse(response, err, "deleteSnapshot");
      }
    });

    _defineProperty(this, "createSnapshot", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const {
          repository
        } = request.query;
        const params = {
          repository: repository,
          snapshot: id,
          body: JSON.stringify(request.body)
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const resp = await callWithRequest("snapshot.create", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: resp
          }
        });
      } catch (err) {
        return this.errorResponse(response, err, "createSnapshot");
      }
    });

    _defineProperty(this, "createPolicy", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          policyId: id,
          body: JSON.stringify(request.body)
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const rawRes = await callWithRequest("ism.createSMPolicy", params);
        const res = {
          seqNo: rawRes._seq_no,
          primaryTerm: rawRes._primary_term,
          id: rawRes._id,
          policy: rawRes.sm_policy
        };
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: res
          }
        });
      } catch (err) {
        return this.errorResponse(response, err, "createPolicy");
      }
    });

    _defineProperty(this, "updatePolicy", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const {
          seqNo,
          primaryTerm
        } = request.query;
        const params = {
          policyId: id,
          ifSeqNo: seqNo,
          ifPrimaryTerm: primaryTerm,
          body: JSON.stringify(request.body)
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const rawRes = await callWithRequest("ism.updateSMPolicy", params);
        const res = {
          seqNo: rawRes._seq_no,
          primaryTerm: rawRes._primary_term,
          id: rawRes._id,
          policy: rawRes.sm_policy
        };
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: res
          }
        });
      } catch (err) {
        return this.errorResponse(response, err, "updatePolicy");
      }
    });

    _defineProperty(this, "getPolicies", async (context, request, response) => {
      try {
        const {
          from,
          size,
          sortField,
          sortOrder,
          queryString
        } = request.query;
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        let params = {
          from,
          size,
          sortField: `sm_policy.${sortField}`,
          sortOrder,
          queryString: queryString.trim() ? `${queryString.trim()}` : "*"
        };
        const res = await callWithRequest("ism.getSMPolicies", params);
        const policies = res.policies.map(p => ({
          seqNo: p._seq_no,
          primaryTerm: p._primary_term,
          id: p._id,
          policy: p.sm_policy
        }));
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              policies,
              totalPolicies: res.total_policies
            }
          }
        });
      } catch (err) {
        if (err.statusCode === 404 && err.body.error.reason === "Snapshot management config index not found") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: {
                policies: [],
                totalPolicies: 0
              }
            }
          });
        }

        return this.errorResponse(response, err, "getPolicies");
      }
    });

    _defineProperty(this, "getPolicy", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          id: id
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const getResponse = await callWithRequest("ism.getSMPolicy", params);
        const metadata = await callWithRequest("ism.explainSnapshotPolicy", params);
        const documentPolicy = {
          id: id,
          seqNo: getResponse._seq_no,
          primaryTerm: getResponse._primary_term,
          policy: getResponse.sm_policy,
          metadata: metadata.policies[0]
        };
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: documentPolicy
          }
        });
      } catch (err) {
        if (err.statusCode === 404 && err.body.error.reason === "Snapshot management config index not found") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: null
            }
          });
        }

        return this.errorResponse(response, err, "getPolicy");
      }
    });

    _defineProperty(this, "deletePolicy", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          policyId: id
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const deletePolicyResponse = await callWithRequest("ism.deleteSMPolicy", params);

        if (deletePolicyResponse.result !== "deleted") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: deletePolicyResponse.result
            }
          });
        }

        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: true
          }
        });
      } catch (err) {
        return this.errorResponse(response, err, "deletePolicy");
      }
    });

    _defineProperty(this, "startPolicy", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          id: id
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const resp = await callWithRequest("ism.startSnapshotPolicy", params);

        if (resp.acknowledged) {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: true
            }
          });
        } else {
          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: "Failed to start snapshot policy."
            }
          });
        }
      } catch (err) {
        return this.errorResponse(response, err, "startPolicy");
      }
    });

    _defineProperty(this, "stopPolicy", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          id: id
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const resp = await callWithRequest("ism.stopSnapshotPolicy", params);

        if (resp.acknowledged) {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: true
            }
          });
        } else {
          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: "Failed to stop snapshot policy."
            }
          });
        }
      } catch (err) {
        return this.errorResponse(response, err, "stopPolicy");
      }
    });

    _defineProperty(this, "catRepositories", async (context, request, response) => {
      try {
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const res = await callWithRequest("cat.repositories", {
          format: "json"
        });
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: res
          }
        });
      } catch (err) {
        return this.errorResponse(response, err, "catRepositories");
      }
    });

    _defineProperty(this, "catRepositoriesWithSnapshotCount", async (context, request, response) => {
      try {
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const res = await callWithRequest("cat.repositories", {
          format: "json"
        });

        for (let i = 0; i < res.length; i++) {
          const getSnapshotRes = await callWithRequest("snapshot.get", {
            repository: res[i].id,
            snapshot: "_all",
            ignore_unavailable: true
          });
          res[i].snapshotCount = getSnapshotRes.snapshots.length;
        }

        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: res
          }
        });
      } catch (err) {
        return this.errorResponse(response, err, "catRepositoriesWithSnapshotCount");
      }
    });

    _defineProperty(this, "deleteRepository", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const res = await callWithRequest("snapshot.deleteRepository", {
          repository: id
        });
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: res
          }
        });
      } catch (err) {
        return this.errorResponse(response, err, "deleteRepository");
      }
    });

    _defineProperty(this, "getRepository", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const res = await callWithRequest("snapshot.getRepository", {
          repository: id
        });
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: res
          }
        });
      } catch (err) {
        return this.errorResponse(response, err, "getRepository");
      }
    });

    _defineProperty(this, "createRepository", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          repository: id,
          body: JSON.stringify(request.body)
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const res = await callWithRequest("snapshot.createRepository", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: res
          }
        });
      } catch (err) {
        return this.errorResponse(response, err, "createRepository");
      }
    });

    _defineProperty(this, "errorResponse", (response, error, methodName) => {
      console.error(`Index Management - SnapshotManagementService - ${methodName}:`, error);
      return response.custom({
        statusCode: 200,
        // error?.statusCode || 500,
        body: {
          ok: false,
          error: this.parseEsErrorResponse(error)
        }
      });
    });

    _defineProperty(this, "parseEsErrorResponse", error => {
      if (error.response) {
        try {
          const esErrorResponse = JSON.parse(error.response);
          return esErrorResponse.reason || error.response;
        } catch (parsingError) {
          return error.response;
        }
      }

      return error.message;
    });

    this.osDriver = osDriver;
  }

}

exports.default = SnapshotManagementService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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