/*
Copyright (C) 2017 Belledonne Communications SARL

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#ifndef _LINPHONE_CONFERENCE_PARAMS_HH
#define _LINPHONE_CONFERENCE_PARAMS_HH

#include "object.hh"




struct _LinphoneConferenceParams;


namespace linphone {

	class ConferenceParams;

	/**
	 * @brief Object defining parameters for a @ref Conference. 
	 *
	 * Can be created by calling function Core::createConferenceParams(). 
	 */
	class ConferenceParams: public Object {
	
		public:

			ConferenceParams(void *ptr, bool takeRef=true);
			LINPHONECXX_PUBLIC _LinphoneConferenceParams *cPtr() {return (_LinphoneConferenceParams *)mPrivPtr;}

	
		public:
			
		
			/**
			 * @brief Enable audio capabilities. 
			 *
			 * @param enable If true, audio will be enabled during conference 
			 */
			LINPHONECXX_PUBLIC void setAudioEnabled(bool enable);
			
			/**
			 * @brief Enable chat capabilities. 
			 *
			 * @param enable If true, chat is enabled during conference 
			 */
			LINPHONECXX_PUBLIC void setChatEnabled(bool enable);
			
			/**
			 * @brief Check whether audio capabilities are enabled. 
			 *
			 * @return true if the conference supports audio capabilities, false otherwise 
			 */
			LINPHONECXX_PUBLIC bool isAudioEnabled() const;
			
			/**
			 * @brief Check whether chat capabilities are enabled. 
			 *
			 * @return true if the conference supports chat capabilities, false otherwise 
			 */
			LINPHONECXX_PUBLIC bool isChatEnabled() const;
			
			/**
			 * @brief Returns whether local participant has to enter the conference. 
			 *
			 * @return true if local participant is by default part of the conference, false
			 * otherwise 
			 */
			LINPHONECXX_PUBLIC bool isLocalParticipantEnabled() const;
			
			/**
			 * @brief Returns whether conference can have only one participant. 
			 *
			 * @return true if the conference can have only one participant, false otherwise 
			 */
			LINPHONECXX_PUBLIC bool isOneParticipantConferenceEnabled() const;
			
			/**
			 * @brief Check whether video capabilities are enabled. 
			 *
			 * @return true if the conference supports video capabilities, false otherwise 
			 */
			LINPHONECXX_PUBLIC bool isVideoEnabled() const;
			
			/**
			 * @brief Enable local participant to enter the conference. 
			 *
			 * The local participant is the one driving the local @ref Core. It uses the local
			 * sound devices. The default value is true. Setting to false is mostly helpful
			 * when using liblinphone on a server application. 
			 * @param enable true if local participant is automatically added to the
			 * conference, false otherwise 
			 */
			LINPHONECXX_PUBLIC void setLocalParticipantEnabled(bool enable);
			
			/**
			 * @brief Enable conference with one participant. 
			 *
			 * @param enable true if conference can have only one participant, false otherwise 
			 */
			LINPHONECXX_PUBLIC void setOneParticipantConferenceEnabled(bool enable);
			
			/**
			 * @brief Enable video capabilities. 
			 *
			 * @param enable If true, video will be enabled during conference 
			 */
			LINPHONECXX_PUBLIC void setVideoEnabled(bool enable);
			
			/**
			 * @brief Clone a @ref ConferenceParams. 
			 *
			 * @return An allocated @ref ConferenceParams with the same parameters than params
			 *   
			 */
			LINPHONECXX_PUBLIC std::shared_ptr<linphone::ConferenceParams> clone() const;
			
			
	};

};

#endif // _LINPHONE_CONFERENCE_PARAMS_HH
