#! /usr/bin/perl

# ex:ts=8 sw=4:
# $OpenBSD: Quirks.pm,v 1.1316.2.1 2022/01/08 17:42:15 bket Exp $
#
# Copyright (c) 2009 Marc Espie <espie@openbsd.org>
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
# WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
# ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
# WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
# ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
# OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

use strict;
use warnings;
use OpenBSD::PackageName;

package OpenBSD::Quirks;

sub new
{
	my ($class, $version) = @_;
	if ($version <= 4) {
		return OpenBSD::Quirks4->new;
	} else {
		return undef;
	}
}

package OpenBSD::Quirks4;
use Config;
sub new
{
	my $class = shift;

	bless {}, $class;
}


# ->tweak_list(\@l, $state):
#	allows Quirks to do anything to the list of packages to install,
#	if something is needed. Usually, it won't do anything
sub tweak_list
{
}

# packages to remove
# stem => existing file   hash table
#	if file exists, then it's now in base and we can remove it.

my $p5a = $Config{archlib};
my $p5 = "/usr/libdata/perl5";
my $base_exceptions = {
# 6.5
	'libelf' => '/usr/lib/libelf.so.*',
# 6.9
	'acme-client' => '/usr/sbin/acme-client',
	'libexecinfo' => '/usr/include/execinfo.h',
};

my $stem_extensions = {
# 6.3
	'ruby-arirang' => 'arirang',
	'ja-mplus-ttf' => 'mixfont-mplus-ipa',
	'kdirstat' => 'qdirstat',
	'stem' => 'py-stem',
	'arm' => 'nyx',
	'luabitop' => 'lua-bitop',
	'livestreamer' => 'streamlink',
	'newsbeuter' => 'newsboat',
	'php-fastcgi' => 'php-cgi',
	'advancemess' => 'advancemame',
# 6.4
	'gnome-tweak-tool' => 'gnome-tweaks',
	'latexila' => 'gnome-latex',
	'osh' => 'etsh',
	'gnome-themes-standard' => 'gnome-themes-extra',
	'open-cobol' => 'gnucobol',
	'fanstasque-sans' => 'fantasque-sans',
	'rope' => 'py-rope',
	'py-upt-rubygems' => 'upt-rubygems',
	'pygame' => 'py-game',
	'py-cryptodomex' => 'py-cryptodome',
	'py3-cryptodomex' => 'py3-cryptodome',
	'py-crypto' => 'py-cryptodome',
	'py3-crypto' => 'py3-cryptodome',
	'py-link-grammar' => 'py3-link-grammar',
	'py-buildbot' => 'buildbot',
	'py-buildbot-www' => 'py3-buildbot-www',
	'py-buildbot-pkg' => 'py3-buildbot-pkg',
	'py-buildbot-grid-view' => 'py3-buildbot-grid-view',
	'py-buildbot-console-view' => 'py3-buildbot-console-view',
	'py-buildbot-waterfall-view' => 'py3-buildbot-waterfall-view',
	'py-buildslave' => 'buildbot-worker',
	'sdlmame' => 'mame',
	'sdlmame-tools' => 'mame',
	'sdlmess' => 'mame',
	'suricata-update' => 'suricata',
# 6.5
	'ValyriaTear' => 'valyriatear',
	'apertium-es-ast_NO-PN' => 'apertium-es-ast',
	'py-pafy' => 'py3-pafy',
	'py-libmagic' => 'py-magic',
	'py3-libmagic' => 'py3-magic',
	'pecl-chroot' => 'pecl71-chroot',
	'pecl-geoip' => 'pecl71-geoip',
	'pecl-http' => 'pecl71-pecl_http',
	'pecl-imagick' => 'pecl71-imagick',
	'pecl-libsodium' => 'pecl71-libsodium',
	'pecl-lzf' => 'pecl71-lzf',
	'pecl-mailparse' => 'pecl71-mailparse',
	'pecl-memcached' => 'pecl71-memcached',
	'pecl-proctitle' => 'pecl71-proctitle',
	'pecl-propro' => 'pecl71-propro',
	'pecl-raphf' => 'pecl71-raphf',
	'pecl-rar' => 'pecl71-rar',
	'pecl-redis' => 'pecl71-redis',
	'pecl-ssh2' => 'pecl71-ssh2',
	'pecl56-chroot' => 'pecl71-chroot',
	'pecl56-geoip' => 'pecl71-geoip',
	'pecl56-http' => 'pecl71-pecl_http',
	'pecl56-imagick' => 'pecl71-imagick',
	'pecl56-libsodium' => 'pecl71-libsodium',
	'pecl56-lzf' => 'pecl71-lzf',
	'pecl56-mailparse' => 'pecl71-mailparse',
	'pecl56-memcached' => 'pecl71-memcached',
	'pecl56-proctitle' => 'pecl71-proctitle',
	'pecl56-propro' => 'pecl71-propro',
	'pecl56-raphf' => 'pecl71-raphf',
	'pecl56-rar' => 'pecl71-rar',
	'pecl56-redis' => 'pecl71-redis',
	'pecl56-ssh2' => 'pecl71-ssh2',
	'pecl70-chroot' => 'pecl71-chroot',
	'pecl70-geoip' => 'pecl71-geoip',
	'pecl70-http' => 'pecl71-pecl_http',
	'pecl70-imagick' => 'pecl71-imagick',
	'pecl70-libsodium' => 'pecl71-libsodium',
	'pecl70-lzf' => 'pecl71-lzf',
	'pecl70-mailparse' => 'pecl71-mailparse',
	'pecl70-memcached' => 'pecl71-memcached',
	'pecl70-proctitle' => 'pecl71-proctitle',
	'pecl70-propro' => 'pecl71-propro',
	'pecl70-raphf' => 'pecl71-raphf',
	'pecl70-rar' => 'pecl71-rar',
	'pecl70-redis' => 'pecl71-redis',
	'pecl70-ssh2' => 'pecl71-ssh2',
	'zh-bg5ps' => 'bg5ps',
	'zh-bg5pdf' => 'bg5pdf',
	'wxWidgets-gtk2' => 'wxWidgets-gtk3',
	'atf-allwinner' => 'arm-trusted-firmware',
	'u-boot-pinebook' => 'u-boot-aarch64',
	'suricata-update' => 'suricata',
	'wireshark-gtk' => 'wireshark',
	'py-zeroconf' => 'py3-zeroconf',
	'digikam' => 'digikam-kde4',
	'apertium-sv-da' => 'apertium-swe-dan',
# 6.6
	'Xonotic' => 'xonotic',
	'Xonotic-data' => 'xonotic-data',
	'Xonotic-server' => 'xonotic-server',
	'gettext' => 'gettext-runtime',
	'kwebapp' => 'openradtool',
	'py-xmldiff' => 'py3-xmldiff',
	'bro' => 'zeek',
	'filter-rspamd' => 'opensmtpd-filter-rspamd',
	'filter-senderscore' => 'opensmtpd-filter-senderscore',
	'ilmbase' => 'OpenEXR',
	'openexr-viewers' => 'OpenEXR-tools',
	'libvirt-python' => 'py-libvirt',
# 6.7
	'py-jaraco-functoolsjaraco.functools' => 'py-jaraco-functools',
	'py3-jaraco-functoolsjaraco.functools' => 'py3-jaraco-functools',
	'py-netmiko' => 'py3-netmiko',
	'py-napalm' => 'py3-napalm',
	'py-IOSXR' => 'py3-IOSXR',
	'py-cligj' => 'py3-cligj',
	'py-fiona' => 'py3-fiona',
	'py-laspy' => 'py3-laspy',
	'py-owslib' => 'py3-owslib',
	'py-proj' => 'py3-proj',
	'py-planet' => 'py3-planet',
	'py-rasterio' => 'py3-rasterio',
	'py-rio-cogeo' => 'py3-rio-cogeo',
	'py-shapely' => 'py3-shapely',
	'py-terminado' => 'py3-terminado',
	'py-prometheus_client' => 'py3-prometheus_client',
	'py-influxdb' => 'py3-influxdb',
	'py-pandas' => 'py3-pandas',
	'py-siphon' => 'py3-siphon',
	'py-mlpack' => 'py3-mlpack',
	'py-bottleneck' => 'py3-bottleneck',
	'py-daemon' => 'py3-daemon',
	'py-nbval' => 'py3-nbval',
	'py-oauthlib' => 'py3-oauthlib',
	'py-requests-oauthlib' => 'py3-requests-oauthlib',
	'py-discogs-client' => 'py3-discogs-client',
	'py-semver' => 'py3-semver',
	'py-seaborn' => 'py3-seaborn',
	'py-natsort' => 'py3-natsort',
	'py-mastodon.py' => 'py3-mastodon.py',
	'py-bokeh' => 'py3-bokeh',
	'py-algorithm-munkres' => 'py3-algorithm-munkres',
	'py-pdfminer' => 'py3-pdfminer',
	'jupyter-notebook3' => 'jupyter-notebook',
	'py-html2text' => 'py3-html2text',
	'py-acme' => 'py3-acme',
	'py-josepy' => 'py3-josepy',
	'py-spyder-kernels' => 'py3-spyder-kernels',
	'spyder' => 'spyder3',
	'py-qtconsole' => 'py3-qtconsole',
	'py-sympy' => 'py3-sympy',
	'py-wurlitzer' => 'py3-wurlitzer',
	'py-gunicorn' => 'py3-gunicorn',
	'py-multidict' => 'py3-multidict',
	'py-elasticsearch' => 'py3-elasticsearch',
	'py-vmomi' => 'py3-vmomi',
	'py-nbconvert' => 'py3-nbconvert',
	'py-smbc' => 'py3-smbc',
	'py-git2' => 'py3-git2',
	'py-bencode' => 'py3-bencode',
	'py-dfwinreg' => 'py3-dfwinreg',
	'py-dfvfs' => 'py3-dfvfs',
	'py-dfdatetime' => 'py3-dfdatetime',
	'py-artifacts' => 'py3-artifacts',
	'py-rope' => 'py3-rope',
	'py-xlrd' => 'py3-xlrd',
	'ipython' => 'ipython3',
	'py-ipykernel' => 'py3-ipykernel',
	'py-jupyter_client' => 'py3-jupyter_client',
	'py-jupyter_core' => 'py3-jupyter_core',
	'py-nbformat' => 'py3-nbformat',
	'py-traitlets' => 'py3-traitlets',
	'py-parso' => 'py3-parso',
	'py-jedi' => 'py3-jedi',
	'py-botocore' => 'py3-botocore',
	'py-boto3' => 'py3-boto3',
	'py-s3transfer' => 'py3-s3transfer',
	'py-libpst' => 'py3-libpst',
	'Clementine' => 'clementine',
	'py-curtsies' => 'py3-curtsies',
	'bpython3' => 'bpython',
	'py-prompt_toolkit' => 'py3-prompt_toolkit',
	'py-wcwidth' => 'py3-wcwidth',
	'py-twitter' => 'py3-twitter',
	'py-irclib' => 'py3-irclib',
	'py-casttube' => 'py3-casttube',
	'py-libcloud' => 'py3-libcloud',
	'gegl03' => 'gegl04',
	'py-mysql' => 'py-mysqlclient',
# 6.8
	'py-dulwich' => 'py3-dulwich',
	'py-cups' => 'py3-cups',
	'py-fastimport' => 'py3-fastimport',
	'py-impacket' => 'py3-impacket',
	'py-pynetbox' => 'py3-pynetbox',
	'py-libvirt' => 'py3-libvirt',
	'py-libxml' => 'py3-libxml',
	'py-rsa' => 'py3-rsa',
	'py-unicorn' => 'py3-unicorn',
	'prison' => 'prison-kf5',
	'kdegraphics-mobipocket' => 'kdegraphics-mobipocket-kf5',
	'kdegraphics-thumbnailers' => 'kdegraphics-thumbnailers-kf5',
	'svgpart' => 'svgpart-kf5',
	'kdenetwork-filesharing' => 'kdenetwork-filesharing-kf5',
	'py-cloudpickle' => 'py3-cloudpickle',
	'py-podcastparser' => 'py3-podcastparser',
	'py-configargparse' => 'py3-configargparse',
	'py-aiodns' => 'py3-aiodns',
	'ebc' => 'bc-gh',
	'py-cares' => 'py3-cares',
	'py-astroid' => 'py3-astroid',
	'pylint' => 'pylint3',
	'pylint-gui' => 'pylint3-gui',
	'py-M2Crypto' => 'py3-M2Crypto',
	'sslscan-openssl' => 'sslscan',
	'py-rarfile' => 'py3-rarfile',
	'py-pygfm' => 'py3-pygfm',
	'py-humanize' => 'py3-humanize',
	'py-sabyenc' => 'py3-sabyenc',
	'py-cftime' => 'py3-cftime',
	'py-astral' => 'py3-astral',
	'py-ropper' => 'py3-ropper',
	'py-snuggs' => 'py3-snuggs',
	'py-osqp' => 'py3-osqp',
	'py-ecos' => 'py3-ecos',
	'py-numexpr' => 'py3-numexpr',
	'py-notmuch' => 'py3-notmuch',
	'py-webpy' => 'py3-webpy',
	'phonon-qt5' => 'phonon',
	'phonon-qt5-gstreamer' => 'phonon-backend-gstreamer',
	'phonon-qt5-vlc' => 'phonon-backend-vlc',
	'phonon-gstreamer-icons' => 'phonon-backend-gstreamer',
	'iosevka-fonts-default' => 'iosevka',
	'iosevka-fonts-term' => 'iosevka',
	'py-distutils-extra' => 'py3-distutils-extra',
	'uqm-remix1' => 'uqm-remix',
	'uqm-remix2' => 'uqm-remix',
	'uqm-remix3' => 'uqm-remix',
	'uqm-remix4' => 'uqm-remix',
	'py-responses' => 'py3-responses',
	'py-pudb' => 'py3-pudb',
        'py-keyring' => 'py3-keyring',
        'py-secretstorage' => 'py3-secretstorage',
        'rgain' => 'rgain3',
	'py-urwid' => 'py3-urwid',
	'py-ply' => 'py3-ply',
	'py-dtfabric' => 'py3-dtfabric',
# 6.9
	'py-qscintilla' => 'py3-qscintilla',
	'py-gobject3' => 'py3-gobject3',
	'py-nltk' => 'py3-nltk',
	'py-numpydoc' => 'py3-numpydoc',
	'py-passlib' => 'py3-passlib',
	'py-matplotlib' => 'py3-matplotlib',
	'py-patsy' => 'py3-patsy',
	'py-scs' => 'py3-scs',
	'py-augeas' => 'py3-augeas',
	'py-construct' => 'py3-construct',
	'py-sane' => 'py3-sane',
	'py-zmq' => 'py3-zmq',
	'py-kiwisolver' => 'py3-kiwisolver',
	'py-scipy' => 'py3-scipy',
	'spidermonkey' => 'spidermonkey78',
	'spidermonkey68' => 'spidermonkey78',
	'py-relatorio' => 'py3-relatorio',
	'py-croniter' => 'py3-croniter',
	'biopython' => 'py3-biopython',
	'py-rrd' => 'py-rrdtool',
	'py-whisper' => 'py3-whisper',
	'py-carbon' => 'py3-carbon',
	'py-elftools' => 'py3-elftools',
	'py-lockfile' => 'py3-lockfile',
	'py-toml' => 'py3-toml',
	'py-textfsm' => 'py3-textfsm',
	'py-unicodecsv' => 'py3-unicodecsv',
	'py-enchant' => 'py3-enchant',
	'py-sqlparse' => 'py3-sqlparse',
	'py-sqlalchemy-migrate' => 'py3-sqlalchemy-migrate',
	'py-pygresql' => 'py3-pygresql',
	'py-tagpy' => 'py3-tagpy',
	'py-mpd2' => 'py3-mpd2',
	'py-autobahn' => 'py3-autobahn',
	'py-txaio' => 'py3-txaio',
	'py-qrcode' => 'py3-qrcode',
	'py-acoustid' => 'py3-acoustid',
	'openjpeg' => 'openjp2',
	'py-mysqlclient' => 'py3-mysqlclient',
	'py-pyx' => 'py3-pyx',
	'fvwm2+fvicons' => 'fvwm2',
	'fvicons' => 'fvwm2',
	'py-frozen-flask' => 'py3-frozen-flask',
	'glibmm26' => 'glibmm268',
	'docker' => 'docker-cli',
	'py-lazy-object-proxy' => 'py3-lazy-project-proxy',
	'py3-scapy' => 'scapy',
	'py-sqlobject' => 'py3-sqlobject',
	'pyglet' => 'py-pyglet',
	'py-tzlocal' => 'py3-tzlocal',
	'py-sphinx-notfound-page' => 'py3-sphinx-notfound-page',
	'py-mulpyplexer' => 'py3-mulpyplexer',
	'py-commonmark' => 'py3-commonmark',
	'py-recommonmark' => 'py3-recommonmark',
	'py-sphinx_guzzle_theme' => 'py3-sphinx_guzzle_theme',
	'py-tornado' => 'py3-tornado',
	'py-sphinx-intl' => 'py3-sphinx-intl',
	'stegcracker' => 'stegseek',
	'py-ldap3' => 'py3-ldap3',
	'py-qtpy' => 'py3-qtpy',
	'py-sip' => 'py3-sip',
	'py-sip-qt5' => 'py3-pyqt5_sip',
	'py3-sip-qt5' => 'py3-pyqt5_sip',
	'py-qt5' => 'py3-qt5',
	'py-ndg_httpsclient' => 'py3-ndg_httpsclient',
	'py-trustme' => 'py3-trustme',
	'py-mechanize' => 'py3-mechanize',
	'py-ripe.atlas.cousteau' => 'py3-ripe.atlas.cousteau',
	'py-ripe.atlas.sagan' => 'py3-ripe.atlas.sagan',
	'py-qtawesome' => 'py3-qtawesome',
	'py-twisted' => 'py3-twisted',
	'py-hamcrest' => 'py3-hamcrest',
	'py-automat' => 'py3-automat',
	'py-constantly' => 'py3-constantly',
	'py-incremental' => 'py3-incremental',
	'py-zopeevent' => 'py3-zopeevent',
	'py-zopeinterface' => 'py3-zopeinterface',
	'py-zopetesting' => 'py3-zopetesting',
	'py-zopecomponent' => 'py3-zopecomponent',
	'py-service_identity' => 'py3-service_identity',
	'py-hyperlink' => 'py3-hyperlink',
	'py-asn1-modules' => 'py3-asn1-modules',
	'py-ldap' => 'py3-ldap',
	'py-ldap-examples' => 'py3-ldap-examples',
	'py-pexpect' => 'py3-pexpect',
	'py-gevent' => 'py3-gevent',
	'py-statgrab' => 'py3-statgrab',
	'py-parallel-ssh' => 'py3-parallel-ssh',
	'py-geventhttpclient' => 'py3-geventhttpclient',
	'py-gnupg' => 'py3-gnupg',
	'py-babel' => 'py3-babel',
	'py-imagesize' => 'py3-imagesize',
	'py-alabaster' => 'py3-alabaster',
	'py-docutils' => 'py3-docutils',
	'py-pygments' => 'py3-pygments',
	'py-snowballstemmer' => 'py3-snowballstemmer',
	'py-stemmer' => 'py3-stemmer',
	'py-sphinx' => 'py3-sphinx',
	'py-sphinx_rtd_theme' => 'py3-sphinx_rtd_theme',
	'py-codestyle' => 'py3-codestyle',
	'pyflakes3' => 'pyflakes',
	'py-dot' => 'py3-dot',
	'py-last' => 'py3-last',
	'py-flask-principal' => 'py3-flask-principal',
	'py-flask-sqlalchemy' => 'py3-flask-sqlalchemy',
	'py-flask-login' => 'py3-flask-login',
	'py-flask-wtf' => 'py3-flask-wtf',
	'py-pygal' => 'py3-pygal',
	'py-wtforms' => 'py3-wtforms',
	'py-apsw' => 'py3-apsw',
	'cdiff' => 'ydiff',
	'xmonad-lib' => 'xmonad',
	'py-test-httpbin' => 'py3-test-httpbin',
	'py-test-localserver' => 'py3-test-localserver',
	'py-test-vcr' => 'py3-test-vcr',
	'py-flask' => 'py3-flask',
	'py-httpbin' => 'py3-httpbin',
	'py-itsdangerous' => 'py3-itsdangerous',
	'py-vcrpy' => 'py3-vcrpy',
	'py-werkzeug' => 'py3-werkzeug',
	'py-blinker' => 'py3-blinker',
	'py-ifaddr' => 'py3-ifaddr',
	'py-colorama' => 'py3-colorama',
	'py-emoji' => 'py3-emoji',
	'py-query' => 'py3-query',
	'py-ofxparse' => 'py3-ofxparse',
	'py-puppetboard' => 'py3-puppetboard',
	'py-urlgrabber' => 'py3-urlgrabber',
	'py-webob' => 'py3-webob',
	'py-webtest' => 'py3-webtest',
	'py-genshi' => 'py3-genshi',
	'py-musicbrainzngs' => 'py3-musicbrainzngs',
	'py-setproctitle' => 'py3-setproctitle',
	'py-ptyprocess' => 'py3-ptyprocess',
	'py-jwt' => 'py3-jwt',
	'py-pg_activity' => 'pg_activity', # intentionally removing py- prefix
	'py-axolotl' => 'py3-axolotl',
	'py-protobuf' => 'py3-protobuf',
	'py-http_ece' => 'py3-http_ece',
	'yubiserve' => 'yubikeyedup',
	'py-openssl' => 'py3-openssl',
	'py-junos-eznc' => 'py3-junos-eznc',
	'py-ncclient' => 'py3-ncclient',
	'py-beautifulsoup4' => 'py3-beautifulsoup4',
	'py-beautifulsoup' => 'py3-beautifulsoup4',
# 7.0
	'weboob' => 'woob',
	'py-talloc' => 'py3-talloc',
	'py-tdb' => 'py3-tdb',
	'bijiben' => 'gnome-notes',
	'connections' => 'gnome-connections',
	'py-socketio-client' => 'py3-socketio-client',
	'py-cycler' => 'py3-cycler',
	'py-pyRFC3339' => 'py3-pyRFC3339',
	'py-libarchive-c' => 'py3-libarchive-c',
	'py-minimalmodbus' => 'py3-minimalmodbus',
	'baresip-gtk2' => 'baresip-gtk',
	'py-xlib' => 'py3-xlib',
	'py-neovim' => 'py3-neovim',
	'py-click-log' => 'py3-click-log',
	'py-click-plugins' => 'py3-click-plugins',
	'py-click-threading' => 'py3-click-threading',
	'py-spark-parser' => 'py3-spark-parser',
	'py-xdis' => 'py3-xdis',
	'py-uncompyle6' => 'py3-uncompyle6',
	'py-click' => 'py3-click',
	'py-pyinotify' => 'py3-pyinotify',
	'py-virtualdisplay' => 'py3-virtualdisplay',
	'py-meld3' => 'py3-meld3',
	'py-cryptography_vectors' => 'py3-cryptography_vectors',
	'py-boto' => 'py3-boto3',
	'py3-boto' => 'py3-boto3',
	'py-paramiko' => 'py3-paramiko',
	'py-nxos' => 'py3-nxos',
	'py-scp' => 'py3-scp',
	'py-bcrypt' => 'py3-bcrypt',
	'py-iso8601' => 'py3-iso8601',
	'py-asn1crypto' => 'py3-asn1crypto',
	'py-asn1' => 'py3-asn1',
	'py-snmp' => 'py3-snmp',
	'py-PyNaCl' => 'py3-PyNaCl',
	'py-websocket-client' => 'py3-websocket-client',
	'libmesode' => 'libstrophe',
	'py-stem' => 'py3-stem',
};

my $obsolete_reason = {};
my $obsolete_regexp = [];
sub setup_obsolete_reason
{
	while (@_ >= 2) {
		my ($i, $stem) = (shift, shift);
		if (ref $stem eq 'Regexp') {
			push (@$obsolete_regexp, [$stem, $i]);
		} else {
			$obsolete_reason->{$stem} = $i;
		}
	}
}

# this list is put in the "wrong" order (index => stem) because we
# want to put regexps as well in there (see the terraform or hs entries
# for instance)
setup_obsolete_reason(
# 6.3
	5 => 'puppet-dashboard',
	3 => 'dnsfilter',
	6 => 'empathy',
	3 => 'telepathy-salut',
	3 => 'telepthy-haze',
	3 => 'telepathy-idle',
	3 => 'telepathy-gabble',
	3 => 'telepathy-qt',
	3 => 'telepathy-farstream',
	3 => 'farstream',
	6 => 'libnice',
	5 => 'ardour',
	5 => 'aubio',
	5 => 'liblo',
	5 => 'libgnomecanvasmm',
	2 => 'mixmaster',
	0 => 'xscorch',
	5 => 'libwbxml',
	6 => 'haskell-platform',
	0 => 'gnokii',
	5 => 'dee',
	6 => 'pidgin-tlen',
	5 => 'pep8',
	6 => 'gnuvd',
	5 => 'gtkhtml4',
	6 => 'livestreamer-curses',
	6 => 'xdmchoose',
	0 => 'vomit',
	6 => 'p5-WWW-YouTube-Download',
	6 => 'oggtag',
	6 => 'aimsniff',
	6 => 'pork',
	3 => 'ntimed',
	2 => 'man2web',
	6 => 'decss',
	6 => 'libgcal',
	6 => 'akonadi-googledata',
	3 => 'arora',
	6 => 'powerdns-ldap',
	6 => 'p5-Net-LDNS',
	0 => 'sirc',
	6 => 'hgview',
	5 => 'xerces',
# 6.4
	2 => 'mozjs17',
	3 => 'qvwm',
	3 => 'prepop',
	3 => 'centerim',
	3 => 'gpgmepp',
	0 => 'xnc',
	3 => 'nepenthes',
	6 => 'mongrel2',
	6 => 'ja-groff',
	3 => 'gtkglextmm',
	3 => 'goocanvas',
	6 => 'osm2go',
	6 => 'kedpm',
	3 => 'mediatomb',
	6 => 'apache-couchdb',
	6 => 'py-couchdb',
	3 => 'chive',
	3 => 'opengroupware',
	3 => 'git-bz',
	3 => 'hot-babe',
	0 => 'wmgrabimage',
	0 => 'wmphoto',
	0 => 'wmminichess',
	0 => 'wmifinfo',
	0 => 'wmnet',
	0 => 'wmwave',
	0 => 'wmcb',
	0 => 'wmpinboard',
	0 => 'wmbiff',
	0 => 'wmmail',
	0 => 'wmtimer',
	3 => 'py-crypto',
	3 => 'py3-crypto',
	5 => 'tremor',
	5 => 'tremor-tools',
	3 => 'xtrkcad',
	2 => 'webkit',
	5 => 'py-test-capturelog',
	5 => 'py3-test-capturelog',
	3 => 'snapdl',
	6 => 'ccnet',
	10 => 'dkim-milter',
	11 => 'aircontrol',
	3 => 'corebird',
	5 => 'py-pcs',
	5 => 'ifmcstat',
	6 => 'suricata-update',
# 6.5
	5 => 'py-dtopt',
	5 => 'py3-dtopt',
	0 => 'memtest86+',
	5 => 'py-reat',
	12 => 'dovecot-antispam',
	3 => 'py-hgtools',
	3 => 'py3-hgtools',
	3 => 'py-hgsubversion',
	5 => 'py-hgnested',
	6 => 'py-keyczar',
	5 => 'py-hg-git',
	3 => 'py-czmq',
	13 => 'p5-Math-Pari',
	13 => 'p5-Math-BigInt-Pari',
	13 => 'p5-Crypt-Random',
	13 => 'p5-Crypt-DH',
	13 => 'p5-Crypt-Primes',
	13 => 'p5-Crypt-RSA',
	0 => 'directoryassistant',
	3 => 'x-pack',
	5 => 'libepc',
	0 => 'dysnomia-en',
	3 => 'casperjs',
	3 => 'phantomjs',
	6 => 'owncloud',
	13 => 'php-librdf',
	13 => 'nedi',
	13 => 'xcache',
	13 => 'pecl-memcache',
	13 => 'pecl56-memcache',
	13 => 'pecl-swish',
	13 => 'pecl56-swish',
	13 => 'pecl-uploadprogress',
	13 => 'pecl56-uploadprogress',
	3 => 'sickbeard',
	2 => 'nxssh',
	3 => 'opennx',
	3 => 'iaxcomm',
	0 => 'pinot',
	0 => 'gaia',
	0 => 'openafs',
	3 => 'emerillon',
	5 => 'mplex',
	0 => 'gromit',
	3 => 'discwrapper',
	6 => 'libhttpseverywhere',
	3 => 'poppler-qt4',
	3 => 'ushare',
	6 => 'ffmpeg2theora',
	3 => 'p5-WWW-Google-SiteMap',
	3 => 'p5-YAML-Perl',
	5 => 'py-nevow',
	3 => 'gupnp-ui',
	6 => 'suricata-update',
	1 => 'exploitdb-papers',
	3 => 'nautilus-sendto',
	2 => 'ghostview',
	7 => 'libcudf',
	7 => 'ocaml-cudf',
	7 => 'ocaml-cmdliner',
	7 => 'ocaml-cppo',
	7 => 'ocaml-dose',
	7 => 'ocaml-extlib',
	7 => 'ocaml-jsonm',
	7 => 'ocaml-re',
	7 => 'omake',
	7 => 'ounit',
	5 => 'obc',
	3 => 'mlbviewer',
	3 => 'upsd',
	0 => 'vagalume',
	7 => 'ocaml-uutf',
	3 => 'xbat',
	3 => 'xripple',
	3 => 'sliderule',
	3 => 'swisswatch',
	3 => 'xaniroc',
	3 => 'xmascot',
	3 => 'xmold',
	3 => 'xsnow',
	3 => 'xtattr',
	1 => 'ruby25-akami',
	1 => 'ruby25-gyoku',
	1 => 'ruby25-httpclient',
	1 => 'ruby25-httpi',
	1 => 'ruby25-nori',
	1 => 'ruby25-savon',
	1 => 'ruby25-sonos',
	1 => 'ruby25-wasabi',
	6 => 'linuxdoc',
	3 => 'luna',
	3 => 'rsynth',
	3 => 'agm',
	3 => 'connect4',
	3 => 'xchomp',
	3 => 'xdeblock',
	3 => 'xzip',
	3 => 'jive',
	3 => 'compface',
	3 => 'mirror',
	3 => 'dclock',
	3 => 'xfishtank',
	6 => 'mpeg_play',
# 6.6
	5 => 'py-async',
	5 => 'py3-async',
	5 => 'py3-funcsigs',
	1 => 'rust-doc',
	6 => 'glib-openssl',
	5 => 'gcj',
	5 => 'classpath',
	5 => 'jamvm',
	3 => 'gnome-code-assistance',
	3 => 'gedit-code-assistance',
	5 => 'py3-backports-abc',
	0 => 'libkgeomap',
	1 => 'piwik',
	5 => 'wml',
	1 => 'junit',
	6 => 'jeoip',
	0 => 'gnaughty',
	5 => 'mysql++',
	3 => 'mysqlcc',
	6 => 'relayd-updateconf',
	1 => 'java-getopt',
	5 => 'avidemux',
	5 => 'OGIlexicon',
	5 => 'festival',
	5 => 'festlex_CMU',
	5 => 'festlex_OALD',
	5 => 'festlex_POSLEX',
	5 => 'festvox_cmu_us_awb_arctic_hts',
	5 => 'festvox_cmu_us_bdl_arctic_hts',
	5 => 'festvox_cmu_us_jmk_arctic_hts',
	5 => 'festvox_cmu_us_slt_arctic_hts',
	5 => 'festvox_cstr_us_awb_arctic_multisyn',
	5 => 'festvox_cstr_us_jmk_arctic_multisyn',
	5 => 'festvox_don',
	5 => 'festvox_ellpc11k',
	5 => 'festvox_kallpc16k',
	5 => 'festvox_kedlpc16k',
	5 => 'festvox_rablpc16k',
	5 => 'voice_aec_di',
	5 => 'voice_jph_di',
	5 => 'voice_mwm_di',
	5 => 'voice_tll_di',
	3 => 'proxy-suite',
	6 => 'py-amf',
	3 => 'lam',
	6 => 'gxml',
	3 => 'nulib',
	3 => 'py-turbocheetah',
	3 => 'py-turbokid',
	3 => 'py-addons',
	3 => 'py-peak-rules',
	3 => 'py-prioritized_methods',
	3 => 'py-extremes',
	3 => 'py-kid',
	3 => 'py-Chart',
	3 => 'py-id3',
	3 => 'py-tagger',
	3 => 'py-zhCodecs',
	3 => 'py-iconvcodec',
	3 => 'py-cjkcodecs',
	3 => 'py-cdb',
	3 => 'py-pgsql',
	3 => 'pgworksheet',
	3 => 'py-HappyDoc',
	3 => 'py-Rijndael',
	3 => 'py-adns',
	3 => 'py-cherrypy2',
	3 => 'py-clientform',
	3 => 'py-crack',
	3 => 'py-cryptkit',
	3 => 'py-epydoc',
	3 => 'py-flowd',
	3 => 'py-flowtools',
	3 => 'py-gnuplot',
	3 => 'py-htmltmpl',
	3 => 'py-jaxml',
	3 => 'py-kiwi',
	3 => 'py-medusa',
	3 => 'py-monthdelta',
	3 => 'py-optik',
	3 => 'py-osd',
	3 => 'py-probstat',
	3 => 'py-protocols',
	3 => 'py-rtf',
	3 => 'py-ruledispatch',
	3 => 'py-silc',
	3 => 'py-tpg',
	3 => 'py-unit',
	3 => 'pybugz',
	3 => 'pymissile',
	6 => 'arm-elf-binutils',
	6 => 'arm-elf-gcc',
	6 => 'arm-elf-gdb',
	6 => 'arm-elf-newlib',
	5 => 'g77',
	5 => 'libf2c',
	3 => 'p5-WWW-Curl',
	5 => 'java-tanukiwraper',
	6 => 'opencv-docs',
	6 => 'p5-Device-USB',
	0 => 'p5-Crypt-GpgME',
	6 => 'p5-HTTP-BrowserDetect',
# 6.7
	3 => 'awless',
	3 => 'mousetweaks',
	6 => 'py-ioflo',
	6 => 'py3-ioflo',
	5 => 'postgresql-plv8',
	6 => 'py-GeoIP',
	3 => 'cvsync',
	6 => 'pecl-geoip',
	5 => 'clive',
	6 => 'aws-shell',
	5 => 'aqsis',
	3 => 'gtk-xfce-engine',
	0 => 'firewalk',
	5 => 'goffice08',
	3 => 'py-webdav',
	6 => 'yabause',
	5 => 'py-elementtidy',
	5 => 'py-gd',
	5 => 'py-oauth2',
	6 => 'afm',
	6 => 'p5-Geo-IP',
	3 => 'py-libnacl',
	5 => 'py3-typing',
	3 => 'p5-Net-IRC',
	0 => 'pangox-compat',
	5 => 'tmake',
	5 => 'qt3-sqlite3',
	5 => 'eigen',
	5 => 'eigen2',
	5 => 'qt-gstreamer',
	3 => 'qsynergy',
	5 => 'kqoauth',
	3 => 'valknut',
	3 => 'dclib',
	3 => 'luma',
	3 => 'qhacc',
	17 => 'nginx-geoip',
	3 => 'qgo',
	3 => 'qrfcview',
	3 => 'mutella',
	5 => 'erl-bcrypt',
	5 => 'erl-mochiweb',
	5 => 'rebar18',
	5 => 'rebar17',
	3 => 'bouml',
	3 => 'beediff',
	3 => 'php-mcrypt',
	3 => 'sqliteman',
	3 => 'entomologist',
	6 => 'fcitx-qt4',
	3 => 'partiwm',
	2 => 'ejabberd',
	5 => 'rebar16',
	5 => 'tortoisehg',
	3 => 'cjc',
	3 => 'spectrum',
	3 => 'py-xmpp',
	0 => 'mirage',
	0 => 'rpm',
	3 => 'oggconvert',
	3 => 'radiotray',
	0 => 'xevil',
	3 => 'py-vatnumber',
	6 => 'nap',
	5 => 'qucs',
	1 => 'py-arrow',
	1 => 'py3-arrow',
	1 => 'py-chai',
	1 => 'py3-chai',
	1 => 'py-binaryornot',
	1 => 'py3-binaryornot',
	1 => 'py-whichcraft',
	1 => 'py3-whichcraft',
	20 => 'celt07',
	1 => qr{^terraform-provider-},
	6 => 'radare2-bindings',
	1 => 'py3-django',
	1 => 'py3-django-lts',
	3 => 'dspam',
	3 => 'dspam-mysql',
	3 => 'dspam-pgsql',
	5 => 'py-iniparse',
	3 => 'qt5-declarative-xmllistmodel',
	3 => 'jabberd',
	22 => 'GeoIP',
	2 => 'akpop3d',
	6 => 'gtk+4-cloudprint',
	21 => qr{^drupal},
# 6.8
	3 => 'planner',
	3 => 'gnome-recipes',
	6 => 'shared-color-targets',
	6 => 'argyll',
	3 => 'libcroco',
	23 => 'libnipper',
	23 => 'nipper',
	13 => 'bg5pdf',
	23 => 'pdflib',
	3 => 'pop3gwd',
	3 => 'tircd',
	6 => 'gtksourceviewmm3',
	3 => 'hpodder',
	5 => 'py-gstreamer',
	3 => 'py-libxslt',
	3 => 'gconf-editor',
	3 => 'amide',
	3 => 'xmedcon',
	3 => 'gq',
	3 => 'gnome-doc-utils',
	3 => 'rarian',
	3 => 'qtcanvas3d',
	3 => 'qtcanvas3d-examples',
	3 => 'gtkparasite',
	3 => 'gutenpy',
	3 => 'klogic',
	3 => 'soprano',
	3 => 'automoc',
	3 => 'qimageblitz',
	3 => 'cagibi',
	3 => 'libkvkontakte',
	3 => 'libkfbapi',
	3 => 'libkgapi',
	3 => 'attica',
	3 => 'kwebkitpart',
	3 => 'grantlee',
	3 => 'libkscreen',
	3 => 'polkit-qt',
	3 => 'phonon-gstreamer',
	3 => 'phonon-gstreamer-icons',
	3 => 'phonon-vlc',
	3 => 'kde4-minimal',
	3 => 'kde4',
	3 => 'kde4-extras',
	3 => 'kdeadmin',
	3 => 'kdegames',
	3 => 'kdenetwork',
	3 => 'kdesdk',
	3 => 'kdetoys',
	3 => 'kdeadmin',
	3 => 'kdegames',
	3 => 'kdenetwork',
	3 => 'kdesdk',
	3 => 'kdetoys',
	3 => qr{^kde-l10n-},
	3 => 'amor',
	3 => 'audiocd-kio',
	3 => 'baloo-widgets',
	3 => 'ffmpegthumbs',
	3 => 'jovie',
	3 => 'kaccessible',
	3 => 'kactivities',
	3 => 'kde-base-artwork',
	3 => 'kde-emoticons',
	3 => 'kde-runtime',
	3 => 'kde-shared-data',
	3 => 'kde-wallpapers',
	3 => 'kde-workspace',
	3 => 'kde4-kopete-sounds',
	3 => 'kde4-langlist',
	3 => 'kde4-locale',
	3 => 'kde4-locale-en_US',
	3 => 'kdegraphics-strigi-analyzer',
	3 => 'kdelibs',
	3 => 'kdenetwork',
	3 => 'kdenetwork-strigi-analyzers',
	3 => 'kdepim',
	3 => 'kdepim-runtime',
	3 => 'kdepimlibs',
	3 => 'kdeplasma-addons',
	3 => 'kdesdk-strigi-analyzers',
	3 => 'kdewebdev',
	3 => 'kfilemetadata',
	3 => 'kgamma',
	3 => 'kgpg',
	3 => 'kimono',
	3 => 'kopete',
	3 => 'korundum',
	3 => 'kpat',
	3 => 'kppp',
	3 => 'kremotecontrol',
	3 => 'kross-java',
	3 => 'kscd',
	3 => 'ktux',
	3 => 'kuser',
	3 => 'libkcddb',
	3 => 'libkcompactdisc',
	3 => 'libkdeedu',
	3 => 'libkdegames',
	3 => 'libkexiv2',
	3 => 'mplayerthumbs',
	3 => 'nepomuk-core',
	3 => 'nepomuk-widgets',
	3 => 'p5-kde',
	3 => 'p5-qt',
	3 => 'pairs',
	3 => 'parley',
	3 => 'py-kde',
	3 => 'qyoto',
	3 => 'ruby26-qt4',
	3 => 'smokegen',
	3 => 'smokekde',
	3 => 'smokeqt',
	3 => 'superkaramba',
	3 => 'gbirthday',
	4 => 'wireguard-go',
	4 => 'wiresep',
	4 => 'wireless',
	6 => 'sisctrl',
	0 => 'vteplugin',
	5 => 'p5-Net-GPSD',
	0 => 'mozplugger',
	3 => 'ajaxterm',
	1 => 'uim-kde',
	1 => 'uim-qt',
	1 => 'uim-qt4',
	5 => 'dbusmenu-qt',
	3 => 'qlandkartegt',
	3 => 'qca',
	3 => 'qca-gnupg',
	3 => 'qca-ossl',
	26 => 'keepassx',
	7 => 'go-xlsx',
	7 => 'go-check-v1',
	6 => 'freemat',
	6 => 'scim-qtimm',
	6 => 'gambatte-qt',
	3 => 'liblxqt-l10n',
	3 => 'lxqt-about-l10n',
	3 => 'lxqt-config-l10n',
	3 => 'lxqt-notificationd-l10n',
	3 => 'lxqt-openssh-askpass-l10n',
	3 => 'lxqt-panel-l10n',
	3 => 'lxqt-policykit-l10n',
	3 => 'lxqt-powermanagement-l10n',
	3 => 'lxqt-runner-l10n',
	3 => 'lxqt-session-l10n',
	3 => 'lxqt-sudo-l10n',
	3 => 'lximage-l10n',
	3 => 'compton-conf-l10n',
	3 => 'obconf-l10n',
	3 => 'libfm-qt-l10n',
	3 => 'pavucontrol-qt-l10n',
	3 => 'pcmanfm-qt-l10n',
	3 => 'qtermwidget-l10n',
	3 => 'qterminal-l10n',
	6 => 'gmc4cc',
	5 => 'py-sockjs-tornado',
	5 => 'apache-activemq',
	5 => 'libgda-ui',
	6 => 'ocrfeeder',
	5 => 'goocanvas2',
	5 => 'py-termcolor',
	5 => 'py3-termcolor',
	3 => 'py-carddav',
	5 => 'py-rdflib',
	5 => 'py3-rdflib',
	3 => 'fookebox',
	5 => 'py-pylons',
	5 => 'py-routes',
	5 => 'py-weberror',
	5 => 'py-webhelpers',
	0 => 'hedgewars',
	0 => 'lazarus',
	13 => 'exaile',
	3 => 'instead-launcher',
	3 => qr{^(ruby(19|2[0-5])-|ruby-[^0-9])},
# 6.9
	3 => 'py-notify',
	3 => 'glimpse',
	3 => 'py-metlog',
	3 => 'pdfmod',
	5 => 'hyena',
	5 => 'mono-taglib',
	3 => 'spatialite_gis',
	3 => 'librewms',
	3 => 'libgaiagraphics',
	3 => 'librasterlite',
	5 => 'dbus-sharp',
	5 => 'dbus-sharp-glib',
	5 => 'mono-gdata',
	5 => 'mono-notify',
	5 => 'ndesk-dbus',
	5 => 'ndesk-dbus-glib',
	5 => 'mono-xsp',
	3 => 'py-Numeric',
	3 => 'k3dsurf',
	3 => 'spacehulk',
	3 => 'taxipilot',
	3 => 'pympd',
	5 => 'py-vte',
	5 => 'vte',
	6 => 'keybinder',
	3 => 'vinagre',
	5 => 'gnome-getting-started-docs',
	3 => 'py-poppler',
	3 => 'pdfshuffler',
	27 => 'colord',
	27 => 'colord-gtk',
	5 => 'gnome-color-manager',
	3 => 'zeya',
	3 => 'py-trollius',
	3 => qr{^kde-i18n-},
	3 => 'kde-shared-data',
	3 => 'kde3-langlist',
	3 => 'kde3-locale',
	3 => 'kde3-locale-en_US',
	3 => 'kdeaccessibility',
	3 => 'kdebase',
	3 => 'kdegames',
	3 => 'kdelibs',
	3 => 'kdesamba',
	3 => 'qt3-examples',
	3 => 'qt3-html',
	3 => 'qt3-mt',
	3 => 'qt3-mysql',
	3 => 'qt3-postgresql',
	3 => 'qt3-sqlite2',
	6 => 'gtk3mm-documentation',
	5 => 'libxml++3',
	3 => 'orage',
	3 => 'xfce4-statusnotifier',
	3 => 'pylint3-gui',
	3 => 'e_dbus',
	3 => 'ecore',
	3 => 'edje',
	3 => 'eet',
	3 => 'efreet',
	3 => 'eina',
	3 => 'eio',
	3 => 'elementary',
	3 => 'embryo',
	3 => 'emotion',
	3 => 'enlightenment',
	3 => 'ethumb',
	3 => 'evas',
	3 => 'qtstyleplugins',
	0 => 'p5-Regexp-Copy',
	28 => 'mcollective-puppet-agent',
	3 => 'kdepim-apps-libs',
	29 => 'gns3',
	3 => 'py-qt4',
	3 => 'py-qt4-docs',
	29 => 'hitch',
	30 => 'xtraceroute',
	31 => 'py-vorbis',
	31 => 'py-ogg',
	32 => 'sattrack',
	33 => 'sattrack',
	34 => 'packit',
	34 => 'nemesis',
	6 => 'lxnb',
	35 => 'siphon',
	6 => 'avinfo',
	0 => 'xfed',
	36 => 'www6to4',
	37 => 'gimmix',
	3 => 'xhippo',
	0 => 'hnb',
	38 => 'icinga',
	38 => 'icinga-web',
	38 => 'icinga-cgi',
	38 => 'icinga-idoutils',
	39 => 'swftools',
	39 => 'swfmill',
	39 => 'ming',
	39 => 'quvi',
	39 => 'libquvi',
	39 => 'libquvi-scripts',
	6 => 'cclive',
	4 => 'viz',
	40 => 'driftnet',
	6 => 'enjoympeg',
	6 => 'trac-ldapplugin',
	0 => 'xboing',
	3 => 'beaver',
	3 => 'tracnav',
	39 => 'flasm',
	31 => 'mpd-add-similar',
	31 => 'py-mpd',
	3 => 'abs',
	3 => 'arena',
	3 => 'teagtk',
	5 => 'tempwatch',
	29 => 'avenger',
	41 => 'mk',
	1 => 'sabredav',
	3 => 'mollify',
	3 => 'seyon',
	6 => 'mingw',
	3 => 'bsd-airtools',
	3 => 'gtkpod',
	0 => 'golem',
	3 => 'wmmultipop3',
	3 => 'wmpop3',
	3 => 'angst',
	3 => 'audiopreview',
	3 => 'pcsxr',
	41 => 'libbio',
	41 => 'libfmt',
	41 => 'libregexp9',
	41 => 'libutf',
	5 => qr{^hs-},
	6 => 'gstreamer',
	6 => 'gstreamer-ffmpeg',
	6 => 'gstreamer-plugins-bad',
	6 => 'gstreamer-plugins-base',
	6 => 'gstreamer-plugins-gl',
	6 => 'gstreamer-plugins-good',
	6 => 'gstreamer-plugins-ugly',
	29 => 'lives',
	5 => 'py-wsgiproxy',
	5 => 'py-sqlite',
	43 => 'upt',
	43 => 'upt-cpan',
	43 => 'upt-openbsd',
	43 => 'upt-pypi',
	43 => 'upt-rubygems',
	3 => 'p5-VCP-autrijus',
# 7.0
	3 => 'mailpile',
	7 => 'p5-Geo-GDAL',
	3 => 'gnome-latex',
	3 => 'p5-Puppet-Tidy',
	6 => 'kf5-icons-baloo',
	3 => 'canto',
	6 => 'py-mini-amf',
	6 => 'py3-mini-amf',
	44 => 'qucs-s',
	3 => 'qt4',
	3 => 'qt4-examples',
	3 => 'qt4-html',
	3 => 'qt4-mysql',
	3 => 'qt4-postgresql',
	3 => 'qt4-sqlite2',
	3 => 'qt4-tds',
	1 => 'antlr3',
	6 => 'enigmail',
	6 => 'enigmail-seamonkey',
	29 => 'google-compute-engine',
	3 => 'bzr',
	3 => 'bzr-svn',
	6 => 'libvstr',
);

# though it's not yet used, these should be pkgnames, so that eventually
# pkg_add is able to act on it
my $obsolete_suggestion = {
	'xgrab' => [qw(scrot xwd)],
	'dkim-milter' => 'opendkim',
	'qhacc' => [qw(homebank gnucash kmymoney)],
	'qgo' => 'kigo',
	'beediff' => 'kompare',
	'sqliteman' => [qw(sqlitebrowser kexi)],
	'gbirthday' => 'qbirthday',
	'qlandkartegt' => 'qmapshack',
	'keepassx' => 'keepassxc',
	'lives' => [qw(kdenlive shotcut)],
	'pdfshuffler' => 'pdfarranger',
};

# reasons for obsolete packages
my $obsolete_message = {
	0 => "ancient software that doesn't work",
	1 => "no benefit to being packaged",
	2 => "no longer maintained and full of security holes",
	3 => "no longer maintained upstream",
	4 => "superseded by base component",
	5 => "outdated and/or no longer required by other ports",
	6 => "no longer useful",
	7 => "removed in favor of using the language's package manager",
	10 => "has been replaced by opendkim",
	11 => "no longer packageable",
	12 => "replace with IMAPSieve, see https://wiki.dovecot.org/HowTo/AntispamWithSieve",
	13 => "has a dependency on obsolete software",
	17 => "old GeoIP databases end-of-life, see alternative using geoip2/libmaxminddb",
	20 => "merged into IETF Opus codec, obsolete, audio/mumble uses bundled version now",
	21 => "upstream recommends to use composer to build a drupal site",
	22 => "the original GeoIP database is end of life; use libmaxminddb/GeoIP2",
	23 => "no longer maintained upstream, became commercial over ten years ago",
	26 => "outdated Qt4 application",
	27 => "requires GUdev or GUsb",
	28 => "consider migrating MCollective agents and filters using tools like Bolt and PuppetDB's Puppet Query Language",
	29 => "unmaintained port that was blocking other changes in ports",
	30 => "crashes in many different ways at runtime, does not build with '-fno-common'",
	31 => "upstream is dead, python2-only, nothing uses it",
	33 => "no longer maintained upstream, website suggests https://cronometer.com until rewrite is done",
	34 => "upstream is dead, basic functionality does not work, use tcpdump(1) and scapy(1) instead",
	35 => "upstream is dead, no longer useful, does not support IPv6",
	36 => "ancient software, use pf.conf(5) 'af-to'",
	37 => "upstream is dead, default settings cause crash when connecting to mpd",
	38 => "icinga 1 is end-of-life, migrate to icinga 2 (needs config rewrite)",
	39 => "Flash/SWF is end-of-life",
	40 => "ancient software that often crashes and relies on single HTTP (no TLS) connections, use wireshark",
	41 => "upstream moved to unversioned tarballs, use the plan9port (same upstream) package instead",
	43 => "using portgen instead is recommended",
	44 => "qucs-s Qt4 UI not working correctly, suggest xschem or kicad's eeschema instead",
};

# ->is_base_system($handle, $state):
#	checks whether an existing handle is now part of the base system
#	and thus no longer needed.

sub is_base_system
{
	my ($self, $handle, $state) = @_;

	my $pkgname = $handle->pkgname;
	my $stem = OpenBSD::PackageName::splitstem($pkgname);

	my $test = $base_exceptions->{$stem};
	if (defined $test) {
		require File::Glob;
		if (defined File::Glob::bsd_glob($test)) {
			$state->say("Removing #1 #2", $handle->pkgname,
			    " (part of base system now)");
			return 1;
		} else {
			$state->say("Not removing #1 #2 #3 #4", $handle->pkgname,
			 ", ", $test, " not found");
			return 0;
		}
	} else {
		return 0;
	}
}

# ->filter_obsolete(\@list)
# explicitly mark packages as no longer there. Remove them from the
# list of "normal" stuff.

sub filter_obsolete
{
	my ($self, $list, $state) = @_;
	my @in = @$list;
	$list = [];
	for my $pkgname (@in) {
		my $stem = OpenBSD::PackageName::splitstem($pkgname);
		my $reason = $obsolete_reason->{$stem};
		if (!defined $reason) {
			for my $o (@$obsolete_regexp) {
				if ($pkgname =~ m/$o->[0]/) {
					$reason = $o->[1];
					last;
				}
			}
		}
		my $suggestion = $obsolete_suggestion->{$stem};
		my @l = ();
		if (defined $reason) {
			push(@l, $obsolete_message->{$reason});
		}
		if (defined $suggestion) {
			if (ref($suggestion)) {
				$suggestion = join(', ', @$suggestion);
			}
			push(@l, $state->f("suggest #1", $suggestion));
		}
		if (@l > 0) {
			$state->say("Obsolete package: #1 (#2)", $pkgname, 
			    join(', ', @l));
		} else {
			push(@$list, $pkgname);
		}
	}
}


# ->tweak_search(\@s, $handle, $state):
#	tweaks the normal search for a given handle, in case (for instance)
#	of a stem name change.

sub tweak_search
{
	my ($self, $l, $handle, $state) = @_;

	if (@$l == 0 || !$l->[0]->can("add_stem")) {
		return;
	}
	my $stem = OpenBSD::PackageName::splitstem($handle->pkgname);
	my $extra = $stem_extensions->{$stem};
	if (defined $extra) {
		if (ref $extra) {
			for my $e (@$extra) {
				$l->[0]->add_stem($e);
			}
		} else {
			$l->[0]->add_stem($extra);
		}
	}
}

# list of
#   cat/path => badspec
my $cve = {
	'archivers/brotli' => 'brotli-<1.0.9',
	'archivers/cabextract' => 'cabextract-<1.8',
	'archivers/libmspack' => 'libmspack-<0.8alpha',
	'archivers/lz4' => 'lz4-<1.9.3p0',
	'archivers/p5-Archive-Zip' => 'p5-Archive-Zip-<1.64',
	'audio/flac' => 'flac-<1.3.0p1',
	'databases/mariadb,-main' => 'mariadb-client-<10.3.22',
	'databases/mariadb,-server' => 'mariadb-server-<10.3.15',
	'databases/postgresql,-main' => 'postgresql-client-<10.6',
	'databases/postgresql,-server' => 'postgresql-server-<10.6',
	'databases/sqlite3' => 'sqlite3-<3.25.3',
	'devel/apache-ant' => 'apache-ant-<1.10.9',
	'devel/git,-main' => 'git-<2.30.2',
	'devel/git,-svn' => 'git-svn-<2.30.2',
	'devel/git,-x11' => 'git-x11-<2.30.2',
	'devel/jenkins/devel' => 'jenkins-<2.154',
	'devel/jenkins/stable' => 'jenkins-<2.138.4',
	'devel/libgit2/libgit2' => 'libgit2-<0.27.7',
	'devel/mercurial,-main' => 'mercurial-<4.5.3p1',
	'devel/mercurial,-x11' => 'mercurial-x11-<4.5.3p1',
	'devel/pcre' => 'pcre-<8.38',
	'devel/sdl2-image' => 'sdl2-image-<2.0.4',
	'editors/vim,-main' => 'vim-<8.1.1483',
	'games/gnuchess' => 'gnuchess-<6.2.9',
	'graphics/png' => 'png-<1.6.37',
	'graphics/tiff' => 'tiff-<4.0.4beta',
	'lang/php/7.1,-main' => 'php->7.1,<7.1.29',
	'lang/php/7.2,-main' => 'php->7.2,<7.2.18',
	'lang/php/7.3,-main' => 'php->7.3,<7.3.5',
	'lang/python/2.7,-main' => 'python->2.7,<2.7.16',
	'lang/python/3.7,-main' => 'python->3.7,<3.7.9',
	'lang/python/3.8,-main' => 'python->3.8,<3.8.8',
	'lang/python/3.9,-main' => 'python->3.9,<3.9.2',
	'lang/ruby/2.3,-main' => 'ruby-<2.3.8',
	'lang/ruby/2.4,-main' => 'ruby->2.4,<2.4.5p2',
	'lang/ruby/2.5,-main' => 'ruby->2.5,<2.5.5',
	'lang/ruby/2.6,-main' => 'ruby->2.6,<2.6.2',
	'mail/dovecot,-server' => 'dovecot-<2.3.10.1',
	'mail/exim' => 'exim-<4.83',
	'mail/isync' => 'isync-<1.4.2',
	'mail/mailman' => 'mailman-<2.1.30',
	'mail/p5-Mail-SpamAssassin' => 'p5-Mail-SpamAssassin-<3.4.4',
	'mail/roundcubemail' => 'roundcubemail-<1.3.8',
	'math/hdf5' => 'hdf5-<1.8.21',
	'multimedia/libquicktime' => 'libquicktime-<1.2.4p13',
	'net/curl' => 'curl-<7.65.0',
	'net/dhcpcd' => 'dhcpcd-<7.2.2',
	'net/dino' => 'dino-<0.2.1',
	'net/haproxy' => 'haproxy-<2.4.4',
	'net/icecast' => 'icecast-<2.4.4',
	'net/irssi' => 'irssi-<1.2.1',
	'net/isc-bind' => 'isc-bind-<9.16.3',
	'net/libssh' => 'libssh-<0.9.5',
	'net/libssh2' => 'libssh2-<1.8.2',
	'net/lldpd' => 'lldpd-<0.7.18p0',
	'net/mosquitto' => 'mosquitto-<1.5.6',
	'net/ntp' => 'ntp-<4.2.8pl7',
	'net/openconnect' => 'openconnect-<8.10',
	'net/openvpn' => 'openvpn-<2.4.9',
	'net/powerdns,-main' => 'powerdns-<4.1.5',
	'net/powerdns,-mysql' => 'powerdns-mysql-<4.1.5',
	'net/powerdns,-pgsql' => 'powerdns-pgsql-<4.1.5',
	'net/powerdns_recursor' => 'powerdns-recursor-<4.3.1',
	'net/rsync' => 'rsync-<3.1.3p0',
	'net/samba,-main' => 'samba-<4.8.18',
	'net/tinc' => 'tinc-<1.0.35v0',
	'net/transmission,-gtk' => 'transmission-gtk-<2.84',
	'net/transmission,-main' => 'transmission-<2.84',
	'net/transmission,-qt' => 'transmission-qt-<2.84',
	'net/wget' => 'wget-<1.20.3',
	'net/wireshark,-gtk' => 'wireshark-gtk-<2.6.3',
	'net/wireshark,-main' => 'wireshark-<2.6.3',
	'net/wireshark,-text' => 'tshark-<2.6.3',
	'net/zeromq' => 'zeromq-<4.3.3',
	'net/znc' => 'znc-<1.7.3',
	'print/cups,-main' => 'cups-<1.7.4',
	'security/clamav' => 'clamav-<0.100.2',
	'security/libssh' => 'libssh-<0.9.4',
	'security/opensc' => 'opensc-<0.20.0',
	'security/openssl/1.1' => 'openssl-<1.1.1g',
	'security/polarssl' => 'mbedtls-<2.16.12',
	'security/sudo' => 'sudo-<1.8.31',
	'shells/bash' => 'bash-<4.3.27',
	'sysutils/ansible,-main' => 'ansible-<2.7.1',
	'sysutils/mcollective' => 'mcollective-<2.5.3',
	'sysutils/rclone' => 'rclone-<1.53.3',
	'sysutils/salt' => 'salt-<3002',
	'telephony/asterisk,-main' => 'asterisk-<13.23.1',
	'telephony/coturn' => 'turnserver-<4.5.1.2',
	'textproc/mdbook' => 'mdbook-<0.4.6',
	'www/apache-httpd,-main' => 'apache-httpd-<2.4.35',
	'www/bozohttpd' => 'bozohttpd-<20130711p0',
	'www/chromium' => 'chromium-<69.0.3497.100',
	'www/drupal7/core7' => 'drupal->=7.0,<7.60',
	'www/drupal7/theme-newsflash' => 'drupal7-theme-newsflash-<2.5',
	'www/gitea' => 'gitea-<1.7.6',
	'www/hiawatha' => 'hiawatha-<10.8.4',
	'www/iridium' => 'iridium-<2018.5.67',
	'www/jupyter-notebook' => 'jupyter-notebook-<5.7.8',
	'www/jupyter-notebook,python3' => 'jupyter-notebook3-<5.7.8',
	'www/mozilla-firefox' => 'firefox-<62.0.2p0',
	'www/nginx' => 'nginx-<1.4.1',
	'www/p5-CGI-Application' => 'p5-CGI-Application-<4.50p0',
	'www/p5-Catalyst-Plugin-Static-Simple' => 'p5-Catalyst-Plugin-Static-Simple-<0.36',
	'www/privoxy' => 'privoxy-<3.0.31',
	'www/py-bottle' => 'py-bottle-<0.12.19',
	'www/py-bottle,python3' => 'py3-bottle-<0.12.19',
	'www/py-django/lts' => 'py-django-lts-<1.11.19',
	'www/py-django/stable' => 'py-django-<2.1.6',
	'www/py-requests' => 'py-requests-<2.20.0',
	'www/py-requests,python3' => 'py3-requests-<2.20.0',
	'www/py-urllib3' => 'py-urllib3-<1.26.5',
	'www/py-urllib3,python3' => 'py3-urllib3-<1.26.5',
	'www/ruby-rack,ruby24' => 'ruby24-rack-<2.0.6',
	'www/ruby-rack,ruby25' => 'ruby25-rack-<2.0.6',
	'www/tomcat/v7' => 'tomcat-<7.0.104',
	'www/tomcat/v8' => 'tomcat-<8.5.55',
	'www/tomcat/v9' => 'tomcat-<9.0.35',
	'www/webkitgtk4' => 'webkitgtk4-<2.20.5',
	'x11/gnome/gdm' => 'gdm-<3.28.3',
	'x11/rdesktop' => 'rdesktop-<1.8.4',
};
# please maintain sort order in above $cve list, future updates need to
# replace existing entries

for my $sub (qw(calendar http_post ldap odbc pgsql snmp speex tds)) {
	$cve->{"telephony/asterisk,-$sub"} = "asterisk-$sub-<13.23.1";
}

for my $sub (qw(apache cgi dbg bz2 curl dba gd gmp intl imap ldap mysqli
    odbc pcntl pdo_mysql pdo_odbc pdo_pgsql pdo_sqlite pgsql pspell
    shmop soap snmp sqlite3 pdo_dblib tidy xmlrpc xsl zip mysql
    sybase_ct mssql mcrypt)) {
	$cve->{"lang/php/7.1,-$sub"} = "php-$sub->7.1,<7.1.22";
	$cve->{"lang/php/7.2,-$sub"} = "php-$sub->7.2,<7.2.10";
}


# ->check_security($path)
#	return an insecure specification for the matching path
#	e.g., you should update.

sub check_security
{
	my ($self, $path) = @_;
	if (defined $cve->{$path}) {
		return $cve->{$path};
	} else {
		return undef;
	}
}

my $optional_tag = {
#	emacs => 'emacs-*|xemacs-*',
};

# -> is_optional_tag($tag)
#	return either undef or a pkgspec where to find the definition

sub is_optional_tag
{
	my ($self, $tag) = @_;
	return $optional_tag->{$tag->name};
}

# doesn't do anything yet, but will be used to check consistency eventually
sub sanity_check
{
	my $self = shift;
	my $rc = 0;
	my $pointed;
	while (my ($k, $v) = each %$obsolete_reason) {
		if (!defined $obsolete_message->{$v}) {
			print STDERR "$k points to $v with no message\n";
			$rc = 1;
		}
		$pointed->{$v} = 1;
	}
	# XXX until we merge regexps
	$pointed->{21} = 1;
	while (my ($k, $v) = each %$obsolete_message) {
		if (!$pointed->{$k}) {
			print STDERR "Message key $k ($v) is unused\n";
			$rc = 1;
		}
	}
	exit $rc if $rc;
}

1;
