/*
 * Decompiled with CFR 0.152.
 */
package org.traccar.database;

import java.sql.SQLException;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.traccar.Context;
import org.traccar.config.Config;
import org.traccar.config.Keys;
import org.traccar.database.BaseObjectManager;
import org.traccar.database.DataManager;
import org.traccar.database.IdentityManager;
import org.traccar.database.ManagableObjects;
import org.traccar.model.Device;
import org.traccar.model.DeviceAccumulators;
import org.traccar.model.DeviceState;
import org.traccar.model.Group;
import org.traccar.model.Position;
import org.traccar.model.Server;

public class DeviceManager
extends BaseObjectManager<Device>
implements IdentityManager,
ManagableObjects {
    private static final Logger LOGGER = LoggerFactory.getLogger(DeviceManager.class);
    private final Config config;
    private final long dataRefreshDelay;
    private Map<String, Device> devicesByUniqueId;
    private Map<String, Device> devicesByPhone;
    private final AtomicLong devicesLastUpdate = new AtomicLong();
    private final Map<Long, Position> positions = new ConcurrentHashMap<Long, Position>();
    private final Map<Long, DeviceState> deviceStates = new ConcurrentHashMap<Long, DeviceState>();

    public DeviceManager(DataManager dataManager) {
        super(dataManager, Device.class);
        this.config = Context.getConfig();
        try {
            this.writeLock();
            if (this.devicesByPhone == null) {
                this.devicesByPhone = new ConcurrentHashMap<String, Device>();
            }
            if (this.devicesByUniqueId == null) {
                this.devicesByUniqueId = new ConcurrentHashMap<String, Device>();
            }
        }
        finally {
            this.writeUnlock();
        }
        this.dataRefreshDelay = this.config.getLong(Keys.DATABASE_REFRESH_DELAY) * 1000L;
        this.refreshLastPositions();
    }

    @Override
    public long addUnknownDevice(String uniqueId) {
        Device device = new Device();
        device.setName(uniqueId);
        device.setUniqueId(uniqueId);
        device.setCategory(Context.getConfig().getString(Keys.DATABASE_REGISTER_UNKNOWN_DEFAULT_CATEGORY));
        long defaultGroupId = Context.getConfig().getLong(Keys.DATABASE_REGISTER_UNKNOWN_DEFAULT_GROUP_ID);
        if (defaultGroupId != 0L) {
            device.setGroupId(defaultGroupId);
        }
        try {
            this.addItem(device);
            LOGGER.info("Automatically registered device " + uniqueId);
            if (defaultGroupId != 0L) {
                Context.getPermissionsManager().refreshDeviceAndGroupPermissions();
                Context.getPermissionsManager().refreshAllExtendedPermissions();
            }
            return device.getId();
        }
        catch (SQLException e) {
            LOGGER.warn("Automatic device registration error", (Throwable)e);
            return 0L;
        }
    }

    public void updateDeviceCache(boolean force) throws SQLException {
        long lastUpdate = this.devicesLastUpdate.get();
        if ((force || System.currentTimeMillis() - lastUpdate > this.dataRefreshDelay) && this.devicesLastUpdate.compareAndSet(lastUpdate, System.currentTimeMillis())) {
            this.refreshItems();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Device getByUniqueId(String uniqueId) throws SQLException {
        boolean forceUpdate;
        try {
            this.readLock();
            forceUpdate = !this.devicesByUniqueId.containsKey(uniqueId) && !this.config.getBoolean(Keys.DATABASE_IGNORE_UNKNOWN);
        }
        finally {
            this.readUnlock();
        }
        this.updateDeviceCache(forceUpdate);
        try {
            this.readLock();
            Device device = this.devicesByUniqueId.get(uniqueId);
            return device;
        }
        finally {
            this.readUnlock();
        }
    }

    @Override
    public String getDevicePassword(long id, String protocol, String defaultPassword) {
        String password = this.lookupAttributeString(id, "devicePassword", null, false, false);
        if (password != null) {
            return password;
        }
        if (protocol != null && (password = Context.getConfig().getString(Keys.PROTOCOL_DEVICE_PASSWORD.withPrefix(protocol))) != null) {
            return password;
        }
        return defaultPassword;
    }

    public Device getDeviceByPhone(String phone) {
        try {
            this.readLock();
            Device device = this.devicesByPhone.get(phone);
            return device;
        }
        finally {
            this.readUnlock();
        }
    }

    @Override
    public Set<Long> getAllItems() {
        Set<Long> result = super.getAllItems();
        if (result.isEmpty()) {
            try {
                this.updateDeviceCache(true);
            }
            catch (SQLException e) {
                LOGGER.warn("Update device cache error", (Throwable)e);
            }
            result = super.getAllItems();
        }
        return result;
    }

    public Collection<Device> getAllDevices() {
        return this.getItems(this.getAllItems());
    }

    public Set<Long> getAllUserItems(long userId) {
        return Context.getPermissionsManager().getDevicePermissions(userId);
    }

    @Override
    public Set<Long> getUserItems(long userId) {
        if (Context.getPermissionsManager() != null) {
            HashSet<Long> result = new HashSet<Long>();
            for (long deviceId : Context.getPermissionsManager().getDevicePermissions(userId)) {
                Device device = (Device)this.getById(deviceId);
                if (device == null || device.getDisabled()) continue;
                result.add(deviceId);
            }
            return result;
        }
        return new HashSet<Long>();
    }

    public Set<Long> getAllManagedItems(long userId) {
        HashSet<Long> result = new HashSet<Long>(this.getAllUserItems(userId));
        for (long managedUserId : Context.getUsersManager().getUserItems(userId)) {
            result.addAll(this.getAllUserItems(managedUserId));
        }
        return result;
    }

    @Override
    public Set<Long> getManagedItems(long userId) {
        HashSet<Long> result = new HashSet<Long>(this.getUserItems(userId));
        for (long managedUserId : Context.getUsersManager().getUserItems(userId)) {
            result.addAll(this.getUserItems(managedUserId));
        }
        return result;
    }

    private void addByUniqueId(Device device) {
        try {
            this.writeLock();
            if (this.devicesByUniqueId == null) {
                this.devicesByUniqueId = new ConcurrentHashMap<String, Device>();
            }
            this.devicesByUniqueId.put(device.getUniqueId(), device);
        }
        finally {
            this.writeUnlock();
        }
    }

    private void removeByUniqueId(String deviceUniqueId) {
        try {
            this.writeLock();
            if (this.devicesByUniqueId != null) {
                this.devicesByUniqueId.remove(deviceUniqueId);
            }
        }
        finally {
            this.writeUnlock();
        }
    }

    private void addByPhone(Device device) {
        try {
            this.writeLock();
            if (this.devicesByPhone == null) {
                this.devicesByPhone = new ConcurrentHashMap<String, Device>();
            }
            this.devicesByPhone.put(device.getPhone(), device);
        }
        finally {
            this.writeUnlock();
        }
    }

    private void removeByPhone(String phone) {
        if (phone == null || phone.isEmpty()) {
            return;
        }
        try {
            this.writeLock();
            if (this.devicesByPhone != null) {
                this.devicesByPhone.remove(phone);
            }
        }
        finally {
            this.writeUnlock();
        }
    }

    @Override
    protected void addNewItem(Device device) {
        Position lastPosition;
        super.addNewItem(device);
        this.addByUniqueId(device);
        if (device.getPhone() != null && !device.getPhone().isEmpty()) {
            this.addByPhone(device);
        }
        if (Context.getGeofenceManager() != null && (lastPosition = this.getLastPosition(device.getId())) != null) {
            device.setGeofenceIds(Context.getGeofenceManager().getCurrentDeviceGeofences(lastPosition));
        }
    }

    @Override
    protected void updateCachedItem(Device device) {
        Device cachedDevice = (Device)this.getById(device.getId());
        cachedDevice.setName(device.getName());
        cachedDevice.setGroupId(device.getGroupId());
        cachedDevice.setCategory(device.getCategory());
        cachedDevice.setContact(device.getContact());
        cachedDevice.setModel(device.getModel());
        cachedDevice.setDisabled(device.getDisabled());
        cachedDevice.setAttributes(device.getAttributes());
        if (!device.getUniqueId().equals(cachedDevice.getUniqueId())) {
            this.removeByUniqueId(cachedDevice.getUniqueId());
            cachedDevice.setUniqueId(device.getUniqueId());
            this.addByUniqueId(cachedDevice);
        }
        if (device.getPhone() != null && !device.getPhone().isEmpty() && !device.getPhone().equals(cachedDevice.getPhone())) {
            String phone = cachedDevice.getPhone();
            this.removeByPhone(phone);
            cachedDevice.setPhone(device.getPhone());
            this.addByPhone(cachedDevice);
        }
    }

    @Override
    protected void removeCachedItem(long deviceId) {
        Device cachedDevice = (Device)this.getById(deviceId);
        if (cachedDevice != null) {
            String deviceUniqueId = cachedDevice.getUniqueId();
            String phone = cachedDevice.getPhone();
            super.removeCachedItem(deviceId);
            this.removeByUniqueId(deviceUniqueId);
            this.removeByPhone(phone);
        }
        this.positions.remove(deviceId);
    }

    public void updateDeviceStatus(Device device) throws SQLException {
        this.getDataManager().updateDeviceStatus(device);
        Device cachedDevice = (Device)this.getById(device.getId());
        if (cachedDevice != null) {
            cachedDevice.setStatus(device.getStatus());
        }
    }

    private void refreshLastPositions() {
        if (this.getDataManager() != null) {
            try {
                for (Position position : this.getDataManager().getLatestPositions()) {
                    this.positions.put(position.getDeviceId(), position);
                }
            }
            catch (SQLException error) {
                LOGGER.warn("Load latest positions error", (Throwable)error);
            }
        }
    }

    @Override
    public boolean isLatestPosition(Position position) {
        Position lastPosition = this.getLastPosition(position.getDeviceId());
        return lastPosition == null || position.getFixTime().compareTo(lastPosition.getFixTime()) >= 0;
    }

    public void updateLatestPosition(Position position) throws SQLException {
        if (this.isLatestPosition(position)) {
            this.getDataManager().updateLatestPosition(position);
            Device device = (Device)this.getById(position.getDeviceId());
            if (device != null) {
                device.setPositionId(position.getId());
            }
            this.positions.put(position.getDeviceId(), position);
            if (Context.getConnectionManager() != null) {
                Context.getConnectionManager().updatePosition(position);
            }
        }
    }

    @Override
    public Position getLastPosition(long deviceId) {
        return this.positions.get(deviceId);
    }

    public Collection<Position> getInitialState(long userId) {
        LinkedList<Position> result = new LinkedList<Position>();
        if (Context.getPermissionsManager() != null) {
            for (long deviceId : Context.getPermissionsManager().getUserAdmin(userId) ? this.getAllUserItems(userId) : this.getUserItems(userId)) {
                if (!this.positions.containsKey(deviceId)) continue;
                result.add(this.positions.get(deviceId));
            }
        }
        return result;
    }

    @Override
    public boolean lookupAttributeBoolean(long deviceId, String attributeName, boolean defaultValue, boolean lookupServer, boolean lookupConfig) {
        Object result = this.lookupAttribute(deviceId, attributeName, lookupServer, lookupConfig);
        if (result != null) {
            return result instanceof String ? Boolean.parseBoolean((String)result) : (Boolean)result;
        }
        return defaultValue;
    }

    @Override
    public String lookupAttributeString(long deviceId, String attributeName, String defaultValue, boolean lookupServer, boolean lookupConfig) {
        Object result = this.lookupAttribute(deviceId, attributeName, lookupServer, lookupConfig);
        return result != null ? (String)result : defaultValue;
    }

    @Override
    public int lookupAttributeInteger(long deviceId, String attributeName, int defaultValue, boolean lookupServer, boolean lookupConfig) {
        Object result = this.lookupAttribute(deviceId, attributeName, lookupServer, lookupConfig);
        if (result != null) {
            return result instanceof String ? Integer.parseInt((String)result) : ((Number)result).intValue();
        }
        return defaultValue;
    }

    @Override
    public long lookupAttributeLong(long deviceId, String attributeName, long defaultValue, boolean lookupServer, boolean lookupConfig) {
        Object result = this.lookupAttribute(deviceId, attributeName, lookupServer, lookupConfig);
        if (result != null) {
            return result instanceof String ? Long.parseLong((String)result) : ((Number)result).longValue();
        }
        return defaultValue;
    }

    @Override
    public double lookupAttributeDouble(long deviceId, String attributeName, double defaultValue, boolean lookupServer, boolean lookupConfig) {
        Object result = this.lookupAttribute(deviceId, attributeName, lookupServer, lookupConfig);
        if (result != null) {
            return result instanceof String ? Double.parseDouble((String)result) : ((Number)result).doubleValue();
        }
        return defaultValue;
    }

    private Object lookupAttribute(long deviceId, String attributeName, boolean lookupServer, boolean lookupConfig) {
        Object result = null;
        Device device = (Device)this.getById(deviceId);
        if (device != null) {
            result = device.getAttributes().get(attributeName);
            if (result == null) {
                long groupId = device.getGroupId();
                while (groupId != 0L) {
                    Group group = (Group)Context.getGroupsManager().getById(groupId);
                    if (group != null) {
                        result = group.getAttributes().get(attributeName);
                        if (result != null) break;
                        groupId = group.getGroupId();
                        continue;
                    }
                    groupId = 0L;
                }
            }
            if (result == null && lookupServer) {
                Server server = Context.getPermissionsManager().getServer();
                result = server.getAttributes().get(attributeName);
            }
            if (result == null && lookupConfig) {
                result = Context.getConfig().getString(attributeName);
            }
        }
        return result;
    }

    public void resetDeviceAccumulators(DeviceAccumulators deviceAccumulators) throws SQLException {
        Position last = this.positions.get(deviceAccumulators.getDeviceId());
        if (last != null) {
            if (deviceAccumulators.getTotalDistance() != null) {
                last.getAttributes().put("totalDistance", deviceAccumulators.getTotalDistance());
            }
            if (deviceAccumulators.getHours() != null) {
                last.getAttributes().put("hours", deviceAccumulators.getHours());
            }
        } else {
            throw new IllegalArgumentException();
        }
        this.getDataManager().addObject(last);
        this.updateLatestPosition(last);
    }

    public DeviceState getDeviceState(long deviceId) {
        DeviceState deviceState = this.deviceStates.get(deviceId);
        if (deviceState == null) {
            deviceState = new DeviceState();
            this.deviceStates.put(deviceId, deviceState);
        }
        return deviceState;
    }

    public void setDeviceState(long deviceId, DeviceState deviceState) {
        this.deviceStates.put(deviceId, deviceState);
    }
}

