/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import React, { useCallback } from 'react';
import { EuiCodeEditor } from '@elastic/eui';
import compactStringify from 'json-stringify-pretty-compact'; // @ts-ignore

import hjson from 'hjson';
import 'brace/mode/hjson';
import { i18n } from '@kbn/i18n';
import { getNotifications } from '../services';
import { VegaHelpMenu } from './vega_help_menu';
import { VegaActionsMenu } from './vega_actions_menu';
var aceOptions = {
  maxLines: Infinity,
  highlightActiveLine: false,
  showPrintMargin: false,
  tabSize: 2,
  useSoftTabs: true,
  wrap: true
};
var hjsonStringifyOptions = {
  bracesSameLine: true,
  keepWsc: true
};

function format(value, stringify, options) {
  try {
    var spec = hjson.parse(value, {
      legacyRoot: false,
      keepWsc: true
    });
    return stringify(spec, options);
  } catch (err) {
    // This is a common case - user tries to format an invalid HJSON text
    getNotifications().toasts.addError(err, {
      title: i18n.translate('visTypeVega.editor.formatError', {
        defaultMessage: 'Error formatting spec'
      })
    });
    return value;
  }
}

function VegaVisEditor(_ref) {
  var stateParams = _ref.stateParams,
      setValue = _ref.setValue;
  var onChange = useCallback(function (value) {
    setValue('spec', value);
  }, [setValue]);
  var formatJson = useCallback(function () {
    return setValue('spec', format(stateParams.spec, compactStringify));
  }, [setValue, stateParams.spec]);
  var formatHJson = useCallback(function () {
    return setValue('spec', format(stateParams.spec, hjson.stringify, hjsonStringifyOptions));
  }, [setValue, stateParams.spec]);
  return /*#__PURE__*/React.createElement("div", {
    className: "vgaEditor"
  }, /*#__PURE__*/React.createElement(EuiCodeEditor, {
    "data-test-subj": "vega-editor",
    mode: "hjson",
    theme: "textmate",
    width: "100%",
    height: "auto",
    onChange: onChange,
    value: stateParams.spec,
    setOptions: aceOptions
  }), /*#__PURE__*/React.createElement("div", {
    className: "vgaEditor__aceEditorActions"
  }, /*#__PURE__*/React.createElement(VegaHelpMenu, null), /*#__PURE__*/React.createElement(VegaActionsMenu, {
    formatHJson: formatHJson,
    formatJson: formatJson
  })));
}

export { VegaVisEditor };