"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildDataTelemetryPayload = buildDataTelemetryPayload;
exports.getDataTelemetry = getDataTelemetry;

var _constants = require("./constants");

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
function findMatchingDescriptors({
  name,
  shipper,
  datasetName,
  datasetType
}) {
  // If we already have the data from the indices' mappings...
  if ([shipper, datasetName, datasetType].some(Boolean)) {
    return [{ ...(shipper && {
        shipper
      }),
      ...(datasetName && {
        datasetName
      }),
      ...(datasetType && {
        datasetType
      })
    } // Using casting here because TS doesn't infer at least one exists from the if clause
    ];
  } // Otherwise, try with the list of known index patterns


  return _constants.DATA_DATASETS_INDEX_PATTERNS_UNIQUE.filter(({
    pattern
  }) => {
    if (!pattern.startsWith('.') && name.startsWith('.')) {
      // avoid system indices caught by very fuzzy index patterns (i.e.: *log* would catch `.kibana-log-...`)
      return false;
    }

    return new RegExp(`^${pattern.replace(/\./g, '\\.').replace(/\*/g, '.*')}$`).test(name);
  });
}

function increaseCounters(previousValue = {
  index_count: 0
}, {
  isECS,
  docCount,
  sizeInBytes
}) {
  return { ...previousValue,
    index_count: previousValue.index_count + 1,
    ...(typeof isECS === 'boolean' ? {
      ecs_index_count: (previousValue.ecs_index_count || 0) + (isECS ? 1 : 0)
    } : {}),
    ...(typeof docCount === 'number' ? {
      doc_count: (previousValue.doc_count || 0) + docCount
    } : {}),
    ...(typeof sizeInBytes === 'number' ? {
      size_in_bytes: (previousValue.size_in_bytes || 0) + sizeInBytes
    } : {})
  };
}

function buildDataTelemetryPayload(indices) {
  const startingDotPatternsUntilTheFirstAsterisk = _constants.DATA_DATASETS_INDEX_PATTERNS_UNIQUE.map(({
    pattern
  }) => pattern.replace(/^\.(.+)\*.*$/g, '.$1')).filter(Boolean); // Filter out the system indices unless they are required by the patterns


  const indexCandidates = indices.filter(({
    name
  }) => !(name.startsWith('.') && !startingDotPatternsUntilTheFirstAsterisk.find(pattern => name.startsWith(pattern))));
  const acc = new Map();

  for (const indexCandidate of indexCandidates) {
    const matchingDescriptors = findMatchingDescriptors(indexCandidate);

    for (const {
      datasetName,
      datasetType,
      shipper,
      patternName
    } of matchingDescriptors) {
      const key = `${datasetName}-${datasetType}-${shipper}-${patternName}`;
      acc.set(key, { ...((datasetName || datasetType) && {
          dataset: {
            name: datasetName,
            type: datasetType
          }
        }),
        ...(shipper && {
          shipper
        }),
        ...(patternName && {
          pattern_name: patternName
        }),
        ...increaseCounters(acc.get(key), indexCandidate)
      });
    }
  }

  return [...acc.values()];
}

async function getDataTelemetry(callCluster) {
  try {
    const index = [..._constants.DATA_DATASETS_INDEX_PATTERNS_UNIQUE.map(({
      pattern
    }) => pattern), '*-*-*-*' // Include new indexing strategy indices {type}-{dataset}-{namespace}-{rollover_counter}
    ];
    const [indexMappings, indexStats] = await Promise.all([// GET */_mapping?filter_path=*.mappings._meta.beat,*.mappings.properties.ecs.properties.version.type,*.mappings.properties.dataset.properties.type.value,*.mappings.properties.dataset.properties.name.value
    callCluster('indices.getMapping', {
      index: '*',
      // Request all indices because filter_path already filters out the indices without any of those fields
      filterPath: [// _meta.beat tells the shipper
      '*.mappings._meta.beat', // Does it have `ecs.version` in the mappings? => It follows the ECS conventions
      '*.mappings.properties.ecs.properties.version.type' // Disable the fields below because they are still pending to be confirmed:
      // https://github.com/elastic/ecs/pull/845
      // TODO: Re-enable when the final fields are confirmed
      // // If `dataset.type` is a `constant_keyword`, it can be reported as a type
      // '*.mappings.properties.dataset.properties.type.value',
      // // If `dataset.name` is a `constant_keyword`, it can be reported as the dataset
      // '*.mappings.properties.dataset.properties.name.value',
      ]
    }), // GET <index>/_stats/docs,store?level=indices&filter_path=indices.*.total
    callCluster('indices.stats', {
      index,
      level: 'indices',
      metric: ['docs', 'store'],
      filterPath: ['indices.*.total']
    })]);
    const indexNames = Object.keys({ ...indexMappings,
      ...(indexStats === null || indexStats === void 0 ? void 0 : indexStats.indices)
    });
    const indices = indexNames.map(name => {
      var _indexMappings$name, _indexMappings$name$m, _indexMappings$name$m2, _indexMappings$name$m3, _indexMappings$name2, _indexMappings$name2$, _indexMappings$name2$2, _indexMappings$name3, _indexMappings$name3$, _indexMappings$name3$2, _indexMappings$name3$3, _indexMappings$name4, _indexMappings$name4$, _indexMappings$name4$2, _indexMappings$name4$3;

      const isECS = !!((_indexMappings$name = indexMappings[name]) === null || _indexMappings$name === void 0 ? void 0 : (_indexMappings$name$m = _indexMappings$name.mappings) === null || _indexMappings$name$m === void 0 ? void 0 : (_indexMappings$name$m2 = _indexMappings$name$m.properties.ecs) === null || _indexMappings$name$m2 === void 0 ? void 0 : (_indexMappings$name$m3 = _indexMappings$name$m2.properties.version) === null || _indexMappings$name$m3 === void 0 ? void 0 : _indexMappings$name$m3.type);
      const shipper = (_indexMappings$name2 = indexMappings[name]) === null || _indexMappings$name2 === void 0 ? void 0 : (_indexMappings$name2$ = _indexMappings$name2.mappings) === null || _indexMappings$name2$ === void 0 ? void 0 : (_indexMappings$name2$2 = _indexMappings$name2$._meta) === null || _indexMappings$name2$2 === void 0 ? void 0 : _indexMappings$name2$2.beat;
      const datasetName = (_indexMappings$name3 = indexMappings[name]) === null || _indexMappings$name3 === void 0 ? void 0 : (_indexMappings$name3$ = _indexMappings$name3.mappings) === null || _indexMappings$name3$ === void 0 ? void 0 : (_indexMappings$name3$2 = _indexMappings$name3$.properties.dataset) === null || _indexMappings$name3$2 === void 0 ? void 0 : (_indexMappings$name3$3 = _indexMappings$name3$2.properties.name) === null || _indexMappings$name3$3 === void 0 ? void 0 : _indexMappings$name3$3.value;
      const datasetType = (_indexMappings$name4 = indexMappings[name]) === null || _indexMappings$name4 === void 0 ? void 0 : (_indexMappings$name4$ = _indexMappings$name4.mappings) === null || _indexMappings$name4$ === void 0 ? void 0 : (_indexMappings$name4$2 = _indexMappings$name4$.properties.dataset) === null || _indexMappings$name4$2 === void 0 ? void 0 : (_indexMappings$name4$3 = _indexMappings$name4$2.properties.type) === null || _indexMappings$name4$3 === void 0 ? void 0 : _indexMappings$name4$3.value;
      const stats = ((indexStats === null || indexStats === void 0 ? void 0 : indexStats.indices) || {})[name];

      if (stats) {
        var _stats$total, _stats$total$docs, _stats$total2, _stats$total2$store;

        return {
          name,
          datasetName,
          datasetType,
          shipper,
          isECS,
          docCount: (_stats$total = stats.total) === null || _stats$total === void 0 ? void 0 : (_stats$total$docs = _stats$total.docs) === null || _stats$total$docs === void 0 ? void 0 : _stats$total$docs.count,
          sizeInBytes: (_stats$total2 = stats.total) === null || _stats$total2 === void 0 ? void 0 : (_stats$total2$store = _stats$total2.store) === null || _stats$total2$store === void 0 ? void 0 : _stats$total2$store.size_in_bytes
        };
      }

      return {
        name,
        datasetName,
        datasetType,
        shipper,
        isECS
      };
    });
    return buildDataTelemetryPayload(indices);
  } catch (e) {
    return [];
  }
}