"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createBundlesRoute = createBundlesRoute;

var _path = require("path");

var UiSharedDeps = _interopRequireWildcard(require("@kbn/ui-shared-deps"));

var _dynamic_asset_response = require("./dynamic_asset_response");

var _file_hash_cache = require("./file_hash_cache");

var _np_ui_plugin_public_dirs = require("../np_ui_plugin_public_dirs");

var _utils = require("../../core/server/utils");

function _getRequireWildcardCache() { if (typeof WeakMap !== "function") return null; var cache = new WeakMap(); _getRequireWildcardCache = function () { return cache; }; return cache; }

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/**
 *  Creates the routes that serves files from `bundlesPath` or from
 *  `dllBundlesPath` (if they are dll bundle's related files). If the
 *  file is js or css then it is searched for instances of
 *  PUBLIC_PATH_PLACEHOLDER and replaces them with `publicPath`.
 *
 *  @param {Object} options
 *  @property {Array<{id,path}>} options.npUiPluginPublicDirs array of ids and paths that should be served for new platform plugins
 *  @property {string} options.regularBundlesPath
 *  @property {string} options.dllBundlesPath
 *  @property {string} options.basePublicPath
 *
 *  @return Array.of({Hapi.Route})
 */
function createBundlesRoute({
  regularBundlesPath,
  dllBundlesPath,
  basePublicPath,
  builtCssPath,
  npUiPluginPublicDirs = [],
  buildHash,
  isDist = false
}) {
  // rather than calculate the fileHash on every request, we
  // provide a cache object to `resolveDynamicAssetResponse()` that
  // will store the 100 most recently used hashes.
  const fileHashCache = new _file_hash_cache.FileHashCache();
  (0, _np_ui_plugin_public_dirs.assertIsNpUiPluginPublicDirs)(npUiPluginPublicDirs);

  if (typeof regularBundlesPath !== 'string' || !(0, _path.isAbsolute)(regularBundlesPath)) {
    throw new TypeError('regularBundlesPath must be an absolute path to the directory containing the regular bundles');
  }

  if (typeof dllBundlesPath !== 'string' || !(0, _path.isAbsolute)(dllBundlesPath)) {
    throw new TypeError('dllBundlesPath must be an absolute path to the directory containing the dll bundles');
  }

  if (typeof basePublicPath !== 'string') {
    throw new TypeError('basePublicPath must be a string');
  }

  if (!basePublicPath.match(/(^$|^\/.*[^\/]$)/)) {
    throw new TypeError('basePublicPath must be empty OR start and not end with a /');
  }

  return [buildRouteForBundles({
    publicPath: `${basePublicPath}/${buildHash}/bundles/kbn-ui-shared-deps/`,
    routePath: `/${buildHash}/bundles/kbn-ui-shared-deps/`,
    bundlesPath: UiSharedDeps.distDir,
    fileHashCache,
    replacePublicPath: false,
    isDist
  }), ...npUiPluginPublicDirs.map(({
    id,
    path
  }) => buildRouteForBundles({
    publicPath: `${basePublicPath}/${buildHash}/bundles/plugin/${id}/`,
    routePath: `/${buildHash}/bundles/plugin/${id}/`,
    bundlesPath: path,
    fileHashCache,
    replacePublicPath: false,
    isDist
  })), buildRouteForBundles({
    publicPath: `${basePublicPath}/${buildHash}/bundles/core/`,
    routePath: `/${buildHash}/bundles/core/`,
    bundlesPath: (0, _utils.fromRoot)((0, _path.join)('src', 'core', 'target', 'public')),
    fileHashCache,
    replacePublicPath: false,
    isDist
  }), buildRouteForBundles({
    publicPath: `${basePublicPath}/${buildHash}/bundles/`,
    routePath: `/${buildHash}/bundles/`,
    bundlesPath: regularBundlesPath,
    fileHashCache,
    isDist
  }), buildRouteForBundles({
    publicPath: `${basePublicPath}/${buildHash}/built_assets/dlls/`,
    routePath: `/${buildHash}/built_assets/dlls/`,
    bundlesPath: dllBundlesPath,
    fileHashCache,
    isDist
  }), buildRouteForBundles({
    publicPath: `${basePublicPath}/`,
    routePath: `/${buildHash}/built_assets/css/`,
    bundlesPath: builtCssPath,
    fileHashCache,
    isDist
  })];
}

function buildRouteForBundles({
  publicPath,
  routePath,
  bundlesPath,
  fileHashCache,
  replacePublicPath = true,
  isDist
}) {
  return {
    method: 'GET',
    path: `${routePath}{path*}`,
    config: {
      auth: false,
      ext: {
        onPreHandler: {
          method(request, h) {
            const ext = (0, _path.extname)(request.params.path);

            if (ext !== '.js' && ext !== '.css') {
              return h.continue;
            }

            return (0, _dynamic_asset_response.createDynamicAssetResponse)({
              request,
              h,
              bundlesPath,
              fileHashCache,
              publicPath,
              replacePublicPath,
              isDist
            });
          }

        }
      }
    },
    handler: {
      directory: {
        path: bundlesPath,
        listing: false,
        lookupCompressed: true
      }
    }
  };
}