/*
 * Decompiled with CFR 0.152.
 */
package ghidra.util.classfinder;

import generic.jar.ResourceFile;
import ghidra.framework.Application;
import ghidra.util.Msg;
import ghidra.util.SystemUtilities;
import ghidra.util.classfinder.ClassFilter;
import ghidra.util.classfinder.ClassFinder;
import ghidra.util.datastruct.WeakDataStructureFactory;
import ghidra.util.datastruct.WeakSet;
import ghidra.util.exception.AssertException;
import ghidra.util.exception.CancelledException;
import ghidra.util.task.TaskMonitor;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.nio.file.InvalidPathException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.swing.event.ChangeListener;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import utilities.util.FileUtilities;

public class ClassSearcher {
    public static final String SEARCH_ALL_JARS_PROPERTY = "class.searcher.search.all.jars";
    private static final String SEARCH_ALL_JARS_PROPERTY_VALUE = System.getProperty("class.searcher.search.all.jars", Boolean.FALSE.toString());
    static final boolean SEARCH_ALL_JARS = Boolean.parseBoolean(SEARCH_ALL_JARS_PROPERTY_VALUE);
    static final Logger log = LogManager.getLogger(ClassSearcher.class);
    private static ClassFinder searcher;
    private static Set<Class<?>> extensionPoints;
    private static WeakSet<ChangeListener> listenerList;
    private static Pattern extensionPointSuffixPattern;
    private static volatile boolean hasSearched;
    private static volatile boolean isSearching;
    private static final ClassFilter DO_NOTHING_FILTER;

    private ClassSearcher() {
    }

    public static <T> Set<Class<? extends T>> getClasses(Class<T> c) {
        return ClassSearcher.getClasses(c, null);
    }

    public static <T> Set<Class<? extends T>> getClasses(Class<T> c, Predicate<Class<? extends T>> classFilter) {
        if (isSearching) {
            throw new IllegalStateException("Cannot call the getClasses() while the ClassSearcher is searching!");
        }
        HashSet<Class<T>> set = new HashSet<Class<T>>();
        if (extensionPoints == null) {
            return set;
        }
        for (Class<?> extensionPoint : extensionPoints) {
            if (!c.isAssignableFrom(extensionPoint) || classFilter != null && !classFilter.test(extensionPoint)) continue;
            set.add(extensionPoint);
        }
        return set;
    }

    public static <T> Set<T> getInstances(Class<T> c) {
        return ClassSearcher.getInstances(c, DO_NOTHING_FILTER);
    }

    public static <T> Set<T> getInstances(Class<T> c, ClassFilter filter) {
        Set<Class<T>> classes = ClassSearcher.getClasses(c);
        HashSet<T> instances = new HashSet<T>();
        for (Class<T> clazz : classes) {
            if (!filter.accepts(clazz)) continue;
            try {
                Constructor<T> constructor = clazz.getConstructor(null);
                T t = constructor.newInstance(null);
                instances.add(t);
            }
            catch (InstantiationException e) {
                Msg.showError(ClassSearcher.class, null, (String)"Error Instantiating Extension Point", (Object)("Error creating class " + clazz.getSimpleName() + " for extension " + c.getName() + ".  Discovered class is not a concrete implementation or does not have a nullary constructor!"), (Throwable)e);
            }
            catch (IllegalAccessException e) {
                Msg.showError(ClassSearcher.class, null, (String)"Error Instantiating Extension Point", (Object)("Error creating class " + clazz.getSimpleName() + " for extension " + c.getName() + ".  Discovered class does not have a public, default constructor!"), (Throwable)e);
            }
            catch (SecurityException e) {
                String message = "Error creating class " + clazz.getSimpleName() + " for extension " + c.getName() + ".  Security Exception!";
                Msg.showError(ClassSearcher.class, null, (String)"Error Instantiating Extension Point", (Object)message, (Throwable)e);
                throw new AssertException(message, (Throwable)e);
            }
            catch (Exception e) {
                Msg.showError(ClassSearcher.class, null, (String)"Error Creating Extension Point", (Object)("Error creating class " + clazz.getSimpleName() + " when creating extension points for " + c.getName()), (Throwable)e);
            }
        }
        return instances;
    }

    public static void addChangeListener(ChangeListener l) {
        listenerList.add(l);
    }

    public static void removeChangeListener(ChangeListener l) {
        listenerList.remove(l);
    }

    public static void search(boolean forceRefresh, TaskMonitor monitor) throws CancelledException {
        if (hasSearched && !forceRefresh) {
            log.trace("Already searched for classes: using cached results");
            return;
        }
        if (Application.inSingleJarMode()) {
            log.trace("Single Jar Mode: using extensions from the jar file");
            ClassSearcher.loadExtensionClassesFromJar();
            return;
        }
        isSearching = true;
        ClassSearcher.loadExtensionPointSuffixes();
        extensionPoints = null;
        long t = new Date().getTime();
        log.trace("Searching for classes...");
        List<String> searchPaths = ClassSearcher.gatherSearchPaths();
        searcher = new ClassFinder(searchPaths, monitor);
        monitor.setMessage("Loading classes...");
        extensionPoints = searcher.getClasses(monitor);
        log.trace("Found extension classes: " + extensionPoints);
        if (extensionPoints.isEmpty()) {
            throw new AssertException("Unable to location extension points!");
        }
        hasSearched = true;
        isSearching = false;
        SystemUtilities.runSwingNow(() -> ClassSearcher.fireClassListChanged());
        t = new Date().getTime() - t;
        String finishedMessage = "Class search complete (" + t + " ms)";
        monitor.setMessage(finishedMessage);
        log.info(finishedMessage);
    }

    private static List<String> gatherSearchPaths() {
        String cp = System.getProperty("java.class.path");
        StringTokenizer st = new StringTokenizer(cp, File.pathSeparator);
        ArrayList<String> rawPaths = new ArrayList<String>();
        while (st.hasMoreTokens()) {
            rawPaths.add(st.nextToken());
        }
        List<String> canonicalPaths = ClassSearcher.canonicalizePaths(rawPaths);
        return canonicalPaths;
    }

    private static List<String> canonicalizePaths(Collection<String> paths) {
        List<String> canonical = paths.stream().map(path -> {
            String normalized = ClassSearcher.normalize(path);
            return normalized;
        }).collect(Collectors.toList());
        return canonical;
    }

    private static String normalize(String path) {
        try {
            Path p = Paths.get(path, new String[0]);
            Path normalized = p.normalize();
            Path absolutePath = normalized.toAbsolutePath();
            return absolutePath.toString();
        }
        catch (InvalidPathException e) {
            Msg.trace(ClassSearcher.class, (Object)("Invalid path '" + path + "'"), (Throwable)e);
            return path;
        }
    }

    private static void loadExtensionClassesFromJar() {
        ResourceFile appRoot = Application.getApplicationRootDirectory();
        ResourceFile extensionClassesFile = new ResourceFile(appRoot, "EXTENSION_POINT_CLASSES");
        try {
            List classNames = FileUtilities.getLines((ResourceFile)extensionClassesFile);
            HashSet extensionClasses = new HashSet();
            for (String className : classNames) {
                try {
                    Class<?> clazz = Class.forName(className);
                    extensionClasses.add(clazz);
                }
                catch (ClassNotFoundException e) {
                    Msg.warn(ClassSearcher.class, (Object)("Can't load extension point: " + className));
                }
            }
            extensionPoints = Collections.unmodifiableSet(extensionClasses);
        }
        catch (IOException e) {
            throw new AssertException("Got unexpected IOException ", (Throwable)e);
        }
    }

    private static void loadExtensionPointSuffixes() {
        HashSet extensionPointSuffixes = new HashSet();
        Collection<ResourceFile> moduleRootDirectories = Application.getModuleRootDirectories();
        if (moduleRootDirectories.isEmpty()) {
            throw new AssertException("Could not find modules for Class Searcher!");
        }
        log.trace("Scanning module root directories: " + moduleRootDirectories);
        for (ResourceFile moduleRoot : moduleRootDirectories) {
            ResourceFile file = new ResourceFile(moduleRoot, "data/ExtensionPoint.manifest");
            if (!file.exists()) continue;
            extensionPointSuffixes.addAll(FileUtilities.getLinesQuietly((ResourceFile)file));
        }
        StringBuilder buffy = new StringBuilder(".*(");
        String between = "";
        for (String suffix : extensionPointSuffixes) {
            if ((suffix = suffix.trim()).isEmpty()) continue;
            buffy.append(between);
            buffy.append(suffix);
            between = "|";
        }
        buffy.append(')');
        extensionPointSuffixPattern = Pattern.compile(buffy.toString());
        log.trace("Using extension point pattern: " + extensionPointSuffixPattern);
    }

    static boolean isExtensionPointName(String name) {
        int innerClassIndex;
        if (name.indexOf("Test$") > 0 || name.endsWith("Test")) {
            return false;
        }
        int packageIndex = name.lastIndexOf(46);
        int maximumIndex = StrictMath.max(packageIndex, innerClassIndex = name.lastIndexOf(36));
        if (maximumIndex > 0) {
            name = name.substring(maximumIndex + 1);
        }
        return extensionPointSuffixPattern.matcher(name).matches();
    }

    private static void fireClassListChanged() {
        for (ChangeListener listener : listenerList) {
            try {
                listener.stateChanged(null);
            }
            catch (Throwable t) {
                Msg.showError(ClassSearcher.class, null, (String)"Exception", (Object)"Error in listener for class list changed", (Throwable)t);
            }
        }
    }

    static {
        listenerList = WeakDataStructureFactory.createCopyOnReadWeakSet();
        DO_NOTHING_FILTER = c -> true;
    }
}

