function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

function _defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } }

function _createClass(Constructor, protoProps, staticProps) { if (protoProps) _defineProperties(Constructor.prototype, protoProps); if (staticProps) _defineProperties(Constructor, staticProps); return Constructor; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/**
 * The HashedItemStore associates JSON objects with states in browser history and persists these
 * objects in sessionStorage. We persist them so that when a tab is closed and re-opened, we can
 * retain access to the state objects referenced by the browser history.
 *
 * Because there is a limit on how much data we can put into sessionStorage, the HashedItemStore
 * will attempt to remove old items from storage once that limit is reached.
 *
 * -------------------------------------------------------------------------------------------------
 *
 * Consideration 1: We can't (easily) mirror the browser history
 *
 * If we use letters to indicate a unique state object, and numbers to represent the same state
 * occurring again (due to action by the user), a history could look like this:
 *
 * Old < - - - - - - - - > New
 * A1 | B1 | C1 | A2 | D1 | E1
 *
 * If the user navigates back to C1 and starts to create new states, persisted history states will
 * become inaccessible:
 *
 * Old < - - - - - - - - - - -> New
 * A1 | B1 | C1 | F1 | G1 | H1 | I1  (new history states)
 *                A2 | D1 | E1       (inaccessible persisted history states)
 *
 * Theoretically, we could build a mirror of the browser history. When the onpopstate event is
 * dispatched, we could determine whether we have gone back or forward in history. Then, when
 * a new state is persisted, we could delete all of the persisted items which are no longer
 * accessible. (Note that this would require reference-counting so that A isn't removed while D and
 * E are, since A would still have a remaining reference from A1).
 *
 * However, the History API doesn't allow us to read from the history beyond the current state. This
 * means that if a session is restored, we can't rebuild this browser history mirror.
 *
 * Due to this imperfect implementation, HashedItemStore ignores the possibility of inaccessible
 * history states. In the future, we could implement this history mirror and persist it in
 * sessionStorage too. Then, when restoring a session, we can just retrieve it from sessionStorage.
 *
 * -------------------------------------------------------------------------------------------------
 *
 * Consideration 2: We can't tell when we've hit the browser history limit
 *
 * Because some of our persisted history states may no longer be referenced by the browser history,
 * and we have no way of knowing which ones, we have no way of knowing whether we've persisted a
 * number of accessible states beyond the browser history length limit.
 *
 * More fundamentally, the browser history length limit is a browser implementation detail, so it
 * can change from browser to browser, or over time. Respecting this limit would introduce a lot of
 * (unnecessary?) complexity.
 *
 * For these reasons, HashedItemStore doesn't concern itself with this constraint.
 */
import { pull, sortBy } from 'lodash';
export var HashedItemStore = /*#__PURE__*/function () {
  /**
   * HashedItemStore uses objects called indexed items to refer to items that have been persisted
   * in storage. An indexed item is shaped {hash, touched}. The touched date is when the item
   * was last referenced by the browser history.
   */
  function HashedItemStore(storage) {
    _classCallCheck(this, HashedItemStore);

    _defineProperty(this, "storage", void 0);

    _defineProperty(this, "ensuredSorting", false);

    this.storage = storage;
  }

  _createClass(HashedItemStore, [{
    key: "setItem",
    value: function setItem(hash, item) {
      var isItemPersisted = this.persistItem(hash, item);

      if (isItemPersisted) {
        this.touchHash(hash);
      }

      return isItemPersisted;
    }
  }, {
    key: "getItem",
    value: function getItem(hash) {
      var item = this.storage.getItem(hash);

      if (item !== null) {
        this.touchHash(hash);
      }

      return item;
    }
  }, {
    key: "removeItem",
    value: function removeItem(hash) {
      var indexedItems = this.getIndexedItems();
      var itemToRemove = this.storage.getItem(hash);
      var indexedItemToRemove = this.getIndexedItem(hash, indexedItems);

      if (indexedItemToRemove) {
        pull(indexedItems, indexedItemToRemove);
        this.setIndexedItems(indexedItems);
      }

      if (itemToRemove) {
        this.storage.removeItem(hash);
      }

      return itemToRemove || null;
    }
  }, {
    key: "clear",
    value: function clear() {
      var _this = this;

      var indexedItems = this.getIndexedItems();
      indexedItems.forEach(function (_ref) {
        var hash = _ref.hash;

        _this.storage.removeItem(hash);
      });
      this.setIndexedItems([]);
    } // Store indexed items in descending order by touched (oldest first, newest last). We'll use
    // this to remove older items when we run out of storage space.

  }, {
    key: "getIndexedItems",
    value: function getIndexedItems() {
      // Restore a previously persisted index
      var persistedItemIndex = this.storage.getItem(HashedItemStore.PERSISTED_INDEX_KEY);
      var items = persistedItemIndex ? JSON.parse(persistedItemIndex) || [] : []; // ensure sorting once, as sorting all indexed items on each get is a performance hit

      if (!this.ensuredSorting) {
        items = sortBy(items, 'touched');
        this.setIndexedItems(items);
        this.ensuredSorting = true;
      }

      return items;
    }
  }, {
    key: "setIndexedItems",
    value: function setIndexedItems(items) {
      this.storage.setItem(HashedItemStore.PERSISTED_INDEX_KEY, JSON.stringify(items));
    }
  }, {
    key: "getIndexedItem",
    value: function getIndexedItem(hash) {
      var indexedItems = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : this.getIndexedItems();
      return indexedItems.find(function (indexedItem) {
        return indexedItem.hash === hash;
      });
    }
  }, {
    key: "persistItem",
    value: function persistItem(hash, item) {
      try {
        this.storage.setItem(hash, item);
        return true;
      } catch (e) {
        // If there was an error then we need to make some space for the item.
        if (this.getIndexedItems().length === 0) {
          // If there's nothing left to remove, then we've run out of space and we're trying to
          // persist too large an item.
          return false;
        } // We need to try to make some space for the item by removing older items (i.e. items that
        // haven't been accessed recently).


        this.removeOldestItem(); // Try to persist again.

        return this.persistItem(hash, item);
      }
    }
  }, {
    key: "removeOldestItem",
    value: function removeOldestItem() {
      var indexedItems = this.getIndexedItems();
      var oldestIndexedItem = indexedItems.shift();

      if (oldestIndexedItem) {
        // Remove oldest item from storage.
        this.storage.removeItem(oldestIndexedItem.hash);
        this.setIndexedItems(indexedItems);
      }
    }
  }, {
    key: "touchHash",
    value: function touchHash(hash) {
      var indexedItems = this.getIndexedItems(); // Touching a hash indicates that it's been used recently, so it won't be the first in line
      // when we remove items to free up storage space.
      // either get or create an indexedItem

      var indexedItem = this.getIndexedItem(hash, indexedItems) || {
        hash: hash
      }; // set/update the touched time to now so that it's the "newest" item in the index

      indexedItem.touched = Date.now(); // ensure that the item is last in the index

      pull(indexedItems, indexedItem);
      indexedItems.push(indexedItem); // Regardless of whether this is a new or updated item, we need to persist the index.

      this.setIndexedItems(indexedItems);
    }
  }]);

  return HashedItemStore;
}();

_defineProperty(HashedItemStore, "PERSISTED_INDEX_KEY", 'kbn.hashedItemsIndex.v1');