function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); if (enumerableOnly) symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; }); keys.push.apply(keys, symbols); } return keys; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; if (i % 2) { ownKeys(Object(source), true).forEach(function (key) { _defineProperty(target, key, source[key]); }); } else if (Object.getOwnPropertyDescriptors) { Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)); } else { ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import _ from 'lodash';
import { i18n } from '@kbn/i18n';
import { MetricAggType } from './metric_agg_type';
import { METRIC_TYPES } from './metric_agg_types';
import { KBN_FIELD_TYPES } from '../../../../common';

var isNumericFieldSelected = function isNumericFieldSelected(agg) {
  var field = agg.getParam('field');
  return field && field.type && field.type === KBN_FIELD_TYPES.NUMBER;
};

export var getTopHitMetricAgg = function getTopHitMetricAgg() {
  return new MetricAggType({
    name: METRIC_TYPES.TOP_HITS,
    title: i18n.translate('data.search.aggs.metrics.topHitTitle', {
      defaultMessage: 'Top Hit'
    }),
    makeLabel: function makeLabel(aggConfig) {
      var lastPrefixLabel = i18n.translate('data.search.aggs.metrics.topHit.lastPrefixLabel', {
        defaultMessage: 'Last'
      });
      var firstPrefixLabel = i18n.translate('data.search.aggs.metrics.topHit.firstPrefixLabel', {
        defaultMessage: 'First'
      });
      var prefix = aggConfig.getParam('sortOrder').value === 'desc' ? lastPrefixLabel : firstPrefixLabel;
      var size = aggConfig.getParam('size');

      if (size !== 1) {
        prefix += " ".concat(size);
      }

      var field = aggConfig.getParam('field');
      return "".concat(prefix, " ").concat(field ? field.displayName : '');
    },
    params: [{
      name: 'field',
      type: 'field',
      onlyAggregatable: false,
      filterFieldTypes: Object.values(KBN_FIELD_TYPES).filter(function (type) {
        return type !== KBN_FIELD_TYPES.HISTOGRAM;
      }),
      write: function write(agg, output) {
        var field = agg.getParam('field');
        output.params = {};

        if (field.scripted) {
          output.params.script_fields = _defineProperty({}, field.name, {
            script: {
              source: field.script,
              lang: field.lang
            }
          });
        } else {
          if (field.readFromDocValues) {
            output.params.docvalue_fields = [_objectSpread({
              field: field.name
            }, field.type === KBN_FIELD_TYPES.DATE && {
              format: 'date_time'
            })];
          }

          output.params._source = field.name === '_source' ? true : field.name;
        }
      }
    }, {
      name: 'aggregate',
      type: 'optioned',
      options: [{
        text: i18n.translate('data.search.aggs.metrics.topHit.minLabel', {
          defaultMessage: 'Min'
        }),
        isCompatible: isNumericFieldSelected,
        disabled: true,
        value: 'min'
      }, {
        text: i18n.translate('data.search.aggs.metrics.topHit.maxLabel', {
          defaultMessage: 'Max'
        }),
        isCompatible: isNumericFieldSelected,
        disabled: true,
        value: 'max'
      }, {
        text: i18n.translate('data.search.aggs.metrics.topHit.sumLabel', {
          defaultMessage: 'Sum'
        }),
        isCompatible: isNumericFieldSelected,
        disabled: true,
        value: 'sum'
      }, {
        text: i18n.translate('data.search.aggs.metrics.topHit.averageLabel', {
          defaultMessage: 'Average'
        }),
        isCompatible: isNumericFieldSelected,
        disabled: true,
        value: 'average'
      }, {
        text: i18n.translate('data.search.aggs.metrics.topHit.concatenateLabel', {
          defaultMessage: 'Concatenate'
        }),
        isCompatible: function isCompatible(aggConfig) {
          return _.get(aggConfig.params, 'field.filterFieldTypes', '*') === '*';
        },
        disabled: true,
        value: 'concat'
      }],
      write: _.noop
    }, {
      name: 'size',
      default: 1
    }, {
      name: 'sortField',
      type: 'field',
      filterFieldTypes: [KBN_FIELD_TYPES.NUMBER, KBN_FIELD_TYPES.DATE, KBN_FIELD_TYPES.IP, KBN_FIELD_TYPES.STRING],
      default: function _default(agg) {
        return agg.getIndexPattern().timeFieldName;
      },
      write: _.noop // prevent default write, it is handled below

    }, {
      name: 'sortOrder',
      type: 'optioned',
      default: 'desc',
      options: [{
        text: i18n.translate('data.search.aggs.metrics.topHit.descendingLabel', {
          defaultMessage: 'Descending'
        }),
        value: 'desc'
      }, {
        text: i18n.translate('data.search.aggs.metrics.topHit.ascendingLabel', {
          defaultMessage: 'Ascending'
        }),
        value: 'asc'
      }],
      write: function write(agg, output) {
        var sortField = agg.params.sortField;
        var sortOrder = agg.params.sortOrder;

        if (sortField.scripted) {
          output.params.sort = [{
            _script: {
              script: {
                source: sortField.script,
                lang: sortField.lang
              },
              type: sortField.type,
              order: sortOrder.value
            }
          }];
        } else {
          output.params.sort = [_defineProperty({}, sortField.name, {
            order: sortOrder.value
          })];
        }
      }
    }],
    getValue: function getValue(agg, bucket) {
      var hits = _.get(bucket, "".concat(agg.id, ".hits.hits"));

      if (!hits || !hits.length) {
        return null;
      }

      var path = agg.getParam('field').name;

      var values = _.flatten(hits.map(function (hit) {
        return path === '_source' ? hit._source : agg.getIndexPattern().flattenHit(hit, true)[path];
      }));

      if (values.length === 1) {
        values = values[0];
      }

      if (Array.isArray(values)) {
        if (!_.compact(values).length) {
          return null;
        }

        var aggregate = agg.getParam('aggregate');

        switch (aggregate.value) {
          case 'max':
            return _.max(values);

          case 'min':
            return _.min(values);

          case 'sum':
            return _.sum(values);

          case 'average':
            return _.sum(values) / values.length;
        }
      }

      return values;
    }
  });
};