/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.common.cloud;

import java.lang.invoke.MethodHandles;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.solr.common.SolrException;
import org.apache.solr.common.cloud.BeforeReconnect;
import org.apache.solr.common.cloud.OnReconnect;
import org.apache.solr.common.cloud.SolrZkClient;
import org.apache.solr.common.cloud.SolrZooKeeper;
import org.apache.solr.common.cloud.ZkClientConnectionStrategy;
import org.apache.solr.common.cloud.ZooKeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConnectionManager
implements Watcher {
    private static final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private final String name;
    private volatile boolean connected = false;
    private final ZkClientConnectionStrategy connectionStrategy;
    private final String zkServerAddress;
    private final SolrZkClient client;
    private final OnReconnect onReconnect;
    private final BeforeReconnect beforeReconnect;
    private volatile boolean isClosed = false;
    private volatile LikelyExpiredState likelyExpiredState = LikelyExpiredState.access$000();
    private IsClosed isClosedCheck;

    public ConnectionManager(String name, SolrZkClient client, String zkServerAddress, ZkClientConnectionStrategy strat, OnReconnect onConnect, BeforeReconnect beforeReconnect, IsClosed isClosed) {
        this.name = name;
        this.client = client;
        this.connectionStrategy = strat;
        this.zkServerAddress = zkServerAddress;
        this.onReconnect = onConnect;
        this.beforeReconnect = beforeReconnect;
        this.isClosedCheck = isClosed;
    }

    private synchronized void connected() {
        this.connected = true;
        this.likelyExpiredState = LikelyExpiredState.NOT_EXPIRED;
        this.notifyAll();
    }

    private synchronized void disconnected() {
        this.connected = false;
        if (!this.likelyExpiredState.isLikelyExpired(0L)) {
            this.likelyExpiredState = new LikelyExpiredState(LikelyExpiredState.StateType.TRACKING_TIME, System.nanoTime());
        }
        this.notifyAll();
    }

    public void process(WatchedEvent event) {
        if (event.getState() == Watcher.Event.KeeperState.AuthFailed || event.getState() == Watcher.Event.KeeperState.Disconnected || event.getState() == Watcher.Event.KeeperState.Expired) {
            log.warn("Watcher {} name: {} got event {} path: {} type: {}", new Object[]{this, this.name, event, event.getPath(), event.getType()});
        } else if (log.isDebugEnabled()) {
            log.debug("Watcher {} name: {} got event {} path: {} type: {}", new Object[]{this, this.name, event, event.getPath(), event.getType()});
        }
        if (this.isClosed()) {
            log.debug("Client->ZooKeeper status change trigger but we are already closed");
            return;
        }
        Watcher.Event.KeeperState state = event.getState();
        if (state == Watcher.Event.KeeperState.SyncConnected) {
            log.info("zkClient has connected");
            this.connected();
            this.connectionStrategy.connected();
        } else if (state == Watcher.Event.KeeperState.Expired) {
            if (this.isClosed()) {
                return;
            }
            this.connected = false;
            this.likelyExpiredState = LikelyExpiredState.EXPIRED;
            log.warn("Our previous ZooKeeper session was expired. Attempting to reconnect to recover relationship with ZooKeeper...");
            if (this.beforeReconnect != null) {
                try {
                    this.beforeReconnect.command();
                }
                catch (Exception e) {
                    log.warn("Exception running beforeReconnect command", (Throwable)e);
                }
            }
            while (true) {
                try {
                    this.connectionStrategy.reconnect(this.zkServerAddress, this.client.getZkClientTimeout(), this, new ZkClientConnectionStrategy.ZkUpdate(){

                        @Override
                        public void update(SolrZooKeeper keeper) {
                            try {
                                ConnectionManager.this.waitForConnected(Long.MAX_VALUE);
                                try {
                                    ConnectionManager.this.client.updateKeeper(keeper);
                                }
                                catch (InterruptedException e) {
                                    ConnectionManager.this.closeKeeper(keeper);
                                    Thread.currentThread().interrupt();
                                    throw new RuntimeException(e);
                                }
                                if (ConnectionManager.this.onReconnect != null) {
                                    ConnectionManager.this.onReconnect.command();
                                }
                            }
                            catch (Exception e1) {
                                ConnectionManager.this.closeKeeper(keeper);
                                throw new RuntimeException(e1);
                            }
                        }
                    });
                }
                catch (Exception e) {
                    SolrException.log(log, "", e);
                    log.info("Could not connect due to error, sleeping for 1s and trying again");
                    this.waitSleep(1000L);
                    if (!this.isClosed()) continue;
                }
                break;
            }
            log.info("zkClient Connected: {}", (Object)this.connected);
        } else if (state == Watcher.Event.KeeperState.Disconnected) {
            log.warn("zkClient has disconnected");
            this.disconnected();
            this.connectionStrategy.disconnected();
        } else if (state == Watcher.Event.KeeperState.AuthFailed) {
            log.warn("zkClient received AuthFailed");
        }
    }

    public synchronized boolean isConnectedAndNotClosed() {
        return !this.isClosed() && this.connected;
    }

    public synchronized boolean isConnected() {
        return this.connected;
    }

    public void close() {
        this.isClosed = true;
        this.likelyExpiredState = LikelyExpiredState.EXPIRED;
    }

    private boolean isClosed() {
        return this.isClosed || this.isClosedCheck.isClosed();
    }

    public boolean isLikelyExpired() {
        return this.isClosed() || this.likelyExpiredState.isLikelyExpired((long)((double)this.client.getZkClientTimeout() * 0.9));
    }

    public synchronized void waitSleep(long waitFor) {
        try {
            this.wait(waitFor);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    public synchronized void waitForConnected(long waitForConnection) throws TimeoutException {
        log.info("Waiting for client to connect to ZooKeeper");
        long expire = System.nanoTime() + TimeUnit.NANOSECONDS.convert(waitForConnection, TimeUnit.MILLISECONDS);
        long left = 1L;
        while (!this.connected && left > 0L && !this.isClosed()) {
            try {
                this.wait(500L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                break;
            }
            left = expire - System.nanoTime();
        }
        if (!this.connected) {
            throw new TimeoutException("Could not connect to ZooKeeper " + this.zkServerAddress + " within " + waitForConnection + " ms");
        }
        log.info("Client is connected to ZooKeeper");
    }

    public synchronized void waitForDisconnected(long timeout) throws InterruptedException, TimeoutException {
        long expire = System.nanoTime() + TimeUnit.NANOSECONDS.convert(timeout, TimeUnit.MILLISECONDS);
        long left = timeout;
        while (this.connected && left > 0L) {
            this.wait(left);
            left = expire - System.nanoTime();
        }
        if (this.connected) {
            throw new TimeoutException("Did not disconnect");
        }
    }

    private void closeKeeper(SolrZooKeeper keeper) {
        try {
            keeper.close();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            log.error("", (Throwable)e);
            throw new ZooKeeperException(SolrException.ErrorCode.SERVER_ERROR, "", (Throwable)e);
        }
    }

    public static abstract class IsClosed {
        public abstract boolean isClosed();
    }

    private static class LikelyExpiredState {
        private static LikelyExpiredState NOT_EXPIRED = new LikelyExpiredState(StateType.NOT_EXPIRED, 0L);
        private static LikelyExpiredState EXPIRED = new LikelyExpiredState(StateType.EXPIRED, 0L);
        private StateType stateType;
        private long lastDisconnectTime;

        public LikelyExpiredState(StateType stateType, long lastDisconnectTime) {
            this.stateType = stateType;
            this.lastDisconnectTime = lastDisconnectTime;
        }

        public boolean isLikelyExpired(long timeToExpire) {
            return this.stateType == StateType.EXPIRED || this.stateType == StateType.TRACKING_TIME && System.nanoTime() - this.lastDisconnectTime > TimeUnit.NANOSECONDS.convert(timeToExpire, TimeUnit.MILLISECONDS);
        }

        public static enum StateType {
            NOT_EXPIRED,
            EXPIRED,
            TRACKING_TIME;

        }
    }
}

