/*
 * Decompiled with CFR 0.152.
 */
package ghidra.util.data;

import ghidra.app.plugin.core.datamgr.util.DataTypeUtils;
import ghidra.app.services.DataTypeManagerService;
import ghidra.program.database.data.DataTypeUtilities;
import ghidra.program.database.data.ProgramDataTypeManager;
import ghidra.program.model.data.ArrayDataType;
import ghidra.program.model.data.BuiltInDataTypeManager;
import ghidra.program.model.data.CategoryPath;
import ghidra.program.model.data.DataType;
import ghidra.program.model.data.DataTypeManager;
import ghidra.program.model.data.Dynamic;
import ghidra.program.model.data.FactoryDataType;
import ghidra.program.model.data.InvalidDataTypeException;
import ghidra.program.model.data.PointerDataType;
import ghidra.program.model.data.TypeDef;
import java.util.ArrayList;
import java.util.List;

public class DataTypeParser {
    private DataTypeManager sourceDataTypeManager;
    private DataTypeManager destinationDataTypeManager;
    private DataTypeManagerService dataTypeManagerService;
    private AllowedDataTypes allowedTypes;

    public DataTypeParser(DataTypeManagerService dataTypeManagerService, AllowedDataTypes allowedTypes) {
        this.dataTypeManagerService = dataTypeManagerService;
        this.allowedTypes = allowedTypes;
    }

    public DataTypeParser(DataTypeManager sourceDataTypeManager, DataTypeManager destinationDataTypeManager, DataTypeManagerService dataTypeManagerService, AllowedDataTypes allowedTypes) {
        this.sourceDataTypeManager = sourceDataTypeManager;
        this.destinationDataTypeManager = destinationDataTypeManager;
        this.dataTypeManagerService = dataTypeManagerService;
        this.allowedTypes = allowedTypes;
    }

    public DataType parse(String dataTypeString) throws InvalidDataTypeException {
        return this.parse(dataTypeString, (CategoryPath)null);
    }

    public DataType parse(String dataTypeString, CategoryPath category) throws InvalidDataTypeException {
        String dataTypeName = DataTypeParser.getBaseString(dataTypeString = dataTypeString.replaceAll("\\s+", " ").trim());
        DataType namedDt = this.getNamedDataType(dataTypeName, category);
        if (namedDt == null) {
            throw new InvalidDataTypeException("valid data-type not specified");
        }
        return this.parseDataTypeModifiers(namedDt, dataTypeString.substring(dataTypeName.length()));
    }

    public DataType parse(String dataTypeString, DataType suggestedBaseDataType) throws InvalidDataTypeException {
        DataType namedDt;
        String dataTypeName = DataTypeParser.getBaseString(dataTypeString = dataTypeString.replaceAll("\\s+", " ").trim());
        if (dataTypeName == null || dataTypeName.length() == 0) {
            throw new InvalidDataTypeException("missing base data-type name");
        }
        if (suggestedBaseDataType != null && dataTypeName.equals(suggestedBaseDataType.getName())) {
            namedDt = suggestedBaseDataType;
            if (namedDt.getDataTypeManager() != this.destinationDataTypeManager) {
                namedDt = namedDt.clone(this.destinationDataTypeManager);
            }
        } else {
            namedDt = this.getNamedDataType(dataTypeName, null);
            if (namedDt == null) {
                throw new InvalidDataTypeException("valid data-type not specified");
            }
        }
        return this.parseDataTypeModifiers(namedDt, dataTypeString.substring(dataTypeName.length()));
    }

    public static void checkAllowableType(DataType dt, AllowedDataTypes allowedTypes) throws InvalidDataTypeException {
        if (allowedTypes == AllowedDataTypes.DYNAMIC) {
            if (dt instanceof FactoryDataType) {
                throw new InvalidDataTypeException("factory data-type not allowed");
            }
        } else if (allowedTypes == AllowedDataTypes.SIZABLE_DYNAMIC) {
            if (dt instanceof FactoryDataType) {
                throw new InvalidDataTypeException("factory data-type not allowed");
            }
            if (dt instanceof Dynamic && !((Dynamic)dt).canSpecifyLength()) {
                throw new InvalidDataTypeException("non-sizable data-type not allowed");
            }
        } else if (allowedTypes == AllowedDataTypes.FIXED_LENGTH && dt.getLength() < 0) {
            throw new InvalidDataTypeException("fixed-length data-type required");
        }
    }

    private DataType parseDataTypeModifiers(DataType namedDataType, String dataTypeModifiers) throws InvalidDataTypeException {
        int arraySequenceStartIndex = -1;
        ArrayList<DtPiece> modifiers = new ArrayList<DtPiece>();
        for (String piece : DataTypeParser.splitDataTypeModifiers(dataTypeModifiers)) {
            if (piece.startsWith("*")) {
                modifiers.add(new PointerSpecPiece(piece));
                arraySequenceStartIndex = -1;
                continue;
            }
            if (piece.startsWith("[")) {
                ArraySpecPiece arraySpec = new ArraySpecPiece(piece);
                if (arraySequenceStartIndex >= 0) {
                    modifiers.add(arraySequenceStartIndex, arraySpec);
                    continue;
                }
                arraySequenceStartIndex = modifiers.size();
                modifiers.add(arraySpec);
                continue;
            }
            if (!piece.startsWith("{")) continue;
            modifiers.add(new ElementSizeSpecPiece(piece));
            arraySequenceStartIndex = -1;
        }
        DataType dt = namedDataType;
        int elementLength = dt.getLength();
        try {
            for (DtPiece modifier : modifiers) {
                if (modifier instanceof PointerSpecPiece) {
                    int pointerSize = ((PointerSpecPiece)modifier).getPointerSize();
                    dt = new PointerDataType(dt, pointerSize, this.destinationDataTypeManager);
                    elementLength = dt.getLength();
                    continue;
                }
                if (modifier instanceof ElementSizeSpecPiece) {
                    if (elementLength > 0) continue;
                    elementLength = ((ElementSizeSpecPiece)modifier).getElementSize();
                    continue;
                }
                int elementCount = ((ArraySpecPiece)modifier).getElementCount();
                dt = this.createArrayDataType(dt, elementLength, elementCount);
                elementLength = dt.getLength();
            }
        }
        catch (IllegalArgumentException e) {
            throw new InvalidDataTypeException(e.getMessage());
        }
        DataTypeParser.checkAllowableType(dt, this.allowedTypes);
        return dt;
    }

    private DataType getNamedDataType(String baseName, CategoryPath category) throws InvalidDataTypeException {
        ArrayList<DataType> results = new ArrayList<DataType>();
        DataType dt = this.findDataType(this.sourceDataTypeManager, baseName, category, results);
        if (dt != null) {
            return dt;
        }
        if (results.isEmpty() && DataType.DEFAULT.getDisplayName().equals(baseName)) {
            dt = DataType.DEFAULT;
        } else if (category == null) {
            dt = this.findDataTypeInAllDataTypeManagers(baseName, results);
        }
        if (dt == null) {
            String msg = "Unrecognized data type of \"" + baseName + "\"";
            throw new InvalidDataTypeException(msg);
        }
        return dt.clone(this.destinationDataTypeManager);
    }

    private DataType findDataTypeInAllDataTypeManagers(String baseName, List<DataType> results) {
        if (results.isEmpty() && this.dataTypeManagerService != null) {
            results.addAll(DataTypeUtils.getExactMatchingDataTypes(baseName, this.dataTypeManagerService));
        }
        DataType dt = null;
        if (!results.isEmpty() && (dt = DataTypeParser.pickFromPossibleEquivalentDataTypes(results)) == null && this.dataTypeManagerService != null) {
            dt = this.dataTypeManagerService.getDataType(baseName);
        }
        return dt;
    }

    private DataType findDataType(DataTypeManager dtm, String baseName, CategoryPath category, List<DataType> list) {
        BuiltInDataTypeManager builtInDTM = BuiltInDataTypeManager.getDataTypeManager();
        if (dtm == null) {
            return this.findDataType((DataTypeManager)builtInDTM, baseName, category, list);
        }
        if (category != null) {
            DataType dt = dtm.getDataType(category, baseName);
            if (dt != null) {
                list.add(dt);
                return dt;
            }
        } else {
            DataType dataType = DataTypeUtilities.getCPrimitiveDataType((String)baseName);
            if (dataType != null) {
                return dataType;
            }
            dtm.findDataTypes(baseName, list);
            if (list.size() == 1) {
                return list.get(0);
            }
        }
        if (list.isEmpty() && dtm != builtInDTM) {
            return this.findDataType((DataTypeManager)builtInDTM, baseName, category, list);
        }
        return null;
    }

    private static DataType pickFromPossibleEquivalentDataTypes(List<DataType> dtList) {
        DataType programDataType = null;
        for (DataType dataType : dtList) {
            DataTypeManager manager = dataType.getDataTypeManager();
            if (manager instanceof BuiltInDataTypeManager) {
                programDataType = dataType;
                continue;
            }
            if (!(manager instanceof ProgramDataTypeManager)) continue;
            programDataType = dataType;
            break;
        }
        if (programDataType == null) {
            return null;
        }
        for (DataType dataType : dtList) {
            if (programDataType.isEquivalent(dataType)) continue;
            return null;
        }
        return programDataType;
    }

    private static String getBaseString(String dataTypeString) {
        for (int nextIndex = 0; nextIndex < dataTypeString.length(); ++nextIndex) {
            char c = dataTypeString.charAt(nextIndex);
            if (c != '*' && c != '[' && c != '{') continue;
            return dataTypeString.substring(0, nextIndex).trim();
        }
        return dataTypeString;
    }

    private static String[] splitDataTypeModifiers(String dataTypeModifiers) {
        int nextIndex;
        if ((dataTypeModifiers = dataTypeModifiers.replaceAll("[ \\t]", "")).length() == 0) {
            return new String[0];
        }
        ArrayList<String> list = new ArrayList<String>();
        int startIndex = 0;
        for (nextIndex = 1; nextIndex < dataTypeModifiers.length(); ++nextIndex) {
            char c = dataTypeModifiers.charAt(nextIndex);
            if (c != '*' && c != '[' && c != '{') continue;
            list.add(dataTypeModifiers.substring(startIndex, nextIndex));
            startIndex = nextIndex;
        }
        list.add(dataTypeModifiers.substring(startIndex, nextIndex));
        String[] pieces = new String[list.size()];
        list.toArray(pieces);
        return pieces;
    }

    private DataType createArrayDataType(DataType baseDataType, int elementLength, int elementCount) throws InvalidDataTypeException {
        DataType dt = baseDataType;
        if (dt instanceof TypeDef) {
            dt = ((TypeDef)dt).getBaseDataType();
        }
        if (elementLength <= 0) {
            throw new InvalidDataTypeException("only a positive datatype element size may be used for array: " + baseDataType.getName());
        }
        return new ArrayDataType(baseDataType, elementCount, elementLength, this.destinationDataTypeManager);
    }

    private static int parseArraySize(String numStr) {
        String string = numStr = numStr == null ? "" : numStr.trim();
        if (numStr.length() == 0) {
            throw new NumberFormatException();
        }
        if (numStr.startsWith("0x") || numStr.startsWith("0X")) {
            return Integer.parseInt(numStr.substring(2), 16);
        }
        return Integer.parseInt(numStr);
    }

    private static class ElementSizeSpecPiece
    implements DtPiece {
        int elementSize;

        ElementSizeSpecPiece(String piece) throws InvalidDataTypeException {
            if (piece.startsWith("{") && piece.endsWith("}")) {
                String elementSizeStr = piece.substring(1, piece.length() - 1);
                try {
                    this.elementSize = DataTypeParser.parseArraySize(elementSizeStr);
                    return;
                }
                catch (NumberFormatException numberFormatException) {
                    // empty catch block
                }
            }
            throw new InvalidDataTypeException("invalid array element size specification: " + piece);
        }

        int getElementSize() {
            return this.elementSize;
        }
    }

    private static class PointerSpecPiece
    implements DtPiece {
        int pointerSize = -1;

        PointerSpecPiece(String piece) throws InvalidDataTypeException {
            if (!piece.startsWith("*")) {
                throw new InvalidDataTypeException("invalid pointer specification: " + piece);
            }
            if (piece.length() == 1) {
                return;
            }
            try {
                this.pointerSize = Integer.parseInt(piece.substring(1));
            }
            catch (NumberFormatException e) {
                throw new InvalidDataTypeException("invalid pointer specification: " + piece);
            }
            int mod = this.pointerSize % 8;
            this.pointerSize /= 8;
            if (mod != 0 || this.pointerSize <= 0 || this.pointerSize > 8) {
                throw new InvalidDataTypeException("invalid pointer size: " + piece);
            }
        }

        int getPointerSize() {
            return this.pointerSize;
        }
    }

    private static class ArraySpecPiece
    implements DtPiece {
        int elementCount;

        ArraySpecPiece(String piece) throws InvalidDataTypeException {
            if (piece.startsWith("[") && piece.endsWith("]")) {
                String elementCountStr = piece.substring(1, piece.length() - 1);
                try {
                    this.elementCount = DataTypeParser.parseArraySize(elementCountStr);
                    return;
                }
                catch (NumberFormatException numberFormatException) {
                    // empty catch block
                }
            }
            throw new InvalidDataTypeException("invalid array specification: " + piece);
        }

        int getElementCount() {
            return this.elementCount;
        }
    }

    private static interface DtPiece {
    }

    public static enum AllowedDataTypes {
        ALL,
        DYNAMIC,
        SIZABLE_DYNAMIC,
        FIXED_LENGTH;

    }
}

